var tsImport = require('ts-import');
var module$1 = require('module');
var fs = require('fs');
var path = require('path');
var child_process = require('child_process');

function _interopDefaultLegacy (e) { return e && typeof e === 'object' && 'default' in e ? e : { 'default': e }; }

var fs__default = /*#__PURE__*/_interopDefaultLegacy(fs);
var path__default = /*#__PURE__*/_interopDefaultLegacy(path);

function _extends() {
  _extends = Object.assign || function (target) {
    for (var i = 1; i < arguments.length; i++) {
      var source = arguments[i];

      for (var key in source) {
        if (Object.prototype.hasOwnProperty.call(source, key)) {
          target[key] = source[key];
        }
      }
    }

    return target;
  };

  return _extends.apply(this, arguments);
}

function importMetaUrl() {
  return (typeof document === 'undefined' ? new (require('u' + 'rl').URL)('file:' + __filename).href : (document.currentScript && document.currentScript.src || new URL('i18n-unused.cjs', document.baseURI).href));
}

const getFileSizeKb = str => Buffer.byteLength(str, "utf8") / 1000;
const resolveFile = async (filePath, resolver = m => m, loader) => {
  const [, ext] = filePath.match(/\.([0-9a-z]+)(?:[?#]|$)/i) || [];
  let m = {};

  if (loader) {
    m = loader(filePath);
  } else if (ext === "ts") {
    m = await tsImport.tsImport.compile(filePath);
  } else if (["js", "cjs"].includes(ext)) {
    let r = module$1.createRequire(importMetaUrl());
    r = r("esm")(m
    /*, options*/
    );
    m = r(filePath);
  } else if (ext === "json") {
    const r = module$1.createRequire(importMetaUrl());
    m = r(filePath);
  }

  return resolver(m);
};

const useFileNameResolver = (resolver, name) => {
  if (resolver instanceof RegExp) {
    return resolver.test(name);
  }

  if (typeof resolver === "function") {
    return resolver(name);
  }

  return false;
};

const generateFilesPaths = async (srcPath, {
  basePath,
  ignorePaths,
  srcExtensions,
  fileNameResolver
}) => {
  // Dirent: https://nodejs.org/api/fs.html#class-fsdirent
  const entries = await fs.promises.readdir(srcPath, {
    withFileTypes: true
  });
  const files = await entries.reduce(async (accPromise, dirent) => {
    const nextPath = path__default['default'].resolve(srcPath, dirent.name);
    const acc = await accPromise;

    if (ignorePaths) {
      const fullBasePath = path__default['default'].resolve(`${process.cwd()}/${basePath}`);
      const pathFromBasePath = `${path__default['default'].relative(fullBasePath, nextPath)}${dirent.isDirectory() ? "/" : ""}`;

      if (ignorePaths.some(ignorePath => pathFromBasePath.startsWith(`${ignorePath}/`))) {
        return acc;
      }
    }

    if (dirent.isDirectory()) {
      const generatedNextPath = await generateFilesPaths(nextPath, {
        basePath,
        ignorePaths,
        srcExtensions,
        fileNameResolver
      });
      acc.push(...generatedNextPath);
      return acc;
    }

    const fileName = path__default['default'].basename(nextPath);

    if (srcExtensions) {
      const [, ext] = fileName.match(/\.([0-9a-z]+)(?:[?#]|$)/i) || [];
      const validExtension = srcExtensions.some(_ext => {
        if (_ext === ext && fileNameResolver) {
          return useFileNameResolver(fileNameResolver, fileName);
        }

        return _ext === ext;
      });

      if (!validExtension) {
        return acc;
      }
    }

    if (fileNameResolver && !useFileNameResolver(fileNameResolver, fileName)) {
      return acc;
    }

    acc.push(nextPath);
    return acc;
  }, Promise.resolve([]));
  return files;
};

/**
 * Transform the escaped string provided into a valid regex
 * @param {string} str
 * @return {RegExp}
 */
const parseRegex = str => {
  const parts = str.split("/");
  return new RegExp(`${parts[1]}`.replace(/\\\\/g, "\\"), parts[2]);
};

const defaultValues = {
  srcPath: "",
  context: true,
  excludeKey: "",
  marker: "[UNUSED]",
  ignoreComments: false,
  flatTranslations: false,
  translationSeparator: ".",
  translationContextSeparator: "_",
  // Based on https://www.i18next.com/misc/json-format
  translationContextMatcher: /^(zero|one|two|few|many|other|male|female|0|1|2|3|4|5|plural|11|100)$/,
  srcExtensions: ["js", "ts", "jsx", "tsx", "vue"],
  translationKeyMatcher: /(?:[$ .](_|t|tc|i18nKey))\(([\n\r\s]|.)*?\)/gi,
  localeFileParser: m => m.default || m,
  missedTranslationParser: /\(([^)]+)\)/,
  localeJsonStringifyIndent: 2
};
const initialize = async inlineOptions => {
  let config = _extends({}, inlineOptions);

  try {
    const base = process.cwd();
    let configFile = {};

    for (const ext of ["js", "cjs", "json"]) {
      const path = `${base}/i18n-unused.config.${ext}`;

      if (fs__default['default'].existsSync(path)) {
        configFile = await resolveFile(path); // ⛔ There is no safe/reliable way to parse a function
        // ✔ When the file is a JSON need to parse the regex

        if (ext === "json") {
          const potentialRegex = ["translationContextMatcher", "translationKeyMatcher", "missedTranslationParser", "localeNameResolver"];
          potentialRegex.forEach(value => {
            if (Object.prototype.hasOwnProperty.call(configFile, value)) {
              configFile[value] = parseRegex(configFile[value]);
            }
          });
        }

        break;
      }
    }

    config = _extends({}, configFile, inlineOptions);
  } catch (e) {}

  if (!config.localesPath) {
    throw new Error("Locales path is required");
  }

  if (!config.localesExtensions && !config.localeNameResolver) {
    config.localesExtensions = ["json"];
  }

  return _extends({}, defaultValues, config);
};

const generateTranslationsFlatKeys = (source, {
  parent,
  keys = [],
  excludeKey,
  context,
  contextSeparator,
  contextMatcher
}) => {
  Object.keys(source).forEach(key => {
    const flatKey = parent ? `${parent}.${key}` : key;

    if (!Array.isArray(source[key]) && typeof source[key] === "object") {
      generateTranslationsFlatKeys(source[key], {
        contextSeparator,
        parent: flatKey,
        excludeKey,
        context,
        contextMatcher,
        keys
      });
    } else {
      keys.push(context ? getKeyWithoutContext(flatKey, contextSeparator, contextMatcher) : flatKey);
    }
  });
  const resultKeys = excludeKey ? keys.filter(k => typeof excludeKey === "string" ? !k.includes(excludeKey) : excludeKey.every(ek => !k.includes(ek))) : keys; // The context removal can cause duplicates, so we need to remove them

  return [...new Set(resultKeys)];
};
/**
 * Removes context from key.
 *
 * Makes sure translation keys like `some_key_i_have` is not treated as context.
 */

const getKeyWithoutContext = (flatKey, contextSeparator, contextMatcher) => {
  const splitted = flatKey.split(contextSeparator);
  if (splitted.length === 1) return flatKey;
  const lastPart = splitted[splitted.length - 1]; // If the last part is a context, remove it

  if (lastPart.match(contextMatcher)) {
    return splitted.slice(0, splitted.length - 2).join(contextSeparator);
  } // Otherwise, join all parts


  return splitted.join(contextSeparator);
};

const replaceQuotes = v => v.replace(/['"`]/gi, "");

const isStaticKey = v => !v.includes("${") && /['"]/.test(v);

const isDynamicKey = v => v.includes("${") || !/['"]/.test(v);

const isInlineComment = str => /^(\/\/)/.test(str);

const isHTMLComment = str => /^(<!--)/.test(str);

const isStartOfMultilineComment = str => /^(\/\*)/.test(str);

const isEndOfMultilineComment = str => /^(\*\/)/.test(str);

const removeComments = fileTxt => {
  let skip = false;
  return fileTxt.split("\n").reduce((acc, str) => {
    const _str = str.trim();

    if (isStartOfMultilineComment(_str) || isEndOfMultilineComment(_str)) {
      skip = isStartOfMultilineComment(_str);
    }

    if (skip || isInlineComment(_str) || isHTMLComment(_str)) {
      return acc;
    }

    acc.push(str);
    return acc;
  }, []).join("\n");
};

const collectUnusedTranslations = async (localesPaths, srcFilesPaths, {
  ignoreComments,
  localeFileParser,
  localeFileLoader,
  customChecker,
  excludeTranslationKey,
  contextMatcher,
  contextSeparator,
  context,
  translationKeyMatcher
}) => {
  const translations = [];

  for (const localePath of localesPaths) {
    const locale = await resolveFile(localePath, localeFileParser, localeFileLoader);
    const translationsKeys = generateTranslationsFlatKeys(locale, {
      excludeKey: excludeTranslationKey,
      contextMatcher,
      contextSeparator,
      context
    });
    srcFilesPaths.forEach(filePath => {
      const file = fs.readFileSync(filePath).toString();
      const matchKeys = (ignoreComments ? removeComments(file) : file).match(translationKeyMatcher) || [];
      const matchKeysSet = new Set(matchKeys);

      if (customChecker) {
        customChecker(matchKeysSet, translationsKeys);
      } else {
        const matchKeysSetArrStr = [...matchKeysSet].toString();
        [...translationsKeys].forEach(key => {
          if (matchKeysSetArrStr.includes(key)) {
            translationsKeys.splice(translationsKeys.indexOf(key), 1);
          }
        });
      }
    });
    translations.push({
      localePath: localePath,
      keys: translationsKeys,
      count: translationsKeys.length
    });
  }

  return {
    translations,
    totalCount: translations.reduce((acc, {
      count
    }) => acc + count, 0)
  };
};
const collectMissedTranslations = async (localesPaths, srcFilesPaths, {
  context,
  ignoreComments,
  localeFileParser,
  localeFileLoader,
  contextSeparator,
  excludeTranslationKey,
  translationKeyMatcher,
  missedTranslationParser,
  contextMatcher
}) => {
  const translations = [];
  const flatKeys = [...new Set(await localesPaths.reduce(async (asyncAcc, localePath) => {
    const acc = await asyncAcc;
    const locale = await resolveFile(localePath, localeFileParser, localeFileLoader);
    const translationsKeys = generateTranslationsFlatKeys(locale, {
      excludeKey: excludeTranslationKey,
      contextMatcher,
      contextSeparator,
      context
    });
    return [...acc, ...translationsKeys];
  }, Promise.resolve([])))];
  const filesMissedTranslationsKeys = await srcFilesPaths.reduce(async (asyncAcc, filePath) => {
    const acc = await asyncAcc;
    acc[filePath] = acc[filePath] || [];
    const file = fs.readFileSync(filePath).toString();
    const matchKeys = ((ignoreComments ? removeComments(file) : file).match(translationKeyMatcher) || []).map(v => {
      if (typeof missedTranslationParser === "function") {
        return missedTranslationParser(v);
      }

      const [, translation] = v.match(missedTranslationParser) || [];
      return translation;
    }).filter(v => v && !flatKeys.includes(replaceQuotes(v)));

    if (matchKeys.length) {
      acc[filePath].push(...matchKeys);
    }

    return acc;
  }, Promise.resolve({}));
  Object.keys(filesMissedTranslationsKeys).forEach(filePath => {
    if (!filesMissedTranslationsKeys[filePath].length) {
      return;
    }

    const staticKeys = filesMissedTranslationsKeys[filePath].filter(isStaticKey).map(replaceQuotes);
    const dynamicKeys = filesMissedTranslationsKeys[filePath].filter(isDynamicKey).map(replaceQuotes);
    translations.push({
      filePath,
      staticKeys,
      dynamicKeys,
      staticCount: staticKeys.length,
      dynamicCount: dynamicKeys.length
    });
  });
  return {
    translations,
    totalStaticCount: translations.reduce((acc, {
      staticCount: c
    }) => acc + c, 0),
    totalDynamicCount: translations.reduce((acc, {
      dynamicCount: c
    }) => acc + c, 0)
  };
};

const displayUnusedTranslations = async options => {
  const config = await initialize(options);
  const localesFilesPaths = await generateFilesPaths(config.localesPath, {
    srcExtensions: config.localesExtensions,
    fileNameResolver: config.localeNameResolver
  });
  const srcFilesPaths = await generateFilesPaths(`${process.cwd()}/${config.srcPath}`, {
    srcExtensions: config.srcExtensions,
    ignorePaths: config.ignorePaths,
    basePath: config.srcPath
  });
  const unusedTranslations = await collectUnusedTranslations(localesFilesPaths, srcFilesPaths, {
    context: config.context,
    contextSeparator: config.translationContextSeparator,
    contextMatcher: config.translationContextMatcher,
    ignoreComments: config.ignoreComments,
    localeFileParser: config.localeFileParser,
    localeFileLoader: config.localeFileLoader,
    customChecker: config.customChecker,
    excludeTranslationKey: config.excludeKey,
    translationKeyMatcher: config.translationKeyMatcher
  });
  unusedTranslations.translations.forEach(translation => {
    console.log("<<<==========================================================>>>");
    console.log(`Unused translations in: ${translation.localePath}`);
    console.log(`Unused translations count: ${translation.count}`);
    console.table(translation.keys.map(key => ({
      Translation: key
    })));
  });
  console.log(`Total unused translations count: ${unusedTranslations.totalCount}`);
  console.log(`Can free up memory: ~${getFileSizeKb(unusedTranslations.translations.reduce((acc, {
    keys
  }) => `${acc}, ${keys.join(", ")}`, ""))}kb`);
  return unusedTranslations;
};
const displayMissedTranslations = async options => {
  const config = await initialize(options);
  const localesFilesPaths = await generateFilesPaths(config.localesPath, {
    srcExtensions: config.localesExtensions,
    fileNameResolver: config.localeNameResolver
  });
  const srcFilesPaths = await generateFilesPaths(`${process.cwd()}/${config.srcPath}`, {
    srcExtensions: config.srcExtensions,
    ignorePaths: config.ignorePaths,
    basePath: config.srcPath
  });
  const missedTranslations = await collectMissedTranslations(localesFilesPaths, srcFilesPaths, {
    context: config.context,
    contextSeparator: config.translationContextSeparator,
    contextMatcher: config.translationContextMatcher,
    ignoreComments: config.ignoreComments,
    localeFileParser: config.localeFileParser,
    localeFileLoader: config.localeFileLoader,
    excludeTranslationKey: config.excludeKey,
    translationKeyMatcher: config.translationKeyMatcher,
    missedTranslationParser: config.missedTranslationParser
  });
  missedTranslations.translations.forEach(translation => {
    console.log("<<<==========================================================>>>");
    console.log(`Missed translations in: ${translation.filePath}`);
    console.log(`Missed static translations count: ${translation.staticCount}`);
    console.log(`Missed dynamic translations count: ${translation.dynamicCount}`);

    if (translation.staticKeys.length) {
      console.log("--------------------------------------------");
      console.log("Static keys:");
      console.table(translation.staticKeys.map(key => ({
        Key: key
      })));
    }

    if (translation.dynamicKeys.length) {
      console.log("--------------------------------------------");
      console.log("Dynamic keys:");
      console.table(translation.dynamicKeys.map(key => ({
        Key: key
      })));
    }
  });
  console.log(`Total missed static translations count: ${missedTranslations.totalStaticCount}`);
  console.log(`Total missed dynamic translations count: ${missedTranslations.totalDynamicCount}`);
  return missedTranslations;
};

const applyToFlatKey = (source, key, cb, options) => {
  const separatedKey = options.flatTranslations ? [key] : key.split(options.separator);
  const keyLength = separatedKey.length - 1;
  separatedKey.reduce((acc, _k, i) => {
    if (i === keyLength) {
      cb(acc, _k);
    } else {
      acc = acc[_k];
    }

    return acc;
  }, source);
  return true;
};

const GREEN = "\x1b[32m";
const MAGENTA = "\x1b[35m";

const checkUncommittedChanges = () => {
  const result = child_process.execSync("git status --porcelain").toString();

  if (result) {
    console.log(MAGENTA, "Working tree is dirty: you might want to commit your changes before running the script");
    return true;
  } else {
    console.log(GREEN, "Working tree is clean");
    return false;
  }
};

const writeJsonFile = (filePath, data, config) => {
  const jsonString = JSON.stringify(data, null, config.localeJsonStringifyIndent);
  const jsonStringWithNewLine = `${jsonString}\n`;
  fs.writeFileSync(filePath, jsonStringWithNewLine);
};

const removeUnusedTranslations = async options => {
  const config = await initialize(options);
  const localesFilesPaths = await generateFilesPaths(config.localesPath, {
    srcExtensions: ["json"] // @TODO implement other types when add other types writes

  });
  const srcFilesPaths = await generateFilesPaths(`${process.cwd()}/${config.srcPath}`, {
    srcExtensions: config.srcExtensions,
    ignorePaths: config.ignorePaths,
    basePath: config.srcPath
  });
  const unusedTranslations = await collectUnusedTranslations(localesFilesPaths, srcFilesPaths, {
    context: config.context,
    contextSeparator: config.translationContextSeparator,
    contextMatcher: config.translationContextMatcher,
    ignoreComments: config.ignoreComments,
    localeFileParser: config.localeFileParser,
    localeFileLoader: config.localeFileLoader,
    customChecker: config.customChecker,
    excludeTranslationKey: config.excludeKey,
    translationKeyMatcher: config.translationKeyMatcher
  });

  if (config.gitCheck) {
    checkUncommittedChanges();
  }

  unusedTranslations.translations.forEach(translation => {
    const r = module$1.createRequire(importMetaUrl());
    const locale = r(translation.localePath);
    translation.keys.forEach(key => applyToFlatKey(locale, key, (source, lastKey) => {
      delete source[lastKey];
    }, {
      flatTranslations: config.flatTranslations,
      separator: config.translationSeparator
    }));
    writeJsonFile(translation.localePath, locale, config);
    console.log(GREEN, `Successfully removed: ${translation.localePath}`);
  });
  return unusedTranslations;
};

const markUnusedTranslations = async options => {
  const config = await initialize(options);
  const localesFilesPaths = await generateFilesPaths(config.localesPath, {
    srcExtensions: ["json"] // @TODO implement other types when add other types writes

  });
  const srcFilesPaths = await generateFilesPaths(`${process.cwd()}/${config.srcPath}`, {
    srcExtensions: config.srcExtensions,
    ignorePaths: config.ignorePaths,
    basePath: config.srcPath
  });
  const unusedTranslations = await collectUnusedTranslations(localesFilesPaths, srcFilesPaths, {
    context: config.context,
    contextSeparator: config.translationContextSeparator,
    contextMatcher: config.translationContextMatcher,
    ignoreComments: config.ignoreComments,
    localeFileParser: config.localeFileParser,
    localeFileLoader: config.localeFileLoader,
    customChecker: config.customChecker,
    excludeTranslationKey: config.excludeKey,
    translationKeyMatcher: config.translationKeyMatcher
  });

  if (config.gitCheck) {
    checkUncommittedChanges();
  }

  unusedTranslations.translations.forEach(translation => {
    const r = module$1.createRequire(importMetaUrl());
    const locale = r(translation.localePath);
    translation.keys.forEach(key => applyToFlatKey(locale, key, (source, lastKey) => {
      source[lastKey] = `${config.marker} ${source[lastKey]}`;
    }, {
      flatTranslations: config.flatTranslations,
      separator: config.translationSeparator
    }));
    writeJsonFile(translation.localePath, locale, config);
    console.log(GREEN, `Successfully marked: ${translation.localePath}`);
  });
  return unusedTranslations;
};

const mergeLocaleData = (source, target) => {
  const keys = Object.keys(source);
  keys.forEach(key => {
    if (typeof source[key] === "object") {
      target[key] = target[key] || {};
      mergeLocaleData(source[key], target[key]);
    } else {
      target[key] = target[key] || source[key];
    }
  });
  return target;
};
const syncTranslations = async (source, target, options) => {
  const config = await initialize(options);
  const [sourcePath] = await generateFilesPaths(config.localesPath, {
    fileNameResolver: n => n === source
  });
  const [targetPath] = await generateFilesPaths(config.localesPath, {
    fileNameResolver: n => n === target
  });
  const r = module$1.createRequire(importMetaUrl());
  const sourceLocale = r(sourcePath);
  const targetLocale = r(targetPath);
  const mergedLocale = mergeLocaleData(sourceLocale, targetLocale);

  if (config.gitCheck) {
    checkUncommittedChanges();
  }

  writeJsonFile(targetPath, mergedLocale, config);
  console.log(GREEN, "Translations are synchronized");
  return true;
};

exports.collectUnusedTranslations = collectUnusedTranslations;
exports.displayMissedTranslations = displayMissedTranslations;
exports.displayUnusedTranslations = displayUnusedTranslations;
exports.generateFilesPaths = generateFilesPaths;
exports.markUnusedTranslations = markUnusedTranslations;
exports.parseRegex = parseRegex;
exports.removeUnusedTranslations = removeUnusedTranslations;
exports.syncTranslations = syncTranslations;
//# sourceMappingURL=i18n-unused.cjs.map
