import _asyncToGenerator from "@babel/runtime/helpers/asyncToGenerator";
import _defineProperty from "@babel/runtime/helpers/defineProperty";
function ownKeys(e, r) { var t = Object.keys(e); if (Object.getOwnPropertySymbols) { var o = Object.getOwnPropertySymbols(e); r && (o = o.filter(function (r) { return Object.getOwnPropertyDescriptor(e, r).enumerable; })), t.push.apply(t, o); } return t; }
function _objectSpread(e) { for (var r = 1; r < arguments.length; r++) { var t = null != arguments[r] ? arguments[r] : {}; r % 2 ? ownKeys(Object(t), !0).forEach(function (r) { _defineProperty(e, r, t[r]); }) : Object.getOwnPropertyDescriptors ? Object.defineProperties(e, Object.getOwnPropertyDescriptors(t)) : ownKeys(Object(t)).forEach(function (r) { Object.defineProperty(e, r, Object.getOwnPropertyDescriptor(t, r)); }); } return e; }
/*
Copyright 2015, 2016 OpenMarket Ltd
Copyright 2017 New Vector Ltd
Copyright 2019, 2020 The Matrix.org Foundation C.I.C.
Copyright 2021 - 2022 Šimon Brandner <simon.bra.ag@gmail.com>

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

/**
 * This is an internal module. See {@link createNewMatrixCall} for the public API.
 */

import { v4 as uuidv4 } from "uuid";
import { parse as parseSdp, write as writeSdp } from "sdp-transform";
import { logger } from "../logger.js";
import { checkObjectHasKeys, isNullOrUndefined, recursivelyAssign } from "../utils.js";
import { EventType, ToDeviceMessageId } from "../@types/event.js";
import { secureRandomString } from "../randomstring.js";
import { SDPStreamMetadataPurpose, SDPStreamMetadataKey } from "./callEventTypes.js";
import { CallFeed } from "./callFeed.js";
import { EventEmitterEvents, TypedEventEmitter } from "../models/typed-event-emitter.js";
import { GroupCallUnknownDeviceError } from "./groupCall.js";
import { MatrixError } from "../http-api/index.js";
var MediaType = /*#__PURE__*/function (MediaType) {
  MediaType["AUDIO"] = "audio";
  MediaType["VIDEO"] = "video";
  return MediaType;
}(MediaType || {});
var CodecName = /*#__PURE__*/function (CodecName) {
  CodecName["OPUS"] = "opus"; // add more as needed
  return CodecName;
}(CodecName || {}); // Used internally to specify modifications to codec parameters in SDP
export var CallState = /*#__PURE__*/function (CallState) {
  CallState["Fledgling"] = "fledgling";
  CallState["InviteSent"] = "invite_sent";
  CallState["WaitLocalMedia"] = "wait_local_media";
  CallState["CreateOffer"] = "create_offer";
  CallState["CreateAnswer"] = "create_answer";
  CallState["Connecting"] = "connecting";
  CallState["Connected"] = "connected";
  CallState["Ringing"] = "ringing";
  CallState["Ended"] = "ended";
  return CallState;
}({});
export var CallType = /*#__PURE__*/function (CallType) {
  CallType["Voice"] = "voice";
  CallType["Video"] = "video";
  return CallType;
}({});
export var CallDirection = /*#__PURE__*/function (CallDirection) {
  CallDirection["Inbound"] = "inbound";
  CallDirection["Outbound"] = "outbound";
  return CallDirection;
}({});
export var CallParty = /*#__PURE__*/function (CallParty) {
  CallParty["Local"] = "local";
  CallParty["Remote"] = "remote";
  return CallParty;
}({});
export var CallEvent = /*#__PURE__*/function (CallEvent) {
  CallEvent["Hangup"] = "hangup";
  CallEvent["State"] = "state";
  CallEvent["Error"] = "error";
  CallEvent["Replaced"] = "replaced";
  // The value of isLocalOnHold() has changed
  CallEvent["LocalHoldUnhold"] = "local_hold_unhold";
  // The value of isRemoteOnHold() has changed
  CallEvent["RemoteHoldUnhold"] = "remote_hold_unhold";
  // backwards compat alias for LocalHoldUnhold: remove in a major version bump
  CallEvent["HoldUnhold"] = "hold_unhold";
  // Feeds have changed
  CallEvent["FeedsChanged"] = "feeds_changed";
  CallEvent["AssertedIdentityChanged"] = "asserted_identity_changed";
  CallEvent["LengthChanged"] = "length_changed";
  CallEvent["DataChannel"] = "datachannel";
  CallEvent["SendVoipEvent"] = "send_voip_event";
  // When the call instantiates its peer connection
  // For apps that want to access the underlying peer connection, eg for debugging
  CallEvent["PeerConnectionCreated"] = "peer_connection_created";
  return CallEvent;
}({});
export var CallErrorCode = /*#__PURE__*/function (CallErrorCode) {
  /** The user chose to end the call */
  CallErrorCode["UserHangup"] = "user_hangup";
  /** An error code when the local client failed to create an offer. */
  CallErrorCode["LocalOfferFailed"] = "local_offer_failed";
  /**
   * An error code when there is no local mic/camera to use. This may be because
   * the hardware isn't plugged in, or the user has explicitly denied access.
   */
  CallErrorCode["NoUserMedia"] = "no_user_media";
  /**
   * Error code used when a call event failed to send
   * because unknown devices were present in the room
   */
  CallErrorCode["UnknownDevices"] = "unknown_devices";
  /**
   * Error code used when we fail to send the invite
   * for some reason other than there being unknown devices
   */
  CallErrorCode["SendInvite"] = "send_invite";
  /**
   * An answer could not be created
   */
  CallErrorCode["CreateAnswer"] = "create_answer";
  /**
   * An offer could not be created
   */
  CallErrorCode["CreateOffer"] = "create_offer";
  /**
   * Error code used when we fail to send the answer
   * for some reason other than there being unknown devices
   */
  CallErrorCode["SendAnswer"] = "send_answer";
  /**
   * The session description from the other side could not be set
   */
  CallErrorCode["SetRemoteDescription"] = "set_remote_description";
  /**
   * The session description from this side could not be set
   */
  CallErrorCode["SetLocalDescription"] = "set_local_description";
  /**
   * A different device answered the call
   */
  CallErrorCode["AnsweredElsewhere"] = "answered_elsewhere";
  /**
   * No media connection could be established to the other party
   */
  CallErrorCode["IceFailed"] = "ice_failed";
  /**
   * The invite timed out whilst waiting for an answer
   */
  CallErrorCode["InviteTimeout"] = "invite_timeout";
  /**
   * The call was replaced by another call
   */
  CallErrorCode["Replaced"] = "replaced";
  /**
   * Signalling for the call could not be sent (other than the initial invite)
   */
  CallErrorCode["SignallingFailed"] = "signalling_timeout";
  /**
   * The remote party is busy
   */
  CallErrorCode["UserBusy"] = "user_busy";
  /**
   * We transferred the call off to somewhere else
   */
  CallErrorCode["Transferred"] = "transferred";
  /**
   * A call from the same user was found with a new session id
   */
  CallErrorCode["NewSession"] = "new_session";
  return CallErrorCode;
}({});

/**
 * The version field that we set in m.call.* events
 */
var VOIP_PROTO_VERSION = "1";

/** The fallback ICE server to use for STUN or TURN protocols. */
export var FALLBACK_ICE_SERVER = "stun:turn.matrix.org";

/** The length of time a call can be ringing for. */
var CALL_TIMEOUT_MS = 60 * 1000; // ms
/** The time after which we increment callLength */
var CALL_LENGTH_INTERVAL = 1000; // ms
/** The time after which we end the call, if ICE got disconnected */
var ICE_DISCONNECTED_TIMEOUT = 30 * 1000; // ms
/** The time after which we try a ICE restart, if ICE got disconnected */
var ICE_RECONNECTING_TIMEOUT = 2 * 1000; // ms
export class CallError extends Error {
  constructor(code, msg, err) {
    // Still don't think there's any way to have proper nested errors
    super(msg + ": " + err);
    _defineProperty(this, "code", void 0);
    this.code = code;
  }
}
export function genCallID() {
  return Date.now().toString() + secureRandomString(16);
}
function getCodecParamMods(isPtt) {
  var mods = [{
    mediaType: "audio",
    codec: "opus",
    enableDtx: true,
    maxAverageBitrate: isPtt ? 12000 : undefined
  }];
  return mods;
}

/**
 * These now all have the call object as an argument. Why? Well, to know which call a given event is
 * about you have three options:
 *  1. Use a closure as the callback that remembers what call it's listening to. This can be
 *     a pain because you need to pass the listener function again when you remove the listener,
 *     which might be somewhere else.
 *  2. Use not-very-well-known fact that EventEmitter sets 'this' to the emitter object in the
 *     callback. This doesn't really play well with modern Typescript and eslint and doesn't work
 *     with our pattern of re-emitting events.
 *  3. Pass the object in question as an argument to the callback.
 *
 * Now that we have group calls which have to deal with multiple call objects, this will
 * become more important, and I think methods 1 and 2 are just going to cause issues.
 */

// The key of the transceiver map (purpose + media type, separated by ':')

// generates keys for the map of transceivers
// kind is unfortunately a string rather than MediaType as this is the type of
// track.kind
function getTransceiverKey(purpose, kind) {
  return purpose + ":" + kind;
}
export class MatrixCall extends TypedEventEmitter {
  /**
   * Construct a new Matrix Call.
   * @param opts - Config options.
   */
  constructor(opts) {
    var _this, _opts$forceTURN;
    super();
    _this = this;
    _defineProperty(this, "roomId", void 0);
    _defineProperty(this, "callId", void 0);
    _defineProperty(this, "invitee", void 0);
    _defineProperty(this, "hangupParty", void 0);
    _defineProperty(this, "hangupReason", void 0);
    _defineProperty(this, "direction", void 0);
    _defineProperty(this, "ourPartyId", void 0);
    _defineProperty(this, "peerConn", void 0);
    _defineProperty(this, "toDeviceSeq", 0);
    // whether this call should have push-to-talk semantics
    // This should be set by the consumer on incoming & outgoing calls.
    _defineProperty(this, "isPtt", false);
    _defineProperty(this, "_state", CallState.Fledgling);
    _defineProperty(this, "client", void 0);
    _defineProperty(this, "forceTURN", void 0);
    _defineProperty(this, "turnServers", void 0);
    // A queue for candidates waiting to go out.
    // We try to amalgamate candidates into a single candidate message where
    // possible
    _defineProperty(this, "candidateSendQueue", []);
    _defineProperty(this, "candidateSendTries", 0);
    _defineProperty(this, "candidatesEnded", false);
    _defineProperty(this, "feeds", []);
    // our transceivers for each purpose and type of media
    _defineProperty(this, "transceivers", new Map());
    _defineProperty(this, "inviteOrAnswerSent", false);
    _defineProperty(this, "waitForLocalAVStream", false);
    _defineProperty(this, "successor", void 0);
    _defineProperty(this, "opponentMember", void 0);
    _defineProperty(this, "opponentVersion", void 0);
    // The party ID of the other side: undefined if we haven't chosen a partner
    // yet, null if we have but they didn't send a party ID.
    _defineProperty(this, "opponentPartyId", void 0);
    _defineProperty(this, "opponentCaps", void 0);
    _defineProperty(this, "iceDisconnectedTimeout", void 0);
    _defineProperty(this, "iceReconnectionTimeOut", void 0);
    _defineProperty(this, "inviteTimeout", void 0);
    _defineProperty(this, "removeTrackListeners", new Map());
    // The logic of when & if a call is on hold is nontrivial and explained in is*OnHold
    // This flag represents whether we want the other party to be on hold
    _defineProperty(this, "remoteOnHold", false);
    // the stats for the call at the point it ended. We can't get these after we
    // tear the call down, so we just grab a snapshot before we stop the call.
    // The typescript definitions have this type as 'any' :(
    _defineProperty(this, "callStatsAtEnd", void 0);
    // Perfect negotiation state: https://www.w3.org/TR/webrtc/#perfect-negotiation-example
    _defineProperty(this, "makingOffer", false);
    _defineProperty(this, "ignoreOffer", false);
    _defineProperty(this, "isSettingRemoteAnswerPending", false);
    _defineProperty(this, "responsePromiseChain", void 0);
    // If candidates arrive before we've picked an opponent (which, in particular,
    // will happen if the opponent sends candidates eagerly before the user answers
    // the call) we buffer them up here so we can then add the ones from the party we pick
    _defineProperty(this, "remoteCandidateBuffer", new Map());
    _defineProperty(this, "remoteAssertedIdentity", void 0);
    _defineProperty(this, "remoteSDPStreamMetadata", void 0);
    _defineProperty(this, "callLengthInterval", void 0);
    _defineProperty(this, "callStartTime", void 0);
    _defineProperty(this, "opponentDeviceId", void 0);
    _defineProperty(this, "hasOpponentDeviceInfo", void 0);
    _defineProperty(this, "opponentSessionId", void 0);
    _defineProperty(this, "groupCallId", void 0);
    // Used to keep the timer for the delay before actually stopping our
    // video track after muting (see setLocalVideoMuted)
    _defineProperty(this, "stopVideoTrackTimer", void 0);
    // Used to allow connection without Video and Audio. To establish a webrtc connection without media a Data channel is
    // needed At the moment this property is true if we allow MatrixClient with isVoipWithNoMediaAllowed = true
    _defineProperty(this, "isOnlyDataChannelAllowed", void 0);
    _defineProperty(this, "stats", void 0);
    /**
     * Internal
     */
    _defineProperty(this, "gotLocalIceCandidate", event => {
      if (event.candidate) {
        if (this.candidatesEnded) {
          logger.warn("Call ".concat(this.callId, " gotLocalIceCandidate() got candidate after candidates have ended!"));
        }
        logger.debug("Call ".concat(this.callId, " got local ICE ").concat(event.candidate.sdpMid, " ").concat(event.candidate.candidate));
        if (this.callHasEnded()) return;

        // As with the offer, note we need to make a copy of this object, not
        // pass the original: that broke in Chrome ~m43.
        if (event.candidate.candidate === "") {
          this.queueCandidate(null);
        } else {
          this.queueCandidate(event.candidate);
        }
      }
    });
    _defineProperty(this, "onIceGatheringStateChange", event => {
      var _this$peerConn;
      logger.debug("Call ".concat(this.callId, " onIceGatheringStateChange() ice gathering state changed to ").concat(this.peerConn.iceGatheringState));
      if (((_this$peerConn = this.peerConn) === null || _this$peerConn === void 0 ? void 0 : _this$peerConn.iceGatheringState) === "complete") {
        this.queueCandidate(null); // We should leave it to WebRTC to announce the end
        logger.debug("Call ".concat(this.callId, " onIceGatheringStateChange() ice gathering state complete, set candidates have ended"));
      }
    });
    _defineProperty(this, "getLocalOfferFailed", err => {
      logger.error("Call ".concat(this.callId, " getLocalOfferFailed() running"), err);
      this.emit(CallEvent.Error, new CallError(CallErrorCode.LocalOfferFailed, "Failed to get local offer!", err), this);
      this.terminate(CallParty.Local, CallErrorCode.LocalOfferFailed, false);
    });
    _defineProperty(this, "getUserMediaFailed", err => {
      if (this.successor) {
        this.successor.getUserMediaFailed(err);
        return;
      }
      logger.warn("Call ".concat(this.callId, " getUserMediaFailed() failed to get user media - ending call"), err);
      this.emit(CallEvent.Error, new CallError(CallErrorCode.NoUserMedia, "Couldn't start capturing media! Is your microphone set up and does this app have permission?", err), this);
      this.terminate(CallParty.Local, CallErrorCode.NoUserMedia, false);
    });
    _defineProperty(this, "placeCallFailed", err => {
      if (this.successor) {
        this.successor.placeCallFailed(err);
        return;
      }
      logger.warn("Call ".concat(this.callId, " placeCallWithCallFeeds() failed - ending call"), err);
      this.emit(CallEvent.Error, new CallError(CallErrorCode.IceFailed, "Couldn't start call! Invalid ICE server configuration.", err), this);
      this.terminate(CallParty.Local, CallErrorCode.IceFailed, false);
    });
    _defineProperty(this, "onIceConnectionStateChanged", () => {
      var _this$peerConn2, _this$peerConn3, _this$peerConn$iceCon, _this$peerConn4, _this$peerConn5, _this$peerConn8;
      if (this.callHasEnded()) {
        return; // because ICE can still complete as we're ending the call
      }
      logger.debug("Call ".concat(this.callId, " onIceConnectionStateChanged() running (state=").concat((_this$peerConn2 = this.peerConn) === null || _this$peerConn2 === void 0 ? void 0 : _this$peerConn2.iceConnectionState, ", conn=").concat((_this$peerConn3 = this.peerConn) === null || _this$peerConn3 === void 0 ? void 0 : _this$peerConn3.connectionState, ")"));

      // ideally we'd consider the call to be connected when we get media but
      // chrome doesn't implement any of the 'onstarted' events yet
      if (["connected", "completed"].includes((_this$peerConn$iceCon = (_this$peerConn4 = this.peerConn) === null || _this$peerConn4 === void 0 ? void 0 : _this$peerConn4.iceConnectionState) !== null && _this$peerConn$iceCon !== void 0 ? _this$peerConn$iceCon : "")) {
        clearTimeout(this.iceDisconnectedTimeout);
        this.iceDisconnectedTimeout = undefined;
        if (this.iceReconnectionTimeOut) {
          clearTimeout(this.iceReconnectionTimeOut);
        }
        this.state = CallState.Connected;
        if (!this.callLengthInterval && !this.callStartTime) {
          this.callStartTime = Date.now();
          this.callLengthInterval = setInterval(() => {
            this.emit(CallEvent.LengthChanged, Math.round((Date.now() - this.callStartTime) / 1000), this);
          }, CALL_LENGTH_INTERVAL);
        }
      } else if (((_this$peerConn5 = this.peerConn) === null || _this$peerConn5 === void 0 ? void 0 : _this$peerConn5.iceConnectionState) == "failed") {
        var _this$peerConn6;
        this.candidatesEnded = false;
        // Firefox for Android does not yet have support for restartIce()
        // (the types say it's always defined though, so we have to cast
        // to prevent typescript from warning).
        if ((_this$peerConn6 = this.peerConn) !== null && _this$peerConn6 !== void 0 && _this$peerConn6.restartIce) {
          var _this$peerConn7;
          this.candidatesEnded = false;
          logger.debug("Call ".concat(this.callId, " onIceConnectionStateChanged() ice restart (state=").concat((_this$peerConn7 = this.peerConn) === null || _this$peerConn7 === void 0 ? void 0 : _this$peerConn7.iceConnectionState, ")"));
          this.peerConn.restartIce();
        } else {
          logger.info("Call ".concat(this.callId, " onIceConnectionStateChanged() hanging up call (ICE failed and no ICE restart method)"));
          this.hangup(CallErrorCode.IceFailed, false);
        }
      } else if (((_this$peerConn8 = this.peerConn) === null || _this$peerConn8 === void 0 ? void 0 : _this$peerConn8.iceConnectionState) == "disconnected") {
        this.candidatesEnded = false;
        this.iceReconnectionTimeOut = setTimeout(() => {
          var _this$peerConn9, _this$peerConn0, _this$peerConn1;
          logger.info("Call ".concat(this.callId, " onIceConnectionStateChanged() ICE restarting because of ICE disconnected, (state=").concat((_this$peerConn9 = this.peerConn) === null || _this$peerConn9 === void 0 ? void 0 : _this$peerConn9.iceConnectionState, ", conn=").concat((_this$peerConn0 = this.peerConn) === null || _this$peerConn0 === void 0 ? void 0 : _this$peerConn0.connectionState, ")"));
          if ((_this$peerConn1 = this.peerConn) !== null && _this$peerConn1 !== void 0 && _this$peerConn1.restartIce) {
            this.candidatesEnded = false;
            this.peerConn.restartIce();
          }
          this.iceReconnectionTimeOut = undefined;
        }, ICE_RECONNECTING_TIMEOUT);
        this.iceDisconnectedTimeout = setTimeout(() => {
          logger.info("Call ".concat(this.callId, " onIceConnectionStateChanged() hanging up call (ICE disconnected for too long)"));
          this.hangup(CallErrorCode.IceFailed, false);
        }, ICE_DISCONNECTED_TIMEOUT);
        this.state = CallState.Connecting;
      }

      // In PTT mode, override feed status to muted when we lose connection to
      // the peer, since we don't want to block the line if they're not saying anything.
      // Experimenting in Chrome, this happens after 5 or 6 seconds, which is probably
      // fast enough.
      if (this.isPtt && ["failed", "disconnected"].includes(this.peerConn.iceConnectionState)) {
        for (var feed of this.getRemoteFeeds()) {
          feed.setAudioVideoMuted(true, true);
        }
      }
    });
    _defineProperty(this, "onSignallingStateChanged", () => {
      var _this$peerConn10;
      logger.debug("Call ".concat(this.callId, " onSignallingStateChanged() running (state=").concat((_this$peerConn10 = this.peerConn) === null || _this$peerConn10 === void 0 ? void 0 : _this$peerConn10.signalingState, ")"));
    });
    _defineProperty(this, "onTrack", ev => {
      if (ev.streams.length === 0) {
        logger.warn("Call ".concat(this.callId, " onTrack() called with streamless track streamless (kind=").concat(ev.track.kind, ")"));
        return;
      }
      var stream = ev.streams[0];
      this.pushRemoteFeed(stream);
      if (!this.removeTrackListeners.has(stream)) {
        var onRemoveTrack = () => {
          if (stream.getTracks().length === 0) {
            logger.info("Call ".concat(this.callId, " onTrack() removing track (streamId=").concat(stream.id, ")"));
            this.deleteFeedByStream(stream);
            stream.removeEventListener("removetrack", onRemoveTrack);
            this.removeTrackListeners.delete(stream);
          }
        };
        stream.addEventListener("removetrack", onRemoveTrack);
        this.removeTrackListeners.set(stream, onRemoveTrack);
      }
    });
    _defineProperty(this, "onDataChannel", ev => {
      this.emit(CallEvent.DataChannel, ev.channel, this);
    });
    _defineProperty(this, "onNegotiationNeeded", /*#__PURE__*/_asyncToGenerator(function* () {
      logger.info("Call ".concat(_this.callId, " onNegotiationNeeded() negotiation is needed!"));
      if (_this.state !== CallState.CreateOffer && _this.opponentVersion === 0) {
        logger.info("Call ".concat(_this.callId, " onNegotiationNeeded() opponent does not support renegotiation: ignoring negotiationneeded event"));
        return;
      }
      _this.queueGotLocalOffer();
    }));
    _defineProperty(this, "onHangupReceived", msg => {
      logger.debug("Call ".concat(this.callId, " onHangupReceived() running"));

      // party ID must match (our chosen partner hanging up the call) or be undefined (we haven't chosen
      // a partner yet but we're treating the hangup as a reject as per VoIP v0)
      if (this.partyIdMatches(msg) || this.state === CallState.Ringing) {
        // default reason is user_hangup
        this.terminate(CallParty.Remote, msg.reason || CallErrorCode.UserHangup, true);
      } else {
        logger.info("Call ".concat(this.callId, " onHangupReceived() ignoring message from party ID ").concat(msg.party_id, ": our partner is ").concat(this.opponentPartyId));
      }
    });
    _defineProperty(this, "onRejectReceived", msg => {
      logger.debug("Call ".concat(this.callId, " onRejectReceived() running"));

      // No need to check party_id for reject because if we'd received either
      // an answer or reject, we wouldn't be in state InviteSent

      var shouldTerminate =
      // reject events also end the call if it's ringing: it's another of
      // our devices rejecting the call.
      [CallState.InviteSent, CallState.Ringing].includes(this.state) ||
      // also if we're in the init state and it's an inbound call, since
      // this means we just haven't entered the ringing state yet
      this.state === CallState.Fledgling && this.direction === CallDirection.Inbound;
      if (shouldTerminate) {
        this.terminate(CallParty.Remote, msg.reason || CallErrorCode.UserHangup, true);
      } else {
        logger.debug("Call ".concat(this.callId, " onRejectReceived() called in wrong state (state=").concat(this.state, ")"));
      }
    });
    _defineProperty(this, "onAnsweredElsewhere", msg => {
      logger.debug("Call ".concat(this.callId, " onAnsweredElsewhere() running"));
      this.terminate(CallParty.Remote, CallErrorCode.AnsweredElsewhere, true);
    });
    this.roomId = opts.roomId;
    this.invitee = opts.invitee;
    this.client = opts.client;
    if (!this.client.deviceId) throw new Error("Client must have a device ID to start calls");
    this.forceTURN = (_opts$forceTURN = opts.forceTURN) !== null && _opts$forceTURN !== void 0 ? _opts$forceTURN : false;
    this.ourPartyId = this.client.deviceId;
    this.opponentDeviceId = opts.opponentDeviceId;
    this.opponentSessionId = opts.opponentSessionId;
    this.groupCallId = opts.groupCallId;
    // Array of Objects with urls, username, credential keys
    this.turnServers = opts.turnServers || [];
    if (this.turnServers.length === 0 && this.client.isFallbackICEServerAllowed()) {
      this.turnServers.push({
        urls: [FALLBACK_ICE_SERVER]
      });
    }
    for (var server of this.turnServers) {
      checkObjectHasKeys(server, ["urls"]);
    }
    this.callId = genCallID();
    // If the Client provides calls without audio and video we need a datachannel for a webrtc connection
    this.isOnlyDataChannelAllowed = this.client.isVoipWithNoMediaAllowed;
  }

  /**
   * Place a voice call to this room.
   * @throws If you have not specified a listener for 'error' events.
   */
  placeVoiceCall() {
    var _this2 = this;
    return _asyncToGenerator(function* () {
      yield _this2.placeCall(true, false);
    })();
  }

  /**
   * Place a video call to this room.
   * @throws If you have not specified a listener for 'error' events.
   */
  placeVideoCall() {
    var _this3 = this;
    return _asyncToGenerator(function* () {
      yield _this3.placeCall(true, true);
    })();
  }

  /**
   * Create a datachannel using this call's peer connection.
   * @param label - A human readable label for this datachannel
   * @param options - An object providing configuration options for the data channel.
   */
  createDataChannel(label, options) {
    var dataChannel = this.peerConn.createDataChannel(label, options);
    this.emit(CallEvent.DataChannel, dataChannel, this);
    return dataChannel;
  }
  getOpponentMember() {
    return this.opponentMember;
  }
  getOpponentDeviceId() {
    return this.opponentDeviceId;
  }
  getOpponentSessionId() {
    return this.opponentSessionId;
  }
  opponentCanBeTransferred() {
    return Boolean(this.opponentCaps && this.opponentCaps["m.call.transferee"]);
  }
  opponentSupportsDTMF() {
    return Boolean(this.opponentCaps && this.opponentCaps["m.call.dtmf"]);
  }
  getRemoteAssertedIdentity() {
    return this.remoteAssertedIdentity;
  }
  get state() {
    return this._state;
  }
  set state(state) {
    var oldState = this._state;
    this._state = state;
    this.emit(CallEvent.State, state, oldState, this);
  }
  get type() {
    // we may want to look for a video receiver here rather than a track to match the
    // sender behaviour, although in practice they should be the same thing
    return this.hasUserMediaVideoSender || this.hasRemoteUserMediaVideoTrack ? CallType.Video : CallType.Voice;
  }
  get hasLocalUserMediaVideoTrack() {
    var _this$localUsermediaS;
    return !!((_this$localUsermediaS = this.localUsermediaStream) !== null && _this$localUsermediaS !== void 0 && _this$localUsermediaS.getVideoTracks().length);
  }
  get hasRemoteUserMediaVideoTrack() {
    return this.getRemoteFeeds().some(feed => {
      var _feed$stream;
      return feed.purpose === SDPStreamMetadataPurpose.Usermedia && ((_feed$stream = feed.stream) === null || _feed$stream === void 0 ? void 0 : _feed$stream.getVideoTracks().length);
    });
  }
  get hasLocalUserMediaAudioTrack() {
    var _this$localUsermediaS2;
    return !!((_this$localUsermediaS2 = this.localUsermediaStream) !== null && _this$localUsermediaS2 !== void 0 && _this$localUsermediaS2.getAudioTracks().length);
  }
  get hasRemoteUserMediaAudioTrack() {
    return this.getRemoteFeeds().some(feed => {
      var _feed$stream2;
      return feed.purpose === SDPStreamMetadataPurpose.Usermedia && !!((_feed$stream2 = feed.stream) !== null && _feed$stream2 !== void 0 && _feed$stream2.getAudioTracks().length);
    });
  }
  get hasUserMediaAudioSender() {
    var _this$transceivers$ge;
    return Boolean((_this$transceivers$ge = this.transceivers.get(getTransceiverKey(SDPStreamMetadataPurpose.Usermedia, "audio"))) === null || _this$transceivers$ge === void 0 ? void 0 : _this$transceivers$ge.sender);
  }
  get hasUserMediaVideoSender() {
    var _this$transceivers$ge2;
    return Boolean((_this$transceivers$ge2 = this.transceivers.get(getTransceiverKey(SDPStreamMetadataPurpose.Usermedia, "video"))) === null || _this$transceivers$ge2 === void 0 ? void 0 : _this$transceivers$ge2.sender);
  }
  get localUsermediaFeed() {
    return this.getLocalFeeds().find(feed => feed.purpose === SDPStreamMetadataPurpose.Usermedia);
  }
  get localScreensharingFeed() {
    return this.getLocalFeeds().find(feed => feed.purpose === SDPStreamMetadataPurpose.Screenshare);
  }
  get localUsermediaStream() {
    var _this$localUsermediaF;
    return (_this$localUsermediaF = this.localUsermediaFeed) === null || _this$localUsermediaF === void 0 ? void 0 : _this$localUsermediaF.stream;
  }
  get localScreensharingStream() {
    var _this$localScreenshar;
    return (_this$localScreenshar = this.localScreensharingFeed) === null || _this$localScreenshar === void 0 ? void 0 : _this$localScreenshar.stream;
  }
  get remoteUsermediaFeed() {
    return this.getRemoteFeeds().find(feed => feed.purpose === SDPStreamMetadataPurpose.Usermedia);
  }
  get remoteScreensharingFeed() {
    return this.getRemoteFeeds().find(feed => feed.purpose === SDPStreamMetadataPurpose.Screenshare);
  }
  get remoteUsermediaStream() {
    var _this$remoteUsermedia;
    return (_this$remoteUsermedia = this.remoteUsermediaFeed) === null || _this$remoteUsermedia === void 0 ? void 0 : _this$remoteUsermedia.stream;
  }
  get remoteScreensharingStream() {
    var _this$remoteScreensha;
    return (_this$remoteScreensha = this.remoteScreensharingFeed) === null || _this$remoteScreensha === void 0 ? void 0 : _this$remoteScreensha.stream;
  }
  getFeedByStreamId(streamId) {
    return this.getFeeds().find(feed => feed.stream.id === streamId);
  }

  /**
   * Returns an array of all CallFeeds
   * @returns CallFeeds
   */
  getFeeds() {
    return this.feeds;
  }

  /**
   * Returns an array of all local CallFeeds
   * @returns local CallFeeds
   */
  getLocalFeeds() {
    return this.feeds.filter(feed => feed.isLocal());
  }

  /**
   * Returns an array of all remote CallFeeds
   * @returns remote CallFeeds
   */
  getRemoteFeeds() {
    return this.feeds.filter(feed => !feed.isLocal());
  }
  initOpponentCrypto() {
    var _this4 = this;
    return _asyncToGenerator(function* () {
      var _this4$getOpponentMem;
      if (!_this4.opponentDeviceId) return;
      if (!_this4.client.getUseE2eForGroupCall()) return;
      // It's possible to want E2EE and yet not have the means to manage E2EE
      // ourselves (for example if the client is a RoomWidgetClient)
      if (!_this4.client.getCrypto()) {
        // All we know is the device ID
        _this4.hasOpponentDeviceInfo = true;
        return;
      }
      var userId = _this4.invitee || ((_this4$getOpponentMem = _this4.getOpponentMember()) === null || _this4$getOpponentMem === void 0 ? void 0 : _this4$getOpponentMem.userId);
      if (!userId) throw new Error("Couldn't find opponent user ID to init crypto");

      // Here we were calling `MatrixClient.crypto.deviceList.downloadKeys` which is not supported by the rust cryptography.
      _this4.hasOpponentDeviceInfo = false;
      throw new GroupCallUnknownDeviceError(userId);
    })();
  }

  /**
   * Generates and returns localSDPStreamMetadata
   * @returns localSDPStreamMetadata
   */
  getLocalSDPStreamMetadata() {
    var updateStreamIds = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : false;
    var metadata = {};
    for (var localFeed of this.getLocalFeeds()) {
      if (updateStreamIds) {
        localFeed.sdpMetadataStreamId = localFeed.stream.id;
      }
      metadata[localFeed.sdpMetadataStreamId] = {
        purpose: localFeed.purpose,
        audio_muted: localFeed.isAudioMuted(),
        video_muted: localFeed.isVideoMuted()
      };
    }
    return metadata;
  }

  /**
   * Returns true if there are no incoming feeds,
   * otherwise returns false
   * @returns no incoming feeds
   */
  noIncomingFeeds() {
    return !this.feeds.some(feed => !feed.isLocal());
  }
  pushRemoteFeed(stream) {
    // Fallback to old behavior if the other side doesn't support SDPStreamMetadata
    if (!this.opponentSupportsSDPStreamMetadata()) {
      this.pushRemoteFeedWithoutMetadata(stream);
      return;
    }
    var userId = this.getOpponentMember().userId;
    var purpose = this.remoteSDPStreamMetadata[stream.id].purpose;
    var audioMuted = this.remoteSDPStreamMetadata[stream.id].audio_muted;
    var videoMuted = this.remoteSDPStreamMetadata[stream.id].video_muted;
    if (!purpose) {
      logger.warn("Call ".concat(this.callId, " pushRemoteFeed() ignoring stream because we didn't get any metadata about it (streamId=").concat(stream.id, ")"));
      return;
    }
    if (this.getFeedByStreamId(stream.id)) {
      logger.warn("Call ".concat(this.callId, " pushRemoteFeed() ignoring stream because we already have a feed for it (streamId=").concat(stream.id, ")"));
      return;
    }
    this.feeds.push(new CallFeed({
      client: this.client,
      call: this,
      roomId: this.roomId,
      userId,
      deviceId: this.getOpponentDeviceId(),
      stream,
      purpose,
      audioMuted,
      videoMuted
    }));
    this.emit(CallEvent.FeedsChanged, this.feeds, this);
    logger.info("Call ".concat(this.callId, " pushRemoteFeed() pushed stream (streamId=").concat(stream.id, ", active=").concat(stream.active, ", purpose=").concat(purpose, ")"));
  }

  /**
   * This method is used ONLY if the other client doesn't support sending SDPStreamMetadata
   */
  pushRemoteFeedWithoutMetadata(stream) {
    var _this$feeds$find;
    var userId = this.getOpponentMember().userId;
    // We can guess the purpose here since the other client can only send one stream
    var purpose = SDPStreamMetadataPurpose.Usermedia;
    var oldRemoteStream = (_this$feeds$find = this.feeds.find(feed => !feed.isLocal())) === null || _this$feeds$find === void 0 ? void 0 : _this$feeds$find.stream;

    // Note that we check by ID and always set the remote stream: Chrome appears
    // to make new stream objects when transceiver directionality is changed and the 'active'
    // status of streams change - Dave
    // If we already have a stream, check this stream has the same id
    if (oldRemoteStream && stream.id !== oldRemoteStream.id) {
      logger.warn("Call ".concat(this.callId, " pushRemoteFeedWithoutMetadata() ignoring new stream because we already have stream (streamId=").concat(stream.id, ")"));
      return;
    }
    if (this.getFeedByStreamId(stream.id)) {
      logger.warn("Call ".concat(this.callId, " pushRemoteFeedWithoutMetadata() ignoring stream because we already have a feed for it (streamId=").concat(stream.id, ")"));
      return;
    }
    this.feeds.push(new CallFeed({
      client: this.client,
      call: this,
      roomId: this.roomId,
      audioMuted: false,
      videoMuted: false,
      userId,
      deviceId: this.getOpponentDeviceId(),
      stream,
      purpose
    }));
    this.emit(CallEvent.FeedsChanged, this.feeds, this);
    logger.info("Call ".concat(this.callId, " pushRemoteFeedWithoutMetadata() pushed stream (streamId=").concat(stream.id, ", active=").concat(stream.active, ")"));
  }
  pushNewLocalFeed(stream, purpose) {
    var addToPeerConnection = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : true;
    var userId = this.client.getUserId();

    // Tracks don't always start off enabled, eg. chrome will give a disabled
    // audio track if you ask for user media audio and already had one that
    // you'd set to disabled (presumably because it clones them internally).
    setTracksEnabled(stream.getAudioTracks(), true);
    setTracksEnabled(stream.getVideoTracks(), true);
    if (this.getFeedByStreamId(stream.id)) {
      logger.warn("Call ".concat(this.callId, " pushNewLocalFeed() ignoring stream because we already have a feed for it (streamId=").concat(stream.id, ")"));
      return;
    }
    this.pushLocalFeed(new CallFeed({
      client: this.client,
      roomId: this.roomId,
      audioMuted: false,
      videoMuted: false,
      userId,
      deviceId: this.getOpponentDeviceId(),
      stream,
      purpose
    }), addToPeerConnection);
  }

  /**
   * Pushes supplied feed to the call
   * @param callFeed - to push
   * @param addToPeerConnection - whether to add the tracks to the peer connection
   */
  pushLocalFeed(callFeed) {
    var _this5 = this;
    var addToPeerConnection = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : true;
    if (this.feeds.some(feed => callFeed.stream.id === feed.stream.id)) {
      logger.info("Call ".concat(this.callId, " pushLocalFeed() ignoring duplicate local stream (streamId=").concat(callFeed.stream.id, ")"));
      return;
    }
    this.feeds.push(callFeed);
    if (addToPeerConnection) {
      var _loop = function _loop() {
        logger.info("Call ".concat(_this5.callId, " pushLocalFeed() adding track to peer connection (id=").concat(track.id, ", kind=").concat(track.kind, ", streamId=").concat(callFeed.stream.id, ", streamPurpose=").concat(callFeed.purpose, ", enabled=").concat(track.enabled, ")"));
        var tKey = getTransceiverKey(callFeed.purpose, track.kind);
        if (_this5.transceivers.has(tKey)) {
          // we already have a sender, so we re-use it. We try to re-use transceivers as much
          // as possible because they can't be removed once added, so otherwise they just
          // accumulate which makes the SDP very large very quickly: in fact it only takes
          // about 6 video tracks to exceed the maximum size of an Olm-encrypted
          // Matrix event.
          var transceiver = _this5.transceivers.get(tKey);
          transceiver.sender.replaceTrack(track);
          // set the direction to indicate we're going to start sending again
          // (this will trigger the re-negotiation)
          transceiver.direction = transceiver.direction === "inactive" ? "sendonly" : "sendrecv";
        } else {
          // create a new one. We need to use addTrack rather addTransceiver for this because firefox
          // doesn't yet implement RTCRTPSender.setStreams()
          // (https://bugzilla.mozilla.org/show_bug.cgi?id=1510802) so we'd have no way to group the
          // two tracks together into a stream.
          var newSender = _this5.peerConn.addTrack(track, callFeed.stream);

          // now go & fish for the new transceiver
          var newTransceiver = _this5.peerConn.getTransceivers().find(t => t.sender === newSender);
          if (newTransceiver) {
            _this5.transceivers.set(tKey, newTransceiver);
          } else {
            logger.warn("Call ".concat(_this5.callId, " pushLocalFeed() didn't find a matching transceiver after adding track!"));
          }
        }
      };
      for (var track of callFeed.stream.getTracks()) {
        _loop();
      }
    }
    logger.info("Call ".concat(this.callId, " pushLocalFeed() pushed stream (id=").concat(callFeed.stream.id, ", active=").concat(callFeed.stream.active, ", purpose=").concat(callFeed.purpose, ")"));
    this.emit(CallEvent.FeedsChanged, this.feeds, this);
  }

  /**
   * Removes local call feed from the call and its tracks from the peer
   * connection
   * @param callFeed - to remove
   */
  removeLocalFeed(callFeed) {
    var audioTransceiverKey = getTransceiverKey(callFeed.purpose, "audio");
    var videoTransceiverKey = getTransceiverKey(callFeed.purpose, "video");
    for (var transceiverKey of [audioTransceiverKey, videoTransceiverKey]) {
      // this is slightly mixing the track and transceiver API but is basically just shorthand.
      // There is no way to actually remove a transceiver, so this just sets it to inactive
      // (or recvonly) and replaces the source with nothing.
      if (this.transceivers.has(transceiverKey)) {
        var transceiver = this.transceivers.get(transceiverKey);
        if (transceiver.sender) this.peerConn.removeTrack(transceiver.sender);
      }
    }
    if (callFeed.purpose === SDPStreamMetadataPurpose.Screenshare) {
      this.client.getMediaHandler().stopScreensharingStream(callFeed.stream);
    }
    this.deleteFeed(callFeed);
  }
  deleteAllFeeds() {
    for (var feed of this.feeds) {
      if (!feed.isLocal() || !this.groupCallId) {
        feed.dispose();
      }
    }
    this.feeds = [];
    this.emit(CallEvent.FeedsChanged, this.feeds, this);
  }
  deleteFeedByStream(stream) {
    var feed = this.getFeedByStreamId(stream.id);
    if (!feed) {
      logger.warn("Call ".concat(this.callId, " deleteFeedByStream() didn't find the feed to delete (streamId=").concat(stream.id, ")"));
      return;
    }
    this.deleteFeed(feed);
  }
  deleteFeed(feed) {
    feed.dispose();
    this.feeds.splice(this.feeds.indexOf(feed), 1);
    this.emit(CallEvent.FeedsChanged, this.feeds, this);
  }

  // The typescript definitions have this type as 'any' :(
  getCurrentCallStats() {
    var _this6 = this;
    return _asyncToGenerator(function* () {
      if (_this6.callHasEnded()) {
        return _this6.callStatsAtEnd;
      }
      return _this6.collectCallStats();
    })();
  }
  collectCallStats() {
    var _this7 = this;
    return _asyncToGenerator(function* () {
      // This happens when the call fails before it starts.
      // For example when we fail to get capture sources
      if (!_this7.peerConn) return;
      var statsReport = yield _this7.peerConn.getStats();
      var stats = [];
      statsReport.forEach(item => {
        stats.push(item);
      });
      return stats;
    })();
  }

  /**
   * Configure this call from an invite event. Used by MatrixClient.
   * @param event - The m.call.invite event
   */
  initWithInvite(event) {
    var _this8 = this;
    return _asyncToGenerator(function* () {
      var _this8$feeds$find;
      var invite = event.getContent();
      _this8.direction = CallDirection.Inbound;

      // make sure we have valid turn creds. Unless something's gone wrong, it should
      // poll and keep the credentials valid so this should be instant.
      var haveTurnCreds = yield _this8.client.checkTurnServers();
      if (!haveTurnCreds) {
        logger.warn("Call ".concat(_this8.callId, " initWithInvite() failed to get TURN credentials! Proceeding with call anyway..."));
      }
      var sdpStreamMetadata = invite[SDPStreamMetadataKey];
      if (sdpStreamMetadata) {
        _this8.updateRemoteSDPStreamMetadata(sdpStreamMetadata);
      } else {
        logger.debug("Call ".concat(_this8.callId, " initWithInvite() did not get any SDPStreamMetadata! Can not send/receive multiple streams"));
      }
      _this8.peerConn = _this8.createPeerConnection();
      _this8.emit(CallEvent.PeerConnectionCreated, _this8.peerConn, _this8);
      // we must set the party ID before await-ing on anything: the call event
      // handler will start giving us more call events (eg. candidates) so if
      // we haven't set the party ID, we'll ignore them.
      _this8.chooseOpponent(event);
      yield _this8.initOpponentCrypto();
      try {
        yield _this8.peerConn.setRemoteDescription(invite.offer);
        logger.debug("Call ".concat(_this8.callId, " initWithInvite() set remote description: ").concat(invite.offer.type));
        yield _this8.addBufferedIceCandidates();
      } catch (e) {
        logger.debug("Call ".concat(_this8.callId, " initWithInvite() failed to set remote description"), e);
        _this8.terminate(CallParty.Local, CallErrorCode.SetRemoteDescription, false);
        return;
      }
      var remoteStream = (_this8$feeds$find = _this8.feeds.find(feed => !feed.isLocal())) === null || _this8$feeds$find === void 0 ? void 0 : _this8$feeds$find.stream;

      // According to previous comments in this file, firefox at some point did not
      // add streams until media started arriving on them. Testing latest firefox
      // (81 at time of writing), this is no longer a problem, so let's do it the correct way.
      //
      // For example in case of no media webrtc connections like screen share only call we have to allow webrtc
      // connections without remote media. In this case we always use a data channel. At the moment we allow as well
      // only data channel as media in the WebRTC connection with this setup here.
      if (!_this8.isOnlyDataChannelAllowed && (!remoteStream || remoteStream.getTracks().length === 0)) {
        logger.error("Call ".concat(_this8.callId, " initWithInvite() no remote stream or no tracks after setting remote description!"));
        _this8.terminate(CallParty.Local, CallErrorCode.SetRemoteDescription, false);
        return;
      }
      _this8.state = CallState.Ringing;
      if (event.getLocalAge()) {
        // Time out the call if it's ringing for too long
        var ringingTimer = setTimeout(() => {
          if (_this8.state == CallState.Ringing) {
            var _this8$stats;
            logger.debug("Call ".concat(_this8.callId, " initWithInvite() invite has expired. Hanging up."));
            _this8.hangupParty = CallParty.Remote; // effectively
            _this8.state = CallState.Ended;
            _this8.stopAllMedia();
            if (_this8.peerConn.signalingState != "closed") {
              _this8.peerConn.close();
            }
            (_this8$stats = _this8.stats) === null || _this8$stats === void 0 || _this8$stats.removeStatsReportGatherer(_this8.callId);
            _this8.emit(CallEvent.Hangup, _this8);
          }
        }, invite.lifetime - event.getLocalAge());
        var onState = state => {
          if (state !== CallState.Ringing) {
            clearTimeout(ringingTimer);
            _this8.off(CallEvent.State, onState);
          }
        };
        _this8.on(CallEvent.State, onState);
      }
    })();
  }

  /**
   * Configure this call from a hangup or reject event. Used by MatrixClient.
   * @param event - The m.call.hangup event
   */
  initWithHangup(event) {
    // perverse as it may seem, sometimes we want to instantiate a call with a
    // hangup message (because when getting the state of the room on load, events
    // come in reverse order and we want to remember that a call has been hung up)
    this.state = CallState.Ended;
  }
  shouldAnswerWithMediaType(wantedValue, valueOfTheOtherSide, type) {
    if (wantedValue && !valueOfTheOtherSide) {
      // TODO: Figure out how to do this
      logger.warn("Call ".concat(this.callId, " shouldAnswerWithMediaType() unable to answer with ").concat(type, " because the other side isn't sending it either."));
      return false;
    } else if (!isNullOrUndefined(wantedValue) && wantedValue !== valueOfTheOtherSide && !this.opponentSupportsSDPStreamMetadata()) {
      logger.warn("Call ".concat(this.callId, " shouldAnswerWithMediaType() unable to answer with ").concat(type, "=").concat(wantedValue, " because the other side doesn't support it. Answering with ").concat(type, "=").concat(valueOfTheOtherSide, "."));
      return valueOfTheOtherSide;
    }
    return wantedValue !== null && wantedValue !== void 0 ? wantedValue : valueOfTheOtherSide;
  }

  /**
   * Answer a call.
   */
  answer(audio, video) {
    var _this9 = this;
    return _asyncToGenerator(function* () {
      if (_this9.inviteOrAnswerSent) return;
      // TODO: Figure out how to do this
      if (audio === false && video === false) throw new Error("You CANNOT answer a call without media");
      if (!_this9.localUsermediaStream && !_this9.waitForLocalAVStream) {
        var prevState = _this9.state;
        var answerWithAudio = _this9.shouldAnswerWithMediaType(audio, _this9.hasRemoteUserMediaAudioTrack, "audio");
        var answerWithVideo = _this9.shouldAnswerWithMediaType(video, _this9.hasRemoteUserMediaVideoTrack, "video");
        _this9.state = CallState.WaitLocalMedia;
        _this9.waitForLocalAVStream = true;
        try {
          var _this9$client$getDevi;
          var stream = yield _this9.client.getMediaHandler().getUserMediaStream(answerWithAudio, answerWithVideo);
          _this9.waitForLocalAVStream = false;
          var usermediaFeed = new CallFeed({
            client: _this9.client,
            roomId: _this9.roomId,
            userId: _this9.client.getUserId(),
            deviceId: (_this9$client$getDevi = _this9.client.getDeviceId()) !== null && _this9$client$getDevi !== void 0 ? _this9$client$getDevi : undefined,
            stream,
            purpose: SDPStreamMetadataPurpose.Usermedia,
            audioMuted: false,
            videoMuted: false
          });
          var feeds = [usermediaFeed];
          if (_this9.localScreensharingFeed) {
            feeds.push(_this9.localScreensharingFeed);
          }
          _this9.answerWithCallFeeds(feeds);
        } catch (e) {
          if (answerWithVideo) {
            // Try to answer without video
            logger.warn("Call ".concat(_this9.callId, " answer() failed to getUserMedia(), trying to getUserMedia() without video"));
            _this9.state = prevState;
            _this9.waitForLocalAVStream = false;
            yield _this9.answer(answerWithAudio, false);
          } else {
            _this9.getUserMediaFailed(e);
            return;
          }
        }
      } else if (_this9.waitForLocalAVStream) {
        _this9.state = CallState.WaitLocalMedia;
      }
    })();
  }
  answerWithCallFeeds(callFeeds) {
    if (this.inviteOrAnswerSent) return;
    this.queueGotCallFeedsForAnswer(callFeeds);
  }

  /**
   * Replace this call with a new call, e.g. for glare resolution. Used by
   * MatrixClient.
   * @param newCall - The new call.
   */
  replacedBy(newCall) {
    logger.debug("Call ".concat(this.callId, " replacedBy() running (newCallId=").concat(newCall.callId, ")"));
    if (this.state === CallState.WaitLocalMedia) {
      logger.debug("Call ".concat(this.callId, " replacedBy() telling new call to wait for local media (newCallId=").concat(newCall.callId, ")"));
      newCall.waitForLocalAVStream = true;
    } else if ([CallState.CreateOffer, CallState.InviteSent].includes(this.state)) {
      if (newCall.direction === CallDirection.Outbound) {
        newCall.queueGotCallFeedsForAnswer([]);
      } else {
        logger.debug("Call ".concat(this.callId, " replacedBy() handing local stream to new call(newCallId=").concat(newCall.callId, ")"));
        newCall.queueGotCallFeedsForAnswer(this.getLocalFeeds().map(feed => feed.clone()));
      }
    }
    this.successor = newCall;
    this.emit(CallEvent.Replaced, newCall, this);
    this.hangup(CallErrorCode.Replaced, true);
  }

  /**
   * Hangup a call.
   * @param reason - The reason why the call is being hung up.
   * @param suppressEvent - True to suppress emitting an event.
   */
  hangup(reason, suppressEvent) {
    if (this.callHasEnded()) return;
    logger.debug("Call ".concat(this.callId, " hangup() ending call (reason=").concat(reason, ")"));
    this.terminate(CallParty.Local, reason, !suppressEvent);
    // We don't want to send hangup here if we didn't even get to sending an invite
    if ([CallState.Fledgling, CallState.WaitLocalMedia].includes(this.state)) return;
    var content = {};
    // Don't send UserHangup reason to older clients
    if (this.opponentVersion && this.opponentVersion !== 0 || reason !== CallErrorCode.UserHangup) {
      content["reason"] = reason;
    }
    this.sendVoipEvent(EventType.CallHangup, content);
  }

  /**
   * Reject a call
   * This used to be done by calling hangup, but is a separate method and protocol
   * event as of MSC2746.
   */
  reject() {
    if (this.state !== CallState.Ringing) {
      throw Error("Call must be in 'ringing' state to reject!");
    }
    if (this.opponentVersion === 0) {
      logger.info("Call ".concat(this.callId, " reject() opponent version is less than 1: sending hangup instead of reject (opponentVersion=").concat(this.opponentVersion, ")"));
      this.hangup(CallErrorCode.UserHangup, true);
      return;
    }
    logger.debug("Rejecting call: " + this.callId);
    this.terminate(CallParty.Local, CallErrorCode.UserHangup, true);
    this.sendVoipEvent(EventType.CallReject, {});
  }

  /**
   * Adds an audio and/or video track - upgrades the call
   * @param audio - should add an audio track
   * @param video - should add an video track
   */
  upgradeCall(audio, video) {
    var _this0 = this;
    return _asyncToGenerator(function* () {
      // We don't do call downgrades
      if (!audio && !video) return;
      if (!_this0.opponentSupportsSDPStreamMetadata()) return;
      try {
        logger.debug("Call ".concat(_this0.callId, " upgradeCall() upgrading call (audio=").concat(audio, ", video=").concat(video, ")"));
        var getAudio = audio || _this0.hasLocalUserMediaAudioTrack;
        var getVideo = video || _this0.hasLocalUserMediaVideoTrack;

        // updateLocalUsermediaStream() will take the tracks, use them as
        // replacement and throw the stream away, so it isn't reusable
        var stream = yield _this0.client.getMediaHandler().getUserMediaStream(getAudio, getVideo, false);
        yield _this0.updateLocalUsermediaStream(stream, audio, video);
      } catch (error) {
        logger.error("Call ".concat(_this0.callId, " upgradeCall() failed to upgrade the call"), error);
        _this0.emit(CallEvent.Error, new CallError(CallErrorCode.NoUserMedia, "Failed to get camera access: ", error), _this0);
      }
    })();
  }

  /**
   * Returns true if this.remoteSDPStreamMetadata is defined, otherwise returns false
   * @returns can screenshare
   */
  opponentSupportsSDPStreamMetadata() {
    return Boolean(this.remoteSDPStreamMetadata);
  }

  /**
   * If there is a screensharing stream returns true, otherwise returns false
   * @returns is screensharing
   */
  isScreensharing() {
    return Boolean(this.localScreensharingStream);
  }

  /**
   * Starts/stops screensharing
   * @param enabled - the desired screensharing state
   * @param opts - screen sharing options
   * @returns new screensharing state
   */
  setScreensharingEnabled(enabled, opts) {
    var _this1 = this;
    return _asyncToGenerator(function* () {
      // Skip if there is nothing to do
      if (enabled && _this1.isScreensharing()) {
        logger.warn("Call ".concat(_this1.callId, " setScreensharingEnabled() there is already a screensharing stream - there is nothing to do!"));
        return true;
      } else if (!enabled && !_this1.isScreensharing()) {
        logger.warn("Call ".concat(_this1.callId, " setScreensharingEnabled() there already isn't a screensharing stream - there is nothing to do!"));
        return false;
      }

      // Fallback to replaceTrack()
      if (!_this1.opponentSupportsSDPStreamMetadata()) {
        return _this1.setScreensharingEnabledWithoutMetadataSupport(enabled, opts);
      }
      logger.debug("Call ".concat(_this1.callId, " setScreensharingEnabled() running (enabled=").concat(enabled, ")"));
      if (enabled) {
        try {
          var stream = yield _this1.client.getMediaHandler().getScreensharingStream(opts);
          if (!stream) return false;
          _this1.pushNewLocalFeed(stream, SDPStreamMetadataPurpose.Screenshare);
          return true;
        } catch (err) {
          logger.error("Call ".concat(_this1.callId, " setScreensharingEnabled() failed to get screen-sharing stream:"), err);
          return false;
        }
      } else {
        var audioTransceiver = _this1.transceivers.get(getTransceiverKey(SDPStreamMetadataPurpose.Screenshare, "audio"));
        var videoTransceiver = _this1.transceivers.get(getTransceiverKey(SDPStreamMetadataPurpose.Screenshare, "video"));
        for (var transceiver of [audioTransceiver, videoTransceiver]) {
          // this is slightly mixing the track and transceiver API but is basically just shorthand
          // for removing the sender.
          if (transceiver && transceiver.sender) _this1.peerConn.removeTrack(transceiver.sender);
        }
        _this1.client.getMediaHandler().stopScreensharingStream(_this1.localScreensharingStream);
        _this1.deleteFeedByStream(_this1.localScreensharingStream);
        return false;
      }
    })();
  }

  /**
   * Starts/stops screensharing
   * Should be used ONLY if the opponent doesn't support SDPStreamMetadata
   * @param enabled - the desired screensharing state
   * @param opts - screen sharing options
   * @returns new screensharing state
   */
  setScreensharingEnabledWithoutMetadataSupport(enabled, opts) {
    var _this10 = this;
    return _asyncToGenerator(function* () {
      logger.debug("Call ".concat(_this10.callId, " setScreensharingEnabledWithoutMetadataSupport() running (enabled=").concat(enabled, ")"));
      if (enabled) {
        try {
          var _this10$transceivers$;
          var stream = yield _this10.client.getMediaHandler().getScreensharingStream(opts);
          if (!stream) return false;
          var track = stream.getTracks().find(track => track.kind === "video");
          var sender = (_this10$transceivers$ = _this10.transceivers.get(getTransceiverKey(SDPStreamMetadataPurpose.Usermedia, "video"))) === null || _this10$transceivers$ === void 0 ? void 0 : _this10$transceivers$.sender;
          sender === null || sender === void 0 || sender.replaceTrack(track !== null && track !== void 0 ? track : null);
          _this10.pushNewLocalFeed(stream, SDPStreamMetadataPurpose.Screenshare, false);
          return true;
        } catch (err) {
          logger.error("Call ".concat(_this10.callId, " setScreensharingEnabledWithoutMetadataSupport() failed to get screen-sharing stream:"), err);
          return false;
        }
      } else {
        var _this10$localUsermedi, _this10$transceivers$2;
        var _track = (_this10$localUsermedi = _this10.localUsermediaStream) === null || _this10$localUsermedi === void 0 ? void 0 : _this10$localUsermedi.getTracks().find(track => track.kind === "video");
        var _sender = (_this10$transceivers$2 = _this10.transceivers.get(getTransceiverKey(SDPStreamMetadataPurpose.Usermedia, "video"))) === null || _this10$transceivers$2 === void 0 ? void 0 : _this10$transceivers$2.sender;
        _sender === null || _sender === void 0 || _sender.replaceTrack(_track !== null && _track !== void 0 ? _track : null);
        _this10.client.getMediaHandler().stopScreensharingStream(_this10.localScreensharingStream);
        _this10.deleteFeedByStream(_this10.localScreensharingStream);
        return false;
      }
    })();
  }

  /**
   * Replaces/adds the tracks from the passed stream to the localUsermediaStream
   * @param stream - to use a replacement for the local usermedia stream
   */
  updateLocalUsermediaStream(stream) {
    var _arguments = arguments,
      _this11 = this;
    return _asyncToGenerator(function* () {
      var forceAudio = _arguments.length > 1 && _arguments[1] !== undefined ? _arguments[1] : false;
      var forceVideo = _arguments.length > 2 && _arguments[2] !== undefined ? _arguments[2] : false;
      var callFeed = _this11.localUsermediaFeed;
      var audioEnabled = forceAudio || !callFeed.isAudioMuted() && !_this11.remoteOnHold;
      var videoEnabled = forceVideo || !callFeed.isVideoMuted() && !_this11.remoteOnHold;
      logger.log("Call ".concat(_this11.callId, " updateLocalUsermediaStream() running (streamId=").concat(stream.id, ", audio=").concat(audioEnabled, ", video=").concat(videoEnabled, ")"));
      setTracksEnabled(stream.getAudioTracks(), audioEnabled);
      setTracksEnabled(stream.getVideoTracks(), videoEnabled);

      // We want to keep the same stream id, so we replace the tracks rather
      // than the whole stream.

      // Firstly, we replace the tracks in our localUsermediaStream.
      for (var track of _this11.localUsermediaStream.getTracks()) {
        _this11.localUsermediaStream.removeTrack(track);
        track.stop();
      }
      for (var _track2 of stream.getTracks()) {
        _this11.localUsermediaStream.addTrack(_track2);
      }

      // Then replace the old tracks, if possible.
      var _loop2 = function* _loop2() {
        var tKey = getTransceiverKey(SDPStreamMetadataPurpose.Usermedia, _track3.kind);
        var transceiver = _this11.transceivers.get(tKey);
        var oldSender = transceiver === null || transceiver === void 0 ? void 0 : transceiver.sender;
        var added = false;
        if (oldSender) {
          try {
            logger.info("Call ".concat(_this11.callId, " updateLocalUsermediaStream() replacing track (id=").concat(_track3.id, ", kind=").concat(_track3.kind, ", streamId=").concat(stream.id, ", streamPurpose=").concat(callFeed.purpose, ")"));
            yield oldSender.replaceTrack(_track3);
            // Set the direction to indicate we're going to be sending.
            // This is only necessary in the cases where we're upgrading
            // the call to video after downgrading it.
            transceiver.direction = transceiver.direction === "inactive" ? "sendonly" : "sendrecv";
            added = true;
          } catch (error) {
            logger.warn("Call ".concat(_this11.callId, " updateLocalUsermediaStream() replaceTrack failed: adding new transceiver instead"), error);
          }
        }
        if (!added) {
          logger.info("Call ".concat(_this11.callId, " updateLocalUsermediaStream() adding track to peer connection (id=").concat(_track3.id, ", kind=").concat(_track3.kind, ", streamId=").concat(stream.id, ", streamPurpose=").concat(callFeed.purpose, ")"));
          var newSender = _this11.peerConn.addTrack(_track3, _this11.localUsermediaStream);
          var newTransceiver = _this11.peerConn.getTransceivers().find(t => t.sender === newSender);
          if (newTransceiver) {
            _this11.transceivers.set(tKey, newTransceiver);
          } else {
            logger.warn("Call ".concat(_this11.callId, " updateLocalUsermediaStream() couldn't find matching transceiver for newly added track!"));
          }
        }
      };
      for (var _track3 of stream.getTracks()) {
        yield* _loop2();
      }
    })();
  }

  /**
   * Set whether our outbound video should be muted or not.
   * @param muted - True to mute the outbound video.
   * @returns the new mute state
   */
  setLocalVideoMuted(muted) {
    var _this12 = this;
    return _asyncToGenerator(function* () {
      var _this12$localUsermedi2;
      logger.log("Call ".concat(_this12.callId, " setLocalVideoMuted() running ").concat(muted));

      // if we were still thinking about stopping and removing the video
      // track: don't, because we want it back.
      if (!muted && _this12.stopVideoTrackTimer !== undefined) {
        clearTimeout(_this12.stopVideoTrackTimer);
        _this12.stopVideoTrackTimer = undefined;
      }
      if (!(yield _this12.client.getMediaHandler().hasVideoDevice())) {
        return _this12.isLocalVideoMuted();
      }
      if (!_this12.hasUserMediaVideoSender && !muted) {
        var _this12$localUsermedi;
        (_this12$localUsermedi = _this12.localUsermediaFeed) === null || _this12$localUsermedi === void 0 || _this12$localUsermedi.setAudioVideoMuted(null, muted);
        yield _this12.upgradeCall(false, true);
        return _this12.isLocalVideoMuted();
      }

      // we may not have a video track - if not, re-request usermedia
      if (!muted && _this12.localUsermediaStream.getVideoTracks().length === 0) {
        var stream = yield _this12.client.getMediaHandler().getUserMediaStream(true, true);
        yield _this12.updateLocalUsermediaStream(stream);
      }
      (_this12$localUsermedi2 = _this12.localUsermediaFeed) === null || _this12$localUsermedi2 === void 0 || _this12$localUsermedi2.setAudioVideoMuted(null, muted);
      _this12.updateMuteStatus();
      yield _this12.sendMetadataUpdate();

      // if we're muting video, set a timeout to stop & remove the video track so we release
      // the camera. We wait a short time to do this because when we disable a track, WebRTC
      // will send black video for it. If we just stop and remove it straight away, the video
      // will just freeze which means that when we unmute video, the other side will briefly
      // get a static frame of us from before we muted. This way, the still frame is just black.
      // A very small delay is not always enough so the theory here is that it needs to be long
      // enough for WebRTC to encode a frame: 120ms should be long enough even if we're only
      // doing 10fps.
      if (muted) {
        _this12.stopVideoTrackTimer = setTimeout(() => {
          for (var t of _this12.localUsermediaStream.getVideoTracks()) {
            t.stop();
            _this12.localUsermediaStream.removeTrack(t);
          }
        }, 120);
      }
      return _this12.isLocalVideoMuted();
    })();
  }

  /**
   * Check if local video is muted.
   *
   * If there are multiple video tracks, <i>all</i> of the tracks need to be muted
   * for this to return true. This means if there are no video tracks, this will
   * return true.
   * @returns True if the local preview video is muted, else false
   * (including if the call is not set up yet).
   */
  isLocalVideoMuted() {
    var _this$localUsermediaF2, _this$localUsermediaF3;
    return (_this$localUsermediaF2 = (_this$localUsermediaF3 = this.localUsermediaFeed) === null || _this$localUsermediaF3 === void 0 ? void 0 : _this$localUsermediaF3.isVideoMuted()) !== null && _this$localUsermediaF2 !== void 0 ? _this$localUsermediaF2 : false;
  }

  /**
   * Set whether the microphone should be muted or not.
   * @param muted - True to mute the mic.
   * @returns the new mute state
   */
  setMicrophoneMuted(muted) {
    var _this13 = this;
    return _asyncToGenerator(function* () {
      var _this13$localUsermedi;
      logger.log("Call ".concat(_this13.callId, " setMicrophoneMuted() running ").concat(muted));
      if (!(yield _this13.client.getMediaHandler().hasAudioDevice())) {
        return _this13.isMicrophoneMuted();
      }
      if (!muted && (!_this13.hasUserMediaAudioSender || !_this13.hasLocalUserMediaAudioTrack)) {
        yield _this13.upgradeCall(true, false);
        return _this13.isMicrophoneMuted();
      }
      (_this13$localUsermedi = _this13.localUsermediaFeed) === null || _this13$localUsermedi === void 0 || _this13$localUsermedi.setAudioVideoMuted(muted, null);
      _this13.updateMuteStatus();
      yield _this13.sendMetadataUpdate();
      return _this13.isMicrophoneMuted();
    })();
  }

  /**
   * Check if the microphone is muted.
   *
   * If there are multiple audio tracks, <i>all</i> of the tracks need to be muted
   * for this to return true. This means if there are no audio tracks, this will
   * return true.
   * @returns True if the mic is muted, else false (including if the call
   * is not set up yet).
   */
  isMicrophoneMuted() {
    var _this$localUsermediaF4, _this$localUsermediaF5;
    return (_this$localUsermediaF4 = (_this$localUsermediaF5 = this.localUsermediaFeed) === null || _this$localUsermediaF5 === void 0 ? void 0 : _this$localUsermediaF5.isAudioMuted()) !== null && _this$localUsermediaF4 !== void 0 ? _this$localUsermediaF4 : false;
  }

  /**
   * @returns true if we have put the party on the other side of the call on hold
   * (that is, we are signalling to them that we are not listening)
   */
  isRemoteOnHold() {
    return this.remoteOnHold;
  }
  setRemoteOnHold(onHold) {
    if (this.isRemoteOnHold() === onHold) return;
    this.remoteOnHold = onHold;
    for (var transceiver of this.peerConn.getTransceivers()) {
      // We don't send hold music or anything so we're not actually
      // sending anything, but sendrecv is fairly standard for hold and
      // it makes it a lot easier to figure out who's put who on hold.
      transceiver.direction = onHold ? "sendonly" : "sendrecv";
    }
    this.updateMuteStatus();
    this.sendMetadataUpdate();
    this.emit(CallEvent.RemoteHoldUnhold, this.remoteOnHold, this);
  }

  /**
   * Indicates whether we are 'on hold' to the remote party (ie. if true,
   * they cannot hear us).
   * @returns true if the other party has put us on hold
   */
  isLocalOnHold() {
    if (this.state !== CallState.Connected) return false;
    var callOnHold = true;

    // We consider a call to be on hold only if *all* the tracks are on hold
    // (is this the right thing to do?)
    for (var transceiver of this.peerConn.getTransceivers()) {
      var trackOnHold = ["inactive", "recvonly"].includes(transceiver.currentDirection);
      if (!trackOnHold) callOnHold = false;
    }
    return callOnHold;
  }

  /**
   * Sends a DTMF digit to the other party
   * @param digit - The digit (nb. string - '#' and '*' are dtmf too)
   */
  sendDtmfDigit(digit) {
    for (var sender of this.peerConn.getSenders()) {
      var _sender$track;
      if (((_sender$track = sender.track) === null || _sender$track === void 0 ? void 0 : _sender$track.kind) === "audio" && sender.dtmf) {
        sender.dtmf.insertDTMF(digit);
        return;
      }
    }
    throw new Error("Unable to find a track to send DTMF on");
  }
  updateMuteStatus() {
    var micShouldBeMuted = this.isMicrophoneMuted() || this.remoteOnHold;
    var vidShouldBeMuted = this.isLocalVideoMuted() || this.remoteOnHold;
    logger.log("Call ".concat(this.callId, " updateMuteStatus stream ").concat(this.localUsermediaStream.id, " micShouldBeMuted ").concat(micShouldBeMuted, " vidShouldBeMuted ").concat(vidShouldBeMuted));
    setTracksEnabled(this.localUsermediaStream.getAudioTracks(), !micShouldBeMuted);
    setTracksEnabled(this.localUsermediaStream.getVideoTracks(), !vidShouldBeMuted);
  }
  sendMetadataUpdate() {
    var _this14 = this;
    return _asyncToGenerator(function* () {
      yield _this14.sendVoipEvent(EventType.CallSDPStreamMetadataChangedPrefix, {
        [SDPStreamMetadataKey]: _this14.getLocalSDPStreamMetadata()
      });
    })();
  }
  gotCallFeedsForInvite(callFeeds) {
    var requestScreenshareFeed = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    if (this.successor) {
      this.successor.queueGotCallFeedsForAnswer(callFeeds);
      return;
    }
    if (this.callHasEnded()) {
      this.stopAllMedia();
      return;
    }
    for (var feed of callFeeds) {
      this.pushLocalFeed(feed);
    }
    if (requestScreenshareFeed) {
      this.peerConn.addTransceiver("video", {
        direction: "recvonly"
      });
    }
    this.state = CallState.CreateOffer;
    logger.debug("Call ".concat(this.callId, " gotUserMediaForInvite() run"));
    // Now we wait for the negotiationneeded event
  }
  sendAnswer() {
    var _this15 = this;
    return _asyncToGenerator(function* () {
      var answerContent = {
        answer: {
          sdp: _this15.peerConn.localDescription.sdp,
          // type is now deprecated as of Matrix VoIP v1, but
          // required to still be sent for backwards compat
          type: _this15.peerConn.localDescription.type
        },
        [SDPStreamMetadataKey]: _this15.getLocalSDPStreamMetadata(true)
      };
      answerContent.capabilities = {
        "m.call.transferee": _this15.client.supportsCallTransfer,
        "m.call.dtmf": false
      };

      // We have just taken the local description from the peerConn which will
      // contain all the local candidates added so far, so we can discard any candidates
      // we had queued up because they'll be in the answer.
      var discardCount = _this15.discardDuplicateCandidates();
      logger.info("Call ".concat(_this15.callId, " sendAnswer() discarding ").concat(discardCount, " candidates that will be sent in answer"));
      try {
        yield _this15.sendVoipEvent(EventType.CallAnswer, answerContent);
        // If this isn't the first time we've tried to send the answer,
        // we may have candidates queued up, so send them now.
        _this15.inviteOrAnswerSent = true;
      } catch (error) {
        // We've failed to answer: back to the ringing state
        _this15.state = CallState.Ringing;
        if (error instanceof MatrixError && error.event) _this15.client.cancelPendingEvent(error.event);
        var code = CallErrorCode.SendAnswer;
        var message = "Failed to send answer";
        if (error.name == "UnknownDeviceError") {
          code = CallErrorCode.UnknownDevices;
          message = "Unknown devices present in the room";
        }
        _this15.emit(CallEvent.Error, new CallError(code, message, error), _this15);
        throw error;
      }

      // error handler re-throws so this won't happen on error, but
      // we don't want the same error handling on the candidate queue
      _this15.sendCandidateQueue();
    })();
  }
  queueGotCallFeedsForAnswer(callFeeds) {
    // Ensure only one negotiate/answer event is being processed at a time.
    if (this.responsePromiseChain) {
      this.responsePromiseChain = this.responsePromiseChain.then(() => this.gotCallFeedsForAnswer(callFeeds));
    } else {
      this.responsePromiseChain = this.gotCallFeedsForAnswer(callFeeds);
    }
  }

  // Enables DTX (discontinuous transmission) on the given session to reduce
  // bandwidth when transmitting silence
  mungeSdp(description, mods) {
    // The only way to enable DTX at this time is through SDP munging
    var sdp = parseSdp(description.sdp);
    sdp.media.forEach(media => {
      var payloadTypeToCodecMap = new Map();
      var codecToPayloadTypeMap = new Map();
      for (var rtp of media.rtp) {
        payloadTypeToCodecMap.set(rtp.payload, rtp.codec);
        codecToPayloadTypeMap.set(rtp.codec, rtp.payload);
      }
      for (var mod of mods) {
        if (mod.mediaType !== media.type) continue;
        if (!codecToPayloadTypeMap.has(mod.codec)) {
          logger.info("Call ".concat(this.callId, " mungeSdp() ignoring SDP modifications for ").concat(mod.codec, " as it's not present."));
          continue;
        }
        var extraConfig = [];
        if (mod.enableDtx !== undefined) {
          extraConfig.push("usedtx=".concat(mod.enableDtx ? "1" : "0"));
        }
        if (mod.maxAverageBitrate !== undefined) {
          extraConfig.push("maxaveragebitrate=".concat(mod.maxAverageBitrate));
        }
        var found = false;
        for (var fmtp of media.fmtp) {
          if (payloadTypeToCodecMap.get(fmtp.payload) === mod.codec) {
            found = true;
            fmtp.config += ";" + extraConfig.join(";");
          }
        }
        if (!found) {
          media.fmtp.push({
            payload: codecToPayloadTypeMap.get(mod.codec),
            config: extraConfig.join(";")
          });
        }
      }
    });
    description.sdp = writeSdp(sdp);
  }
  createOffer() {
    var _this16 = this;
    return _asyncToGenerator(function* () {
      var offer = yield _this16.peerConn.createOffer();
      _this16.mungeSdp(offer, getCodecParamMods(_this16.isPtt));
      return offer;
    })();
  }
  createAnswer() {
    var _this17 = this;
    return _asyncToGenerator(function* () {
      var answer = yield _this17.peerConn.createAnswer();
      _this17.mungeSdp(answer, getCodecParamMods(_this17.isPtt));
      return answer;
    })();
  }
  gotCallFeedsForAnswer(callFeeds) {
    var _this18 = this;
    return _asyncToGenerator(function* () {
      if (_this18.callHasEnded()) return;
      _this18.waitForLocalAVStream = false;
      for (var feed of callFeeds) {
        _this18.pushLocalFeed(feed);
      }
      _this18.state = CallState.CreateAnswer;
      var answer;
      try {
        _this18.getRidOfRTXCodecs();
        answer = yield _this18.createAnswer();
      } catch (err) {
        logger.debug("Call ".concat(_this18.callId, " gotCallFeedsForAnswer() failed to create answer: "), err);
        _this18.terminate(CallParty.Local, CallErrorCode.CreateAnswer, true);
        return;
      }
      try {
        yield _this18.peerConn.setLocalDescription(answer);

        // make sure we're still going
        if (_this18.callHasEnded()) return;
        _this18.state = CallState.Connecting;

        // Allow a short time for initial candidates to be gathered
        yield new Promise(resolve => {
          setTimeout(resolve, 200);
        });

        // make sure the call hasn't ended before we continue
        if (_this18.callHasEnded()) return;
        _this18.sendAnswer();
      } catch (err) {
        logger.debug("Call ".concat(_this18.callId, " gotCallFeedsForAnswer() error setting local description!"), err);
        _this18.terminate(CallParty.Local, CallErrorCode.SetLocalDescription, true);
        return;
      }
    })();
  }
  onRemoteIceCandidatesReceived(ev) {
    var _this19 = this;
    return _asyncToGenerator(function* () {
      if (_this19.callHasEnded()) {
        //debuglog("Ignoring remote ICE candidate because call has ended");
        return;
      }
      var content = ev.getContent();
      var candidates = content.candidates;
      if (!candidates) {
        logger.info("Call ".concat(_this19.callId, " onRemoteIceCandidatesReceived() ignoring candidates event with no candidates!"));
        return;
      }
      var fromPartyId = content.version === 0 ? null : content.party_id || null;
      if (_this19.opponentPartyId === undefined) {
        // we haven't picked an opponent yet so save the candidates
        if (fromPartyId) {
          logger.info("Call ".concat(_this19.callId, " onRemoteIceCandidatesReceived() buffering ").concat(candidates.length, " candidates until we pick an opponent"));
          var bufferedCandidates = _this19.remoteCandidateBuffer.get(fromPartyId) || [];
          bufferedCandidates.push(...candidates);
          _this19.remoteCandidateBuffer.set(fromPartyId, bufferedCandidates);
        }
        return;
      }
      if (!_this19.partyIdMatches(content)) {
        logger.info("Call ".concat(_this19.callId, " onRemoteIceCandidatesReceived() ignoring candidates from party ID ").concat(content.party_id, ": we have chosen party ID ").concat(_this19.opponentPartyId));
        return;
      }
      yield _this19.addIceCandidates(candidates);
    })();
  }

  /**
   * Used by MatrixClient.
   */
  onAnswerReceived(event) {
    var _this20 = this;
    return _asyncToGenerator(function* () {
      var content = event.getContent();
      logger.debug("Call ".concat(_this20.callId, " onAnswerReceived() running (hangupParty=").concat(content.party_id, ")"));
      if (_this20.callHasEnded()) {
        logger.debug("Call ".concat(_this20.callId, " onAnswerReceived() ignoring answer because call has ended"));
        return;
      }
      if (_this20.opponentPartyId !== undefined) {
        logger.info("Call ".concat(_this20.callId, " onAnswerReceived() ignoring answer from party ID ").concat(content.party_id, ": we already have an answer/reject from ").concat(_this20.opponentPartyId));
        return;
      }
      _this20.chooseOpponent(event);
      yield _this20.addBufferedIceCandidates();
      _this20.state = CallState.Connecting;
      var sdpStreamMetadata = content[SDPStreamMetadataKey];
      if (sdpStreamMetadata) {
        _this20.updateRemoteSDPStreamMetadata(sdpStreamMetadata);
      } else {
        logger.warn("Call ".concat(_this20.callId, " onAnswerReceived() did not get any SDPStreamMetadata! Can not send/receive multiple streams"));
      }
      try {
        _this20.isSettingRemoteAnswerPending = true;
        yield _this20.peerConn.setRemoteDescription(content.answer);
        _this20.isSettingRemoteAnswerPending = false;
        logger.debug("Call ".concat(_this20.callId, " onAnswerReceived() set remote description: ").concat(content.answer.type));
      } catch (e) {
        _this20.isSettingRemoteAnswerPending = false;
        logger.debug("Call ".concat(_this20.callId, " onAnswerReceived() failed to set remote description"), e);
        _this20.terminate(CallParty.Local, CallErrorCode.SetRemoteDescription, false);
        return;
      }

      // If the answer we selected has a party_id, send a select_answer event
      // We do this after setting the remote description since otherwise we'd block
      // call setup on it
      if (_this20.opponentPartyId !== null) {
        try {
          yield _this20.sendVoipEvent(EventType.CallSelectAnswer, {
            selected_party_id: _this20.opponentPartyId
          });
        } catch (err) {
          // This isn't fatal, and will just mean that if another party has raced to answer
          // the call, they won't know they got rejected, so we carry on & don't retry.
          logger.warn("Call ".concat(_this20.callId, " onAnswerReceived() failed to send select_answer event"), err);
        }
      }
    })();
  }
  onSelectAnswerReceived(event) {
    var _this21 = this;
    return _asyncToGenerator(function* () {
      if (_this21.direction !== CallDirection.Inbound) {
        logger.warn("Call ".concat(_this21.callId, " onSelectAnswerReceived() got select_answer for an outbound call: ignoring"));
        return;
      }
      var selectedPartyId = event.getContent().selected_party_id;
      if (selectedPartyId === undefined || selectedPartyId === null) {
        logger.warn("Call ".concat(_this21.callId, " onSelectAnswerReceived() got nonsensical select_answer with null/undefined selected_party_id: ignoring"));
        return;
      }
      if (selectedPartyId !== _this21.ourPartyId) {
        logger.info("Call ".concat(_this21.callId, " onSelectAnswerReceived() got select_answer for party ID ").concat(selectedPartyId, ": we are party ID ").concat(_this21.ourPartyId, "."));
        // The other party has picked somebody else's answer
        yield _this21.terminate(CallParty.Remote, CallErrorCode.AnsweredElsewhere, true);
      }
    })();
  }
  onNegotiateReceived(event) {
    var _this22 = this;
    return _asyncToGenerator(function* () {
      var content = event.getContent();
      var description = content.description;
      if (!description || !description.sdp || !description.type) {
        logger.info("Call ".concat(_this22.callId, " onNegotiateReceived() ignoring invalid m.call.negotiate event"));
        return;
      }
      // Politeness always follows the direction of the call: in a glare situation,
      // we pick either the inbound or outbound call, so one side will always be
      // inbound and one outbound
      var polite = _this22.direction === CallDirection.Inbound;

      // Here we follow the perfect negotiation logic from
      // https://w3c.github.io/webrtc-pc/#perfect-negotiation-example
      var readyForOffer = !_this22.makingOffer && (_this22.peerConn.signalingState === "stable" || _this22.isSettingRemoteAnswerPending);
      var offerCollision = description.type === "offer" && !readyForOffer;
      _this22.ignoreOffer = !polite && offerCollision;
      if (_this22.ignoreOffer) {
        logger.info("Call ".concat(_this22.callId, " onNegotiateReceived() ignoring colliding negotiate event because we're impolite"));
        return;
      }
      var prevLocalOnHold = _this22.isLocalOnHold();
      var sdpStreamMetadata = content[SDPStreamMetadataKey];
      if (sdpStreamMetadata) {
        _this22.updateRemoteSDPStreamMetadata(sdpStreamMetadata);
      } else {
        logger.warn("Call ".concat(_this22.callId, " onNegotiateReceived() received negotiation event without SDPStreamMetadata!"));
      }
      try {
        _this22.isSettingRemoteAnswerPending = description.type == "answer";
        yield _this22.peerConn.setRemoteDescription(description); // SRD rolls back as needed
        _this22.isSettingRemoteAnswerPending = false;
        logger.debug("Call ".concat(_this22.callId, " onNegotiateReceived() set remote description: ").concat(description.type));
        if (description.type === "offer") {
          var _localDescription;
          var answer;
          try {
            _this22.getRidOfRTXCodecs();
            answer = yield _this22.createAnswer();
          } catch (err) {
            logger.debug("Call ".concat(_this22.callId, " onNegotiateReceived() failed to create answer: "), err);
            _this22.terminate(CallParty.Local, CallErrorCode.CreateAnswer, true);
            return;
          }
          yield _this22.peerConn.setLocalDescription(answer);
          logger.debug("Call ".concat(_this22.callId, " onNegotiateReceived() create an answer"));
          _this22.sendVoipEvent(EventType.CallNegotiate, {
            lifetime: CALL_TIMEOUT_MS,
            description: (_localDescription = _this22.peerConn.localDescription) === null || _localDescription === void 0 ? void 0 : _localDescription.toJSON(),
            [SDPStreamMetadataKey]: _this22.getLocalSDPStreamMetadata(true)
          });
        }
      } catch (err) {
        _this22.isSettingRemoteAnswerPending = false;
        logger.warn("Call ".concat(_this22.callId, " onNegotiateReceived() failed to complete negotiation"), err);
      }
      var newLocalOnHold = _this22.isLocalOnHold();
      if (prevLocalOnHold !== newLocalOnHold) {
        _this22.emit(CallEvent.LocalHoldUnhold, newLocalOnHold, _this22);
        // also this one for backwards compat
        _this22.emit(CallEvent.HoldUnhold, newLocalOnHold);
      }
    })();
  }
  updateRemoteSDPStreamMetadata(metadata) {
    this.remoteSDPStreamMetadata = recursivelyAssign(this.remoteSDPStreamMetadata || {}, metadata, true);
    for (var feed of this.getRemoteFeeds()) {
      var _streamId;
      var streamId = feed.stream.id;
      var _metadata = this.remoteSDPStreamMetadata[streamId];
      feed.setAudioVideoMuted(_metadata === null || _metadata === void 0 ? void 0 : _metadata.audio_muted, _metadata === null || _metadata === void 0 ? void 0 : _metadata.video_muted);
      feed.purpose = (_streamId = this.remoteSDPStreamMetadata[streamId]) === null || _streamId === void 0 ? void 0 : _streamId.purpose;
    }
  }
  onSDPStreamMetadataChangedReceived(event) {
    var content = event.getContent();
    var metadata = content[SDPStreamMetadataKey];
    this.updateRemoteSDPStreamMetadata(metadata);
  }
  onAssertedIdentityReceived(event) {
    var _this23 = this;
    return _asyncToGenerator(function* () {
      var content = event.getContent();
      if (!content.asserted_identity) return;
      _this23.remoteAssertedIdentity = {
        id: content.asserted_identity.id,
        displayName: content.asserted_identity.display_name
      };
      _this23.emit(CallEvent.AssertedIdentityChanged, _this23);
    })();
  }
  callHasEnded() {
    // This exists as workaround to typescript trying to be clever and erroring
    // when putting if (this.state === CallState.Ended) return; twice in the same
    // function, even though that function is async.
    return this.state === CallState.Ended;
  }
  queueGotLocalOffer() {
    // Ensure only one negotiate/answer event is being processed at a time.
    if (this.responsePromiseChain) {
      this.responsePromiseChain = this.responsePromiseChain.then(() => this.wrappedGotLocalOffer());
    } else {
      this.responsePromiseChain = this.wrappedGotLocalOffer();
    }
  }
  wrappedGotLocalOffer() {
    var _this24 = this;
    return _asyncToGenerator(function* () {
      _this24.makingOffer = true;
      try {
        // XXX: in what situations do we believe gotLocalOffer actually throws? It appears
        // to handle most of its exceptions itself and terminate the call. I'm not entirely
        // sure it would ever throw, so I can't add a test for these lines.
        // Also the tense is different between "gotLocalOffer" and "getLocalOfferFailed" so
        // it's not entirely clear whether getLocalOfferFailed is just misnamed or whether
        // they've been cross-polinated somehow at some point.
        yield _this24.gotLocalOffer();
      } catch (e) {
        _this24.getLocalOfferFailed(e);
        return;
      } finally {
        _this24.makingOffer = false;
      }
    })();
  }
  gotLocalOffer() {
    var _this25 = this;
    return _asyncToGenerator(function* () {
      logger.debug("Call ".concat(_this25.callId, " gotLocalOffer() running"));
      if (_this25.callHasEnded()) {
        logger.debug("Call ".concat(_this25.callId, " gotLocalOffer() ignoring newly created offer because the call has ended\""));
        return;
      }
      var offer;
      try {
        _this25.getRidOfRTXCodecs();
        offer = yield _this25.createOffer();
      } catch (err) {
        logger.debug("Call ".concat(_this25.callId, " gotLocalOffer() failed to create offer: "), err);
        _this25.terminate(CallParty.Local, CallErrorCode.CreateOffer, true);
        return;
      }
      try {
        yield _this25.peerConn.setLocalDescription(offer);
      } catch (err) {
        logger.debug("Call ".concat(_this25.callId, " gotLocalOffer() error setting local description!"), err);
        _this25.terminate(CallParty.Local, CallErrorCode.SetLocalDescription, true);
        return;
      }
      if (_this25.peerConn.iceGatheringState === "gathering") {
        // Allow a short time for initial candidates to be gathered
        yield new Promise(resolve => {
          setTimeout(resolve, 200);
        });
      }
      if (_this25.callHasEnded()) return;
      var eventType = _this25.state === CallState.CreateOffer ? EventType.CallInvite : EventType.CallNegotiate;
      var content = {
        lifetime: CALL_TIMEOUT_MS
      };
      if (eventType === EventType.CallInvite && _this25.invitee) {
        content.invitee = _this25.invitee;
      }

      // clunky because TypeScript can't follow the types through if we use an expression as the key
      if (_this25.state === CallState.CreateOffer) {
        var _localDescription2;
        content.offer = (_localDescription2 = _this25.peerConn.localDescription) === null || _localDescription2 === void 0 ? void 0 : _localDescription2.toJSON();
      } else {
        var _localDescription3;
        content.description = (_localDescription3 = _this25.peerConn.localDescription) === null || _localDescription3 === void 0 ? void 0 : _localDescription3.toJSON();
      }
      content.capabilities = {
        "m.call.transferee": _this25.client.supportsCallTransfer,
        "m.call.dtmf": false
      };
      content[SDPStreamMetadataKey] = _this25.getLocalSDPStreamMetadata(true);

      // Get rid of any candidates waiting to be sent: they'll be included in the local
      // description we just got and will send in the offer.
      var discardCount = _this25.discardDuplicateCandidates();
      logger.info("Call ".concat(_this25.callId, " gotLocalOffer() discarding ").concat(discardCount, " candidates that will be sent in offer"));
      try {
        yield _this25.sendVoipEvent(eventType, content);
      } catch (error) {
        logger.error("Call ".concat(_this25.callId, " gotLocalOffer() failed to send invite"), error);
        if (error instanceof MatrixError && error.event) _this25.client.cancelPendingEvent(error.event);
        var code = CallErrorCode.SignallingFailed;
        var message = "Signalling failed";
        if (_this25.state === CallState.CreateOffer) {
          code = CallErrorCode.SendInvite;
          message = "Failed to send invite";
        }
        if (error.name == "UnknownDeviceError") {
          code = CallErrorCode.UnknownDevices;
          message = "Unknown devices present in the room";
        }
        _this25.emit(CallEvent.Error, new CallError(code, message, error), _this25);
        _this25.terminate(CallParty.Local, code, false);

        // no need to carry on & send the candidate queue, but we also
        // don't want to rethrow the error
        return;
      }
      _this25.sendCandidateQueue();
      if (_this25.state === CallState.CreateOffer) {
        _this25.inviteOrAnswerSent = true;
        _this25.state = CallState.InviteSent;
        _this25.inviteTimeout = setTimeout(() => {
          _this25.inviteTimeout = undefined;
          if (_this25.state === CallState.InviteSent) {
            _this25.hangup(CallErrorCode.InviteTimeout, false);
          }
        }, CALL_TIMEOUT_MS);
      }
    })();
  }
  /**
   * This method removes all video/rtx codecs from screensharing video
   * transceivers. This is necessary since they can cause problems. Without
   * this the following steps should produce an error:
   *   Chromium calls Firefox
   *   Firefox answers
   *   Firefox starts screen-sharing
   *   Chromium starts screen-sharing
   *   Call crashes for Chromium with:
   *       [96685:23:0518/162603.933321:ERROR:webrtc_video_engine.cc(3296)] RTX codec (PT=97) mapped to PT=96 which is not in the codec list.
   *       [96685:23:0518/162603.933377:ERROR:webrtc_video_engine.cc(1171)] GetChangedRecvParameters called without any video codecs.
   *       [96685:23:0518/162603.933430:ERROR:sdp_offer_answer.cc(4302)] Failed to set local video description recv parameters for m-section with mid='2'. (INVALID_PARAMETER)
   */
  getRidOfRTXCodecs() {
    // RTCRtpReceiver.getCapabilities and RTCRtpSender.getCapabilities don't seem to be supported on FF before v113
    if (!RTCRtpReceiver.getCapabilities || !RTCRtpSender.getCapabilities) return;
    var screenshareVideoTransceiver = this.transceivers.get(getTransceiverKey(SDPStreamMetadataPurpose.Screenshare, "video"));

    // setCodecPreferences isn't supported on FF (as of v113)
    if (!screenshareVideoTransceiver || !screenshareVideoTransceiver.setCodecPreferences) return;
    var recvCodecs = RTCRtpReceiver.getCapabilities("video").codecs;
    var sendCodecs = RTCRtpSender.getCapabilities("video").codecs;
    var codecs = [];
    for (var codec of [...recvCodecs, ...sendCodecs]) {
      if (codec.mimeType !== "video/rtx") {
        codecs.push(codec);
        try {
          screenshareVideoTransceiver.setCodecPreferences(codecs);
        } catch (e) {
          // Specifically, Chrome around version 125 and Electron 30 (which is Chromium 124) return an H.264 codec in
          // the sender's capabilities but throw when you try to set it. Hence... this mess.
          // Specifically, that codec is:
          // {
          //   clockRate: 90000,
          //   mimeType: "video/H264",
          //   sdpFmtpLine: "level-asymmetry-allowed=1;packetization-mode=1;profile-level-id=640034",
          // }
          logger.info("Working around buggy WebRTC impl: claimed to support codec but threw when setting codec preferences", codec, e);
          codecs.pop();
        }
      }
    }
  }
  /**
   * @internal
   */
  sendVoipEvent(eventType, content) {
    var _this26 = this;
    return _asyncToGenerator(function* () {
      var realContent = _objectSpread(_objectSpread({}, content), {}, {
        version: VOIP_PROTO_VERSION,
        call_id: _this26.callId,
        party_id: _this26.ourPartyId,
        conf_id: _this26.groupCallId
      });
      if (_this26.opponentDeviceId) {
        var _this26$getOpponentMe;
        var toDeviceSeq = _this26.toDeviceSeq++;
        var _content = _objectSpread(_objectSpread({}, realContent), {}, {
          device_id: _this26.client.deviceId,
          sender_session_id: _this26.client.getSessionId(),
          dest_session_id: _this26.opponentSessionId,
          seq: toDeviceSeq,
          [ToDeviceMessageId]: uuidv4()
        });
        _this26.emit(CallEvent.SendVoipEvent, {
          type: "toDevice",
          eventType,
          userId: _this26.invitee || ((_this26$getOpponentMe = _this26.getOpponentMember()) === null || _this26$getOpponentMe === void 0 ? void 0 : _this26$getOpponentMe.userId),
          opponentDeviceId: _this26.opponentDeviceId,
          content: _content
        }, _this26);
        var userId = _this26.invitee || _this26.getOpponentMember().userId;
        if (_this26.client.getUseE2eForGroupCall()) {
          if (!_this26.hasOpponentDeviceInfo) {
            logger.warn("Call ".concat(_this26.callId, " sendVoipEvent() failed: we do not have opponentDeviceInfo"));
            return;
          }

          // TODO: Here we were sending the event to the opponent's device as a to-device message with MatrixClient.encryptAndSendToDevice.
          // However due to the switch to Rust cryptography we need to migrate to the new encryptToDeviceMessages API.
          throw new Error("Unimplemented");
        } else {
          yield _this26.client.sendToDevice(eventType, new Map([[userId, new Map([[_this26.opponentDeviceId, _content]])]]));
        }
      } else {
        var _this26$getOpponentMe2;
        _this26.emit(CallEvent.SendVoipEvent, {
          type: "sendEvent",
          eventType,
          roomId: _this26.roomId,
          content: realContent,
          userId: _this26.invitee || ((_this26$getOpponentMe2 = _this26.getOpponentMember()) === null || _this26$getOpponentMe2 === void 0 ? void 0 : _this26$getOpponentMe2.userId)
        }, _this26);
        yield _this26.client.sendEvent(_this26.roomId, eventType, realContent);
      }
    })();
  }

  /**
   * Queue a candidate to be sent
   * @param content - The candidate to queue up, or null if candidates have finished being generated
   *                and end-of-candidates should be signalled
   */
  queueCandidate(content) {
    // We partially de-trickle candidates by waiting for `delay` before sending them
    // amalgamated, in order to avoid sending too many m.call.candidates events and hitting
    // rate limits in Matrix.
    // In practice, it'd be better to remove rate limits for m.call.*

    // N.B. this deliberately lets you queue and send blank candidates, which MSC2746
    // currently proposes as the way to indicate that candidate gathering is complete.
    // This will hopefully be changed to an explicit rather than implicit notification
    // shortly.
    if (content) {
      this.candidateSendQueue.push(content);
    } else {
      this.candidatesEnded = true;
    }

    // Don't send the ICE candidates yet if the call is in the ringing state: this
    // means we tried to pick (ie. started generating candidates) and then failed to
    // send the answer and went back to the ringing state. Queue up the candidates
    // to send if we successfully send the answer.
    // Equally don't send if we haven't yet sent the answer because we can send the
    // first batch of candidates along with the answer
    if (this.state === CallState.Ringing || !this.inviteOrAnswerSent) return;

    // MSC2746 recommends these values (can be quite long when calling because the
    // callee will need a while to answer the call)
    var delay = this.direction === CallDirection.Inbound ? 500 : 2000;
    if (this.candidateSendTries === 0) {
      setTimeout(() => {
        this.sendCandidateQueue();
      }, delay);
    }
  }

  // Discard all non-end-of-candidates messages
  // Return the number of candidate messages that were discarded.
  // Call this method before sending an invite or answer message
  discardDuplicateCandidates() {
    var discardCount = 0;
    var newQueue = [];
    for (var i = 0; i < this.candidateSendQueue.length; i++) {
      var candidate = this.candidateSendQueue[i];
      if (candidate.candidate === "") {
        newQueue.push(candidate);
      } else {
        discardCount++;
      }
    }
    this.candidateSendQueue = newQueue;
    return discardCount;
  }

  /*
   * Transfers this call to another user
   */
  transfer(targetUserId) {
    var _this27 = this;
    return _asyncToGenerator(function* () {
      // Fetch the target user's global profile info: their room avatar / displayname
      // could be different in whatever room we share with them.
      var profileInfo = yield _this27.client.getProfileInfo(targetUserId);
      var replacementId = genCallID();
      var body = {
        replacement_id: genCallID(),
        target_user: {
          id: targetUserId,
          display_name: profileInfo.displayname,
          avatar_url: profileInfo.avatar_url
        },
        create_call: replacementId
      };
      yield _this27.sendVoipEvent(EventType.CallReplaces, body);
      yield _this27.terminate(CallParty.Local, CallErrorCode.Transferred, true);
    })();
  }

  /*
   * Transfers this call to the target call, effectively 'joining' the
   * two calls (so the remote parties on each call are connected together).
   */
  transferToCall(transferTargetCall) {
    var _this28 = this;
    return _asyncToGenerator(function* () {
      var _transferTargetCall$g, _this28$getOpponentMe;
      var targetUserId = (_transferTargetCall$g = transferTargetCall.getOpponentMember()) === null || _transferTargetCall$g === void 0 ? void 0 : _transferTargetCall$g.userId;
      var targetProfileInfo = targetUserId ? yield _this28.client.getProfileInfo(targetUserId) : undefined;
      var opponentUserId = (_this28$getOpponentMe = _this28.getOpponentMember()) === null || _this28$getOpponentMe === void 0 ? void 0 : _this28$getOpponentMe.userId;
      var transfereeProfileInfo = opponentUserId ? yield _this28.client.getProfileInfo(opponentUserId) : undefined;
      var newCallId = genCallID();
      var bodyToTransferTarget = {
        // the replacements on each side have their own ID, and it's distinct from the
        // ID of the new call (but we can use the same function to generate it)
        replacement_id: genCallID(),
        target_user: {
          id: opponentUserId,
          display_name: transfereeProfileInfo === null || transfereeProfileInfo === void 0 ? void 0 : transfereeProfileInfo.displayname,
          avatar_url: transfereeProfileInfo === null || transfereeProfileInfo === void 0 ? void 0 : transfereeProfileInfo.avatar_url
        },
        await_call: newCallId
      };
      yield transferTargetCall.sendVoipEvent(EventType.CallReplaces, bodyToTransferTarget);
      var bodyToTransferee = {
        replacement_id: genCallID(),
        target_user: {
          id: targetUserId,
          display_name: targetProfileInfo === null || targetProfileInfo === void 0 ? void 0 : targetProfileInfo.displayname,
          avatar_url: targetProfileInfo === null || targetProfileInfo === void 0 ? void 0 : targetProfileInfo.avatar_url
        },
        create_call: newCallId
      };
      yield _this28.sendVoipEvent(EventType.CallReplaces, bodyToTransferee);
      yield _this28.terminate(CallParty.Local, CallErrorCode.Transferred, true);
      yield transferTargetCall.terminate(CallParty.Local, CallErrorCode.Transferred, true);
    })();
  }
  terminate(hangupParty, hangupReason, shouldEmit) {
    var _this29 = this;
    return _asyncToGenerator(function* () {
      var _this29$stats;
      if (_this29.callHasEnded()) return;
      _this29.hangupParty = hangupParty;
      _this29.hangupReason = hangupReason;
      _this29.state = CallState.Ended;
      if (_this29.inviteTimeout) {
        clearTimeout(_this29.inviteTimeout);
        _this29.inviteTimeout = undefined;
      }
      if (_this29.iceDisconnectedTimeout !== undefined) {
        clearTimeout(_this29.iceDisconnectedTimeout);
        _this29.iceDisconnectedTimeout = undefined;
      }
      if (_this29.callLengthInterval) {
        clearInterval(_this29.callLengthInterval);
        _this29.callLengthInterval = undefined;
      }
      if (_this29.stopVideoTrackTimer !== undefined) {
        clearTimeout(_this29.stopVideoTrackTimer);
        _this29.stopVideoTrackTimer = undefined;
      }
      for (var [stream, listener] of _this29.removeTrackListeners) {
        stream.removeEventListener("removetrack", listener);
      }
      _this29.removeTrackListeners.clear();
      _this29.callStatsAtEnd = yield _this29.collectCallStats();

      // Order is important here: first we stopAllMedia() and only then we can deleteAllFeeds()
      _this29.stopAllMedia();
      _this29.deleteAllFeeds();
      if (_this29.peerConn && _this29.peerConn.signalingState !== "closed") {
        _this29.peerConn.close();
      }
      (_this29$stats = _this29.stats) === null || _this29$stats === void 0 || _this29$stats.removeStatsReportGatherer(_this29.callId);
      if (shouldEmit) {
        _this29.emit(CallEvent.Hangup, _this29);
      }
      _this29.client.callEventHandler.calls.delete(_this29.callId);
    })();
  }
  stopAllMedia() {
    logger.debug("Call ".concat(this.callId, " stopAllMedia() running"));
    for (var feed of this.feeds) {
      // Slightly awkward as local feed need to go via the correct method on
      // the MediaHandler so they get removed from MediaHandler (remote tracks
      // don't)
      // NB. We clone local streams when passing them to individual calls in a group
      // call, so we can (and should) stop the clones once we no longer need them:
      // the other clones will continue fine.
      if (feed.isLocal() && feed.purpose === SDPStreamMetadataPurpose.Usermedia) {
        this.client.getMediaHandler().stopUserMediaStream(feed.stream);
      } else if (feed.isLocal() && feed.purpose === SDPStreamMetadataPurpose.Screenshare) {
        this.client.getMediaHandler().stopScreensharingStream(feed.stream);
      } else if (!feed.isLocal()) {
        logger.debug("Call ".concat(this.callId, " stopAllMedia() stopping stream (streamId=").concat(feed.stream.id, ")"));
        for (var track of feed.stream.getTracks()) {
          track.stop();
        }
      }
    }
  }
  checkForErrorListener() {
    if (this.listeners(EventEmitterEvents.Error).length === 0) {
      throw new Error("You MUST attach an error listener using call.on('error', function() {})");
    }
  }
  sendCandidateQueue() {
    var _this30 = this;
    return _asyncToGenerator(function* () {
      if (_this30.candidateSendQueue.length === 0 || _this30.callHasEnded()) {
        return;
      }
      var candidates = _this30.candidateSendQueue;
      _this30.candidateSendQueue = [];
      ++_this30.candidateSendTries;
      var content = {
        candidates: candidates.map(candidate => candidate.toJSON())
      };
      if (_this30.candidatesEnded) {
        // If there are no more candidates, signal this by adding an empty string candidate
        content.candidates.push({
          candidate: ""
        });
      }
      logger.debug("Call ".concat(_this30.callId, " sendCandidateQueue() attempting to send ").concat(candidates.length, " candidates"));
      try {
        yield _this30.sendVoipEvent(EventType.CallCandidates, content);
        // reset our retry count if we have successfully sent our candidates
        // otherwise queueCandidate() will refuse to try to flush the queue
        _this30.candidateSendTries = 0;

        // Try to send candidates again just in case we received more candidates while sending.
        _this30.sendCandidateQueue();
      } catch (error) {
        // don't retry this event: we'll send another one later as we might
        // have more candidates by then.
        if (error instanceof MatrixError && error.event) _this30.client.cancelPendingEvent(error.event);

        // put all the candidates we failed to send back in the queue
        _this30.candidateSendQueue.push(...candidates);
        if (_this30.candidateSendTries > 5) {
          logger.debug("Call ".concat(_this30.callId, " sendCandidateQueue() failed to send candidates on attempt ").concat(_this30.candidateSendTries, ". Giving up on this call."), error);
          var code = CallErrorCode.SignallingFailed;
          var message = "Signalling failed";
          _this30.emit(CallEvent.Error, new CallError(code, message, error), _this30);
          _this30.hangup(code, false);
          return;
        }
        var delayMs = 500 * Math.pow(2, _this30.candidateSendTries);
        ++_this30.candidateSendTries;
        logger.debug("Call ".concat(_this30.callId, " sendCandidateQueue() failed to send candidates. Retrying in ").concat(delayMs, "ms"), error);
        setTimeout(() => {
          _this30.sendCandidateQueue();
        }, delayMs);
      }
    })();
  }

  /**
   * Place a call to this room.
   * @throws if you have not specified a listener for 'error' events.
   * @throws if have passed audio=false.
   */
  placeCall(audio, video) {
    var _this31 = this;
    return _asyncToGenerator(function* () {
      if (!audio) {
        throw new Error("You CANNOT start a call without audio");
      }
      _this31.state = CallState.WaitLocalMedia;
      var callFeed;
      try {
        var _this31$client$getDev;
        var stream = yield _this31.client.getMediaHandler().getUserMediaStream(audio, video);

        // make sure all the tracks are enabled (same as pushNewLocalFeed -
        // we probably ought to just have one code path for adding streams)
        setTracksEnabled(stream.getAudioTracks(), true);
        setTracksEnabled(stream.getVideoTracks(), true);
        callFeed = new CallFeed({
          client: _this31.client,
          roomId: _this31.roomId,
          userId: _this31.client.getUserId(),
          deviceId: (_this31$client$getDev = _this31.client.getDeviceId()) !== null && _this31$client$getDev !== void 0 ? _this31$client$getDev : undefined,
          stream,
          purpose: SDPStreamMetadataPurpose.Usermedia,
          audioMuted: false,
          videoMuted: false
        });
      } catch (e) {
        _this31.getUserMediaFailed(e);
        return;
      }
      try {
        yield _this31.placeCallWithCallFeeds([callFeed]);
      } catch (e) {
        _this31.placeCallFailed(e);
        return;
      }
    })();
  }

  /**
   * Place a call to this room with call feed.
   * @param callFeeds - to use
   * @throws if you have not specified a listener for 'error' events.
   * @throws if have passed audio=false.
   */
  placeCallWithCallFeeds(callFeeds) {
    var _arguments2 = arguments,
      _this32 = this;
    return _asyncToGenerator(function* () {
      var requestScreenshareFeed = _arguments2.length > 1 && _arguments2[1] !== undefined ? _arguments2[1] : false;
      _this32.checkForErrorListener();
      _this32.direction = CallDirection.Outbound;
      yield _this32.initOpponentCrypto();

      // XXX Find a better way to do this
      _this32.client.callEventHandler.calls.set(_this32.callId, _this32);

      // make sure we have valid turn creds. Unless something's gone wrong, it should
      // poll and keep the credentials valid so this should be instant.
      var haveTurnCreds = yield _this32.client.checkTurnServers();
      if (!haveTurnCreds) {
        logger.warn("Call ".concat(_this32.callId, " placeCallWithCallFeeds() failed to get TURN credentials! Proceeding with call anyway..."));
      }

      // create the peer connection now so it can be gathering candidates while we get user
      // media (assuming a candidate pool size is configured)
      _this32.peerConn = _this32.createPeerConnection();
      _this32.emit(CallEvent.PeerConnectionCreated, _this32.peerConn, _this32);
      _this32.gotCallFeedsForInvite(callFeeds, requestScreenshareFeed);
    })();
  }
  createPeerConnection() {
    var _this$stats;
    var pc = new window.RTCPeerConnection({
      iceTransportPolicy: this.forceTURN ? "relay" : undefined,
      iceServers: this.turnServers.length ? this.turnServers : undefined,
      iceCandidatePoolSize: this.client.iceCandidatePoolSize,
      bundlePolicy: "max-bundle"
    });

    // 'connectionstatechange' would be better, but firefox doesn't implement that.
    pc.addEventListener("iceconnectionstatechange", this.onIceConnectionStateChanged);
    pc.addEventListener("signalingstatechange", this.onSignallingStateChanged);
    pc.addEventListener("icecandidate", this.gotLocalIceCandidate);
    pc.addEventListener("icegatheringstatechange", this.onIceGatheringStateChange);
    pc.addEventListener("track", this.onTrack);
    pc.addEventListener("negotiationneeded", this.onNegotiationNeeded);
    pc.addEventListener("datachannel", this.onDataChannel);
    var opponentMember = this.getOpponentMember();
    var opponentMemberId = opponentMember ? opponentMember.userId : "unknown";
    (_this$stats = this.stats) === null || _this$stats === void 0 || _this$stats.addStatsReportGatherer(this.callId, opponentMemberId, pc);
    return pc;
  }
  partyIdMatches(msg) {
    // They must either match or both be absent (in which case opponentPartyId will be null)
    // Also we ignore party IDs on the invite/offer if the version is 0, so we must do the same
    // here and use null if the version is 0 (woe betide any opponent sending messages in the
    // same call with different versions)
    var msgPartyId = msg.version === 0 ? null : msg.party_id || null;
    return msgPartyId === this.opponentPartyId;
  }

  // Commits to an opponent for the call
  // ev: An invite or answer event
  chooseOpponent(ev) {
    var _getMember;
    // I choo-choo-choose you
    var msg = ev.getContent();
    logger.debug("Call ".concat(this.callId, " chooseOpponent() running (partyId=").concat(msg.party_id, ")"));
    this.opponentVersion = msg.version;
    if (this.opponentVersion === 0) {
      // set to null to indicate that we've chosen an opponent, but because
      // they're v0 they have no party ID (even if they sent one, we're ignoring it)
      this.opponentPartyId = null;
    } else {
      // set to their party ID, or if they're naughty and didn't send one despite
      // not being v0, set it to null to indicate we picked an opponent with no
      // party ID
      this.opponentPartyId = msg.party_id || null;
    }
    this.opponentCaps = msg.capabilities || {};
    this.opponentMember = (_getMember = this.client.getRoom(this.roomId).getMember(ev.getSender())) !== null && _getMember !== void 0 ? _getMember : undefined;
    if (this.opponentMember) {
      var _this$stats2;
      (_this$stats2 = this.stats) === null || _this$stats2 === void 0 || _this$stats2.updateOpponentMember(this.callId, this.opponentMember.userId);
    }
  }
  addBufferedIceCandidates() {
    var _this33 = this;
    return _asyncToGenerator(function* () {
      var bufferedCandidates = _this33.remoteCandidateBuffer.get(_this33.opponentPartyId);
      if (bufferedCandidates) {
        logger.info("Call ".concat(_this33.callId, " addBufferedIceCandidates() adding ").concat(bufferedCandidates.length, " buffered candidates for opponent ").concat(_this33.opponentPartyId));
        yield _this33.addIceCandidates(bufferedCandidates);
      }
      _this33.remoteCandidateBuffer.clear();
    })();
  }
  addIceCandidates(candidates) {
    var _this34 = this;
    return _asyncToGenerator(function* () {
      for (var candidate of candidates) {
        if ((candidate.sdpMid === null || candidate.sdpMid === undefined) && (candidate.sdpMLineIndex === null || candidate.sdpMLineIndex === undefined)) {
          logger.debug("Call ".concat(_this34.callId, " addIceCandidates() got remote ICE end-of-candidates"));
        } else {
          logger.debug("Call ".concat(_this34.callId, " addIceCandidates() got remote ICE candidate (sdpMid=").concat(candidate.sdpMid, ", candidate=").concat(candidate.candidate, ")"));
        }
        try {
          yield _this34.peerConn.addIceCandidate(candidate);
        } catch (err) {
          if (!_this34.ignoreOffer) {
            logger.info("Call ".concat(_this34.callId, " addIceCandidates() failed to add remote ICE candidate"), err);
          } else {
            logger.debug("Call ".concat(_this34.callId, " addIceCandidates() failed to add remote ICE candidate because ignoring offer"), err);
          }
        }
      }
    })();
  }
  get hasPeerConnection() {
    return Boolean(this.peerConn);
  }
  initStats(stats) {
    var peerId = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : "unknown";
    this.stats = stats;
    this.stats.start();
  }
}
export function setTracksEnabled(tracks, enabled) {
  for (var track of tracks) {
    track.enabled = enabled;
  }
}
export function supportsMatrixCall() {
  // typeof prevents Node from erroring on an undefined reference
  if (typeof window === "undefined" || typeof document === "undefined") {
    // NB. We don't log here as apps try to create a call object as a test for
    // whether calls are supported, so we shouldn't fill the logs up.
    return false;
  }

  // Firefox throws on so little as accessing the RTCPeerConnection when operating in a secure mode.
  // There's some information at https://bugzilla.mozilla.org/show_bug.cgi?id=1542616 though the concern
  // is that the browser throwing a SecurityError will brick the client creation process.
  try {
    var _ref2, _ref3, _window$RTCPeerConnec;
    var supported = Boolean((_ref2 = (_ref3 = (_window$RTCPeerConnec = window.RTCPeerConnection) !== null && _window$RTCPeerConnec !== void 0 ? _window$RTCPeerConnec : window.RTCSessionDescription) !== null && _ref3 !== void 0 ? _ref3 : window.RTCIceCandidate) !== null && _ref2 !== void 0 ? _ref2 : navigator.mediaDevices);
    if (!supported) {
      /* istanbul ignore if */ // Adds a lot of noise to test runs, so disable logging there.
      if (process.env.NODE_ENV !== "test") {
        logger.error("WebRTC is not supported in this browser / environment");
      }
      return false;
    }
  } catch (e) {
    logger.error("Exception thrown when trying to access WebRTC", e);
    return false;
  }
  return true;
}

/**
 * DEPRECATED
 * Use client.createCall()
 *
 * Create a new Matrix call for the browser.
 * @param client - The client instance to use.
 * @param roomId - The room the call is in.
 * @param options - DEPRECATED optional options map.
 * @returns the call or null if the browser doesn't support calling.
 */
export function createNewMatrixCall(client, roomId, options) {
  if (!supportsMatrixCall()) return null;
  var optionsForceTURN = options ? options.forceTURN : false;
  var opts = {
    client: client,
    roomId: roomId,
    invitee: options === null || options === void 0 ? void 0 : options.invitee,
    turnServers: client.getTurnServers(),
    // call level options
    forceTURN: client.forceTURN || optionsForceTURN,
    opponentDeviceId: options === null || options === void 0 ? void 0 : options.opponentDeviceId,
    opponentSessionId: options === null || options === void 0 ? void 0 : options.opponentSessionId,
    groupCallId: options === null || options === void 0 ? void 0 : options.groupCallId
  };
  var call = new MatrixCall(opts);
  client.reEmitter.reEmit(call, Object.values(CallEvent));
  return call;
}
//# sourceMappingURL=call.js.map