import _asyncToGenerator from "@babel/runtime/helpers/asyncToGenerator";
import _defineProperty from "@babel/runtime/helpers/defineProperty";
/*
Copyright 2023 - 2024 The Matrix.org Foundation C.I.C.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

import * as RustSdkCryptoJs from "@matrix-org/matrix-sdk-crypto-wasm";
import { ClientPrefix, MatrixError, Method } from "../http-api/index.js";
import { TypedEventEmitter } from "../models/typed-event-emitter.js";
import { encodeUri, logDuration } from "../utils.js";
import { sleep } from "../utils.js";
import { CryptoEvent, ImportRoomKeyStage } from "../crypto-api/index.js";

/** Authentification of the backup info, depends on algorithm */

/**
 * Holds information of a created keybackup.
 * Useful to get the generated private key material and save it securely somewhere.
 */

/**
 * @internal
 */
export class RustBackupManager extends TypedEventEmitter {
  constructor(logger, olmMachine, http, outgoingRequestProcessor) {
    super();
    this.logger = logger;
    this.olmMachine = olmMachine;
    this.http = http;
    this.outgoingRequestProcessor = outgoingRequestProcessor;
    /** Have we checked if there is a backup on the server which we can use */
    _defineProperty(this, "checkedForBackup", false);
    /**
     * The latest backup version on the server, when we last checked.
     *
     * If there was no backup on the server, `null`. If our attempt to check resulted in an error, `undefined`.
     *
     * Note that the backup was not necessarily verified.
     */
    _defineProperty(this, "serverBackupInfo", undefined);
    _defineProperty(this, "activeBackupVersion", null);
    _defineProperty(this, "stopped", false);
    /** whether {@link backupKeysLoop} is currently running */
    _defineProperty(this, "backupKeysLoopRunning", false);
    _defineProperty(this, "keyBackupCheckInProgress", null);
  }

  /**
   * Tells the RustBackupManager to stop.
   * The RustBackupManager is scheduling background uploads of keys to the backup, this
   * call allows to cancel the process when the client is stoppped.
   */
  stop() {
    this.stopped = true;
  }

  /**
   * Get the backup version we are currently backing up to, if any
   */
  getActiveBackupVersion() {
    var _this = this;
    return _asyncToGenerator(function* () {
      if (!(yield _this.olmMachine.isBackupEnabled())) return null;
      return _this.activeBackupVersion;
    })();
  }

  /**
   * Return the details of the latest backup on the server, when we last checked.
   *
   * This normally returns a cached value, but if we haven't yet made a request to the server, it will fire one off.
   * It will always return the details of the active backup if key backup is enabled.
   *
   * If there was no backup on the server, `null`. If our attempt to check resulted in an error, `undefined`.
   */
  getServerBackupInfo() {
    var _this2 = this;
    return _asyncToGenerator(function* () {
      // Do a validity check if we haven't already done one. The check is likely to fail if we don't yet have the
      // backup keys -- but as a side-effect, it will populate `serverBackupInfo`.
      yield _this2.checkKeyBackupAndEnable(false);
      return _this2.serverBackupInfo;
    })();
  }

  /**
   * Determine if a key backup can be trusted.
   *
   * @param info - key backup info dict from {@link CryptoApi.getKeyBackupInfo}.
   */
  isKeyBackupTrusted(info) {
    var _this3 = this;
    return _asyncToGenerator(function* () {
      var signatureVerification = yield _this3.olmMachine.verifyBackup(info);
      var backupKeys = yield _this3.olmMachine.getBackupKeys();
      var decryptionKey = backupKeys === null || backupKeys === void 0 ? void 0 : backupKeys.decryptionKey;
      var backupMatchesSavedPrivateKey = !!decryptionKey && _this3.backupInfoMatchesBackupDecryptionKey(info, decryptionKey);
      return {
        matchesDecryptionKey: backupMatchesSavedPrivateKey,
        trusted: signatureVerification.trusted()
      };
    })();
  }

  /**
   * Re-check the key backup and enable/disable it as appropriate.
   *
   * @param force - whether we should force a re-check even if one has already happened.
   */
  checkKeyBackupAndEnable(force) {
    if (!force && this.checkedForBackup) {
      return Promise.resolve(null);
    }

    // make sure there is only one check going on at a time
    if (!this.keyBackupCheckInProgress) {
      this.keyBackupCheckInProgress = this.doCheckKeyBackup().finally(() => {
        this.keyBackupCheckInProgress = null;
      });
    }
    return this.keyBackupCheckInProgress;
  }

  /**
   * Handles a backup secret received event and store it if it matches the current backup version.
   *
   * @param secret - The secret as received from a `m.secret.send` event for secret `m.megolm_backup.v1`.
   * @returns true if the secret is valid and has been stored, false otherwise.
   */
  handleBackupSecretReceived(secret) {
    var _this4 = this;
    return _asyncToGenerator(function* () {
      var _latestBackupInfo;
      // Currently we only receive the decryption key without any key backup version. It is important to
      // check that the secret is valid for the current version before storing it.
      // We force a check to ensure to have the latest version.
      var latestBackupInfo;
      try {
        latestBackupInfo = yield _this4.requestKeyBackupVersion();
      } catch (e) {
        _this4.logger.warn("handleBackupSecretReceived: Error checking for latest key backup", e);
        return false;
      }
      if (!((_latestBackupInfo = latestBackupInfo) !== null && _latestBackupInfo !== void 0 && _latestBackupInfo.version)) {
        // There is no server-side key backup.
        // This decryption key is useless to us.
        _this4.logger.warn("handleBackupSecretReceived: Received a backup decryption key, but there is no trusted server-side key backup");
        return false;
      }
      try {
        var backupDecryptionKey = RustSdkCryptoJs.BackupDecryptionKey.fromBase64(secret);
        var privateKeyMatches = _this4.backupInfoMatchesBackupDecryptionKey(latestBackupInfo, backupDecryptionKey);
        if (!privateKeyMatches) {
          _this4.logger.warn("handleBackupSecretReceived: Private decryption key does not match the public key of the current remote backup.");
          // just ignore the secret
          return false;
        }
        _this4.logger.info("handleBackupSecretReceived: A valid backup decryption key has been received and stored in cache.");
        yield _this4.saveBackupDecryptionKey(backupDecryptionKey, latestBackupInfo.version);
        return true;
      } catch (e) {
        _this4.logger.warn("handleBackupSecretReceived: Invalid backup decryption key", e);
      }
      return false;
    })();
  }
  saveBackupDecryptionKey(backupDecryptionKey, version) {
    var _this5 = this;
    return _asyncToGenerator(function* () {
      yield _this5.olmMachine.saveBackupDecryptionKey(backupDecryptionKey, version);
      // Emit an event that we have a new backup decryption key, so that the sdk can start
      // importing keys from backup if needed.
      _this5.emit(CryptoEvent.KeyBackupDecryptionKeyCached, version);
    })();
  }

  /**
   * Import a list of room keys previously exported by exportRoomKeys
   *
   * @param keys - a list of session export objects
   * @param opts - options object
   * @returns a promise which resolves once the keys have been imported
   */
  importRoomKeys(keys, opts) {
    var _this6 = this;
    return _asyncToGenerator(function* () {
      yield _this6.importRoomKeysAsJson(JSON.stringify(keys), opts);
    })();
  }

  /**
   * Import a list of room keys previously exported by exportRoomKeysAsJson
   *
   * @param jsonKeys - a JSON string encoding a list of session export objects,
   *    each of which is an IMegolmSessionData
   * @param opts - options object
   * @returns a promise which resolves once the keys have been imported
   */
  importRoomKeysAsJson(jsonKeys, opts) {
    var _this7 = this;
    return _asyncToGenerator(function* () {
      yield _this7.olmMachine.importExportedRoomKeys(jsonKeys, (progress, total) => {
        var _opts$progressCallbac;
        var importOpt = {
          total: Number(total),
          successes: Number(progress),
          stage: ImportRoomKeyStage.LoadKeys,
          failures: 0
        };
        opts === null || opts === void 0 || (_opts$progressCallbac = opts.progressCallback) === null || _opts$progressCallbac === void 0 || _opts$progressCallbac.call(opts, importOpt);
      });
    })();
  }

  /**
   * Implementation of {@link CryptoBackend#importBackedUpRoomKeys}.
   */
  importBackedUpRoomKeys(keys, backupVersion, opts) {
    var _this8 = this;
    return _asyncToGenerator(function* () {
      var keysByRoom = new Map();
      for (var key of keys) {
        var roomId = new RustSdkCryptoJs.RoomId(key.room_id);
        if (!keysByRoom.has(roomId)) {
          keysByRoom.set(roomId, new Map());
        }
        keysByRoom.get(roomId).set(key.session_id, key);
      }
      yield _this8.olmMachine.importBackedUpRoomKeys(keysByRoom, (progress, total, failures) => {
        var _opts$progressCallbac2;
        var importOpt = {
          total: Number(total),
          successes: Number(progress),
          stage: ImportRoomKeyStage.LoadKeys,
          failures: Number(failures)
        };
        opts === null || opts === void 0 || (_opts$progressCallbac2 = opts.progressCallback) === null || _opts$progressCallbac2 === void 0 || _opts$progressCallbac2.call(opts, importOpt);
      }, backupVersion);
    })();
  }
  /** Helper for `checkKeyBackup` */
  doCheckKeyBackup() {
    var _this9 = this;
    return _asyncToGenerator(function* () {
      _this9.logger.debug("Checking key backup status...");
      var backupInfo;
      try {
        backupInfo = yield _this9.requestKeyBackupVersion();
      } catch (e) {
        _this9.logger.warn("Error checking for active key backup", e);
        _this9.serverBackupInfo = undefined;
        return null;
      }
      _this9.checkedForBackup = true;
      if (backupInfo && !backupInfo.version) {
        _this9.logger.warn("active backup lacks a useful 'version'; ignoring it");
        backupInfo = undefined;
      }
      _this9.serverBackupInfo = backupInfo;
      var activeVersion = yield _this9.getActiveBackupVersion();
      if (!backupInfo) {
        if (activeVersion !== null) {
          _this9.logger.debug("No key backup present on server: disabling key backup");
          yield _this9.disableKeyBackup();
        } else {
          _this9.logger.debug("No key backup present on server: not enabling key backup");
        }
        return null;
      }
      var trustInfo = yield _this9.isKeyBackupTrusted(backupInfo);

      // Per the spec, we should enable key upload if either (a) the backup is signed by a trusted key, or
      // (b) the public key matches the private decryption key that we have received from 4S.
      if (!trustInfo.matchesDecryptionKey && !trustInfo.trusted) {
        if (activeVersion !== null) {
          _this9.logger.debug("Key backup present on server but not trusted: disabling key backup");
          yield _this9.disableKeyBackup();
        } else {
          _this9.logger.debug("Key backup present on server but not trusted: not enabling key backup");
        }
      } else {
        if (activeVersion === null) {
          _this9.logger.debug("Found usable key backup v".concat(backupInfo.version, ": enabling key backups"));
          yield _this9.enableKeyBackup(backupInfo);
        } else if (activeVersion !== backupInfo.version) {
          _this9.logger.debug("On backup version ".concat(activeVersion, " but found version ").concat(backupInfo.version, ": switching."));
          // This will remove any pending backup request, remove the backup key and reset the backup state of each room key we have.
          yield _this9.disableKeyBackup();
          // Enabling will now trigger re-upload of all the keys
          yield _this9.enableKeyBackup(backupInfo);
        } else {
          _this9.logger.debug("Backup version ".concat(backupInfo.version, " still current"));
        }
      }
      return {
        backupInfo,
        trustInfo
      };
    })();
  }
  enableKeyBackup(backupInfo) {
    var _this0 = this;
    return _asyncToGenerator(function* () {
      // we know for certain it must be a Curve25519 key, because we have verified it and only Curve25519
      // keys can be verified.
      //
      // we also checked it has a valid `version`.
      yield _this0.olmMachine.enableBackupV1(backupInfo.auth_data.public_key, backupInfo.version);
      _this0.activeBackupVersion = backupInfo.version;
      _this0.emit(CryptoEvent.KeyBackupStatus, true);
      _this0.backupKeysLoop();
    })();
  }

  /**
   * Restart the backup key loop if there is an active trusted backup.
   * Doesn't try to check the backup server side. To be called when a new
   * megolm key is known locally.
   */
  maybeUploadKey() {
    var _this1 = this;
    return _asyncToGenerator(function* () {
      if (_this1.activeBackupVersion != null) {
        _this1.backupKeysLoop();
      }
    })();
  }
  disableKeyBackup() {
    var _this10 = this;
    return _asyncToGenerator(function* () {
      yield _this10.olmMachine.disableBackup();
      _this10.activeBackupVersion = null;
      _this10.emit(CryptoEvent.KeyBackupStatus, false);
    })();
  }
  backupKeysLoop() {
    var _arguments = arguments,
      _this11 = this;
    return _asyncToGenerator(function* () {
      var maxDelay = _arguments.length > 0 && _arguments[0] !== undefined ? _arguments[0] : 10000;
      if (_this11.backupKeysLoopRunning) {
        _this11.logger.debug("Backup loop already running");
        return;
      }
      _this11.backupKeysLoopRunning = true;
      _this11.logger.debug("Backup: Starting keys upload loop for backup version:".concat(_this11.activeBackupVersion, "."));

      // wait between 0 and `maxDelay` seconds, to avoid backup
      // requests from different clients hitting the server all at
      // the same time when a new key is sent
      var delay = Math.random() * maxDelay;
      yield sleep(delay);
      try {
        // number of consecutive network failures for exponential backoff
        var numFailures = 0;
        // The number of keys left to back up. (Populated lazily: see more comments below.)
        var remainingToUploadCount = null;
        // To avoid computing the key when only a few keys were added (after a sync for example),
        // we compute the count only when at least two iterations are needed.
        var isFirstIteration = true;
        while (!_this11.stopped) {
          // Get a batch of room keys to upload
          var request = undefined;
          try {
            request = yield logDuration(_this11.logger, "BackupRoomKeys: Get keys to backup from rust crypto-sdk", /*#__PURE__*/_asyncToGenerator(function* () {
              return yield _this11.olmMachine.backupRoomKeys();
            }));
          } catch (err) {
            _this11.logger.error("Backup: Failed to get keys to backup from rust crypto-sdk", err);
          }
          if (!request || _this11.stopped || !_this11.activeBackupVersion) {
            _this11.logger.debug("Backup: Ending loop for version ".concat(_this11.activeBackupVersion, "."));
            if (!request) {
              // nothing more to upload
              _this11.emit(CryptoEvent.KeyBackupSessionsRemaining, 0);
            }
            return;
          }
          try {
            yield _this11.outgoingRequestProcessor.makeOutgoingRequest(request);
            numFailures = 0;
            if (_this11.stopped) break;

            // Key count performance (`olmMachine.roomKeyCounts()`) can be pretty bad on some configurations.
            // In particular, we detected on some M1 macs that when the object store reaches a threshold, the count
            // performance stops growing in O(n) and suddenly becomes very slow (40s, 60s or more).
            // For reference, the performance drop occurs around 300-400k keys on the platforms where this issue is observed.
            // Even on other configurations, the count can take several seconds.
            // This will block other operations on the database, like sending messages.
            //
            // This is a workaround to avoid calling `olmMachine.roomKeyCounts()` too often, and only when necessary.
            // We don't call it on the first loop because there could be only a few keys to upload, and we don't want to wait for the count.
            if (!isFirstIteration && remainingToUploadCount === null) {
              try {
                var keyCount = yield _this11.olmMachine.roomKeyCounts();
                remainingToUploadCount = keyCount.total - keyCount.backedUp;
              } catch (err) {
                _this11.logger.error("Backup: Failed to get key counts from rust crypto-sdk", err);
              }
            }
            if (remainingToUploadCount !== null) {
              _this11.emit(CryptoEvent.KeyBackupSessionsRemaining, remainingToUploadCount);
              var keysCountInBatch = _this11.keysCountInBatch(request);
              // `OlmMachine.roomKeyCounts` is called only once for the current backupKeysLoop. But new
              // keys could be added during the current loop (after a sync for example).
              // So the count can get out of sync with the real number of remaining keys to upload.
              // Depending on the number of new keys imported and the time to complete the loop,
              // this could result in multiple events being emitted with a remaining key count of 0.
              remainingToUploadCount = Math.max(remainingToUploadCount - keysCountInBatch, 0);
            }
          } catch (err) {
            numFailures++;
            _this11.logger.error("Backup: Error processing backup request for rust crypto-sdk", err);
            if (err instanceof MatrixError) {
              var errCode = err.data.errcode;
              if (errCode == "M_NOT_FOUND" || errCode == "M_WRONG_ROOM_KEYS_VERSION") {
                _this11.logger.debug("Backup: Failed to upload keys to current vesion: ".concat(errCode, "."));
                try {
                  yield _this11.disableKeyBackup();
                } catch (error) {
                  _this11.logger.error("Backup: An error occurred while disabling key backup:", error);
                }
                _this11.emit(CryptoEvent.KeyBackupFailed, err.data.errcode);
                // There was an active backup and we are out of sync with the server
                // force a check server side
                _this11.backupKeysLoopRunning = false;
                _this11.checkKeyBackupAndEnable(true);
                return;
              } else if (err.isRateLimitError()) {
                // wait for that and then continue?
                try {
                  var waitTime = err.getRetryAfterMs();
                  if (waitTime && waitTime > 0) {
                    yield sleep(waitTime);
                    continue;
                  }
                } catch (error) {
                  _this11.logger.warn("Backup: An error occurred while retrieving a rate-limit retry delay", error);
                } // else go to the normal backoff
              }
            }

            // Some other errors (mx, network, or CORS or invalid urls?) anyhow backoff
            // exponential backoff if we have failures
            yield sleep(1000 * Math.pow(2, Math.min(numFailures - 1, 4)));
          }
          isFirstIteration = false;
        }
      } finally {
        _this11.backupKeysLoopRunning = false;
      }
    })();
  }

  /**
   * Utility method to count the number of keys in a backup request, in order to update the remaining keys count.
   * This should be the chunk size of the backup request for all requests but the last, but we don't have access to it
   * (it's static in the Rust SDK).
   * @param batch - The backup request to count the keys from.
   *
   * @returns The number of keys in the backup request.
   */
  keysCountInBatch(batch) {
    var parsedBody = JSON.parse(batch.body);
    return countKeysInBackup(parsedBody);
  }

  /**
   * Get information about a key backup from the server
   * - If version is provided, get information about that backup version.
   * - If no version is provided, get information about the latest backup.
   *
   * @param version - The version of the backup to get information about.
   * @returns Information object from API or null if there is no active backup.
   */
  requestKeyBackupVersion(version) {
    var _this12 = this;
    return _asyncToGenerator(function* () {
      return yield requestKeyBackupVersion(_this12.http, version);
    })();
  }

  /**
   * Creates a new key backup by generating a new random private key.
   *
   * If there is an existing backup server side it will be deleted and replaced
   * by the new one.
   *
   * @param signObject - Method that should sign the backup with existing device and
   * existing identity.
   * @returns a KeyBackupCreationInfo - All information related to the backup.
   */
  setupKeyBackup(signObject) {
    var _this13 = this;
    return _asyncToGenerator(function* () {
      // Clean up any existing backup
      yield _this13.deleteAllKeyBackupVersions();
      var randomKey = RustSdkCryptoJs.BackupDecryptionKey.createRandomKey();
      var pubKey = randomKey.megolmV1PublicKey;
      var authData = {
        public_key: pubKey.publicKeyBase64
      };
      yield signObject(authData);
      var res = yield _this13.http.authedRequest(Method.Post, "/room_keys/version", undefined, {
        algorithm: pubKey.algorithm,
        auth_data: authData
      }, {
        prefix: ClientPrefix.V3
      });
      yield _this13.saveBackupDecryptionKey(randomKey, res.version);
      return {
        version: res.version,
        algorithm: pubKey.algorithm,
        authData: authData,
        decryptionKey: randomKey
      };
    })();
  }

  /**
   * Deletes all key backups.
   *
   * Will call the API to delete active backup until there is no more present.
   */
  deleteAllKeyBackupVersions() {
    var _this14 = this;
    return _asyncToGenerator(function* () {
      var _yield$_this14$reques, _yield$_this14$reques2;
      // there could be several backup versions. Delete all to be safe.
      var current = (_yield$_this14$reques = (_yield$_this14$reques2 = yield _this14.requestKeyBackupVersion()) === null || _yield$_this14$reques2 === void 0 ? void 0 : _yield$_this14$reques2.version) !== null && _yield$_this14$reques !== void 0 ? _yield$_this14$reques : null;
      while (current != null) {
        var _yield$_this14$reques3, _yield$_this14$reques4;
        yield _this14.deleteKeyBackupVersion(current);
        current = (_yield$_this14$reques3 = (_yield$_this14$reques4 = yield _this14.requestKeyBackupVersion()) === null || _yield$_this14$reques4 === void 0 ? void 0 : _yield$_this14$reques4.version) !== null && _yield$_this14$reques3 !== void 0 ? _yield$_this14$reques3 : null;
      }

      // XXX: Should this also update Secret Storage and delete any existing keys?
    })();
  }

  /**
   * Deletes the given key backup.
   *
   * @param version - The backup version to delete.
   */
  deleteKeyBackupVersion(version) {
    var _this15 = this;
    return _asyncToGenerator(function* () {
      _this15.logger.debug("deleteKeyBackupVersion v:".concat(version));
      var path = encodeUri("/room_keys/version/$version", {
        $version: version
      });
      yield _this15.http.authedRequest(Method.Delete, path, undefined, undefined, {
        prefix: ClientPrefix.V3
      });
      // If the backup we are deleting is the active one, we need to disable the key backup and to have the local properties reset
      if (_this15.activeBackupVersion === version) {
        _this15.serverBackupInfo = null;
        yield _this15.disableKeyBackup();
      }
    })();
  }

  /**
   * Creates a new backup decryptor for the given private key.
   * @param decryptionKey - The private key to use for decryption.
   */
  createBackupDecryptor(decryptionKey) {
    return new RustBackupDecryptor(this.logger, decryptionKey);
  }

  /**
   * Restore a key backup.
   *
   * @param backupVersion - The version of the backup to restore.
   * @param backupDecryptor - The backup decryptor to use to decrypt the keys.
   * @param opts - Options for the restore.
   * @returns The total number of keys and the total imported.
   */
  restoreKeyBackup(backupVersion, backupDecryptor, opts) {
    var _this16 = this;
    return _asyncToGenerator(function* () {
      var keyBackup = yield _this16.downloadKeyBackup(backupVersion);
      return _this16.importKeyBackup(keyBackup, backupVersion, backupDecryptor, opts);
    })();
  }

  /**
   * Call `/room_keys/keys` to download the key backup (room keys) for the given backup version.
   * https://spec.matrix.org/v1.12/client-server-api/#get_matrixclientv3room_keyskeys
   *
   * @param backupVersion
   * @returns The key backup response.
   */
  downloadKeyBackup(backupVersion) {
    return this.http.authedRequest(Method.Get, "/room_keys/keys", {
      version: backupVersion
    }, undefined, {
      prefix: ClientPrefix.V3
    });
  }

  /**
   * Import the room keys from a `/room_keys/keys` call.
   * Calls `opts.progressCallback` with the progress of the import.
   *
   * @param keyBackup - The response from the server containing the keys to import.
   * @param backupVersion - The version of the backup info.
   * @param backupDecryptor - The backup decryptor to use to decrypt the keys.
   * @param opts - Options for the import.
   *
   * @returns The total number of keys and the total imported.
   *
   * @private
   */
  importKeyBackup(keyBackup, backupVersion, backupDecryptor, opts) {
    var _this17 = this;
    return _asyncToGenerator(function* () {
      var _opts$progressCallbac3;
      // We have a full backup here, it can get quite big, so we need to decrypt and import it in chunks.

      var CHUNK_SIZE = 200;
      // Get the total count as a first pass
      var totalKeyCount = countKeysInBackup(keyBackup);
      var totalImported = 0;
      var totalFailures = 0;
      opts === null || opts === void 0 || (_opts$progressCallbac3 = opts.progressCallback) === null || _opts$progressCallbac3 === void 0 || _opts$progressCallbac3.call(opts, {
        total: totalKeyCount,
        successes: totalImported,
        stage: ImportRoomKeyStage.LoadKeys,
        failures: totalFailures
      });

      /**
       * This method is called when we have enough chunks to decrypt.
       * It will decrypt the chunks and try to import the room keys.
       * @param roomChunks
       */
      var handleChunkCallback = /*#__PURE__*/function () {
        var _ref2 = _asyncToGenerator(function* (roomChunks) {
          var _opts$progressCallbac4;
          var currentChunk = [];
          var _loop = function* _loop(roomId) {
            // Decrypt the sessions for the given room
            var decryptedSessions = yield backupDecryptor.decryptSessions(roomChunks.get(roomId));
            // Add the decrypted sessions to the current chunk
            decryptedSessions.forEach(session => {
              // We set the room_id for each session
              session.room_id = roomId;
              currentChunk.push(session);
            });
          };
          for (var roomId of roomChunks.keys()) {
            yield* _loop(roomId);
          }

          // We have a chunk of decrypted keys: import them
          try {
            yield _this17.importBackedUpRoomKeys(currentChunk, backupVersion);
            totalImported += currentChunk.length;
          } catch (e) {
            totalFailures += currentChunk.length;
            // We failed to import some keys, but we should still try to import the rest?
            // Log the error and continue
            _this17.logger.error("Error importing keys from backup", e);
          }
          opts === null || opts === void 0 || (_opts$progressCallbac4 = opts.progressCallback) === null || _opts$progressCallbac4 === void 0 || _opts$progressCallbac4.call(opts, {
            total: totalKeyCount,
            successes: totalImported,
            stage: ImportRoomKeyStage.LoadKeys,
            failures: totalFailures
          });
        });
        return function handleChunkCallback(_x) {
          return _ref2.apply(this, arguments);
        };
      }();
      var groupChunkCount = 0;
      var chunkGroupByRoom = new Map();

      // Iterate over the rooms and sessions to group them in chunks
      // And we call the handleChunkCallback when we have enough chunks to decrypt
      for (var [roomId, roomData] of Object.entries(keyBackup.rooms)) {
        // If there are no sessions for the room, skip it
        if (!roomData.sessions) continue;

        // Initialize a new chunk group for the current room
        chunkGroupByRoom.set(roomId, {});
        for (var [sessionId, session] of Object.entries(roomData.sessions)) {
          // We set previously the chunk group for the current room, so we can safely get it
          var sessionsForRoom = chunkGroupByRoom.get(roomId);
          sessionsForRoom[sessionId] = session;
          groupChunkCount += 1;
          // If we have enough chunks to decrypt, call the block callback
          if (groupChunkCount >= CHUNK_SIZE) {
            // We have enough chunks to decrypt
            yield handleChunkCallback(chunkGroupByRoom);
            // Reset the chunk group
            chunkGroupByRoom = new Map();
            // There might be remaining keys for that room, so add back an entry for the current room.
            chunkGroupByRoom.set(roomId, {});
            groupChunkCount = 0;
          }
        }
      }

      // Handle remaining chunk if needed
      if (groupChunkCount > 0) {
        yield handleChunkCallback(chunkGroupByRoom);
      }
      return {
        total: totalKeyCount,
        imported: totalImported
      };
    })();
  }

  /**
   * Checks if the provided backup info matches the given private key.
   *
   * @param info - The backup info to check.
   * @param backupDecryptionKey - The `BackupDecryptionKey` private key to check against.
   * @returns `true` if the private key can decrypt the backup, `false` otherwise.
   */
  backupInfoMatchesBackupDecryptionKey(info, backupDecryptionKey) {
    var _info$auth_data;
    if (info.algorithm !== "m.megolm_backup.v1.curve25519-aes-sha2") {
      this.logger.warn("backupMatchesPrivateKey: Unsupported backup algorithm", info.algorithm);
      return false;
    }
    return ((_info$auth_data = info.auth_data) === null || _info$auth_data === void 0 ? void 0 : _info$auth_data.public_key) === backupDecryptionKey.megolmV1PublicKey.publicKeyBase64;
  }
}
/**
 * Implementation of {@link BackupDecryptor} for the rust crypto backend.
 */
export class RustBackupDecryptor {
  constructor(logger, decryptionKey) {
    this.logger = logger;
    _defineProperty(this, "decryptionKey", void 0);
    _defineProperty(this, "sourceTrusted", void 0);
    this.decryptionKey = decryptionKey;
    this.sourceTrusted = false;
  }

  /**
   * Implements {@link BackupDecryptor#decryptSessions}
   */
  decryptSessions(ciphertexts) {
    var _this18 = this;
    return _asyncToGenerator(function* () {
      var keys = [];
      for (var [sessionId, sessionData] of Object.entries(ciphertexts)) {
        try {
          var decrypted = JSON.parse(_this18.decryptionKey.decryptV1(sessionData.session_data.ephemeral, sessionData.session_data.mac, sessionData.session_data.ciphertext));
          decrypted.session_id = sessionId;
          keys.push(decrypted);
        } catch (e) {
          _this18.logger.debug("Failed to decrypt megolm session from backup", e, sessionData);
        }
      }
      return keys;
    })();
  }

  /**
   * Implements {@link BackupDecryptor#free}
   */
  free() {
    this.decryptionKey.free();
  }
}

/**
 * Fetch a key backup info from the server.
 *
 * If `version` is provided, calls `GET /room_keys/version/$version` and gets the backup info for that version.
 * See https://spec.matrix.org/v1.12/client-server-api/#get_matrixclientv3room_keysversionversion.
 *
 * If not, calls `GET /room_keys/version` and gets the latest backup info.
 * See https://spec.matrix.org/v1.12/client-server-api/#get_matrixclientv3room_keysversion
 *
 * @param http
 * @param version - the specific version of the backup info to fetch
 * @returns The key backup info or null if there is no backup.
 */
export function requestKeyBackupVersion(_x2, _x3) {
  return _requestKeyBackupVersion.apply(this, arguments);
}

/**
 * Checks if the provided decryption key matches the public key of the key backup info.
 *
 * @param decryptionKey - The decryption key to check.
 * @param keyBackupInfo - The key backup info to check against.
 * @returns `true` if the decryption key matches the key backup info, `false` otherwise.
 */
function _requestKeyBackupVersion() {
  _requestKeyBackupVersion = _asyncToGenerator(function* (http, version) {
    try {
      var path = version ? encodeUri("/room_keys/version/$version", {
        $version: version
      }) : "/room_keys/version";
      return yield http.authedRequest(Method.Get, path, undefined, undefined, {
        prefix: ClientPrefix.V3
      });
    } catch (e) {
      if (e.errcode === "M_NOT_FOUND") {
        return null;
      } else {
        throw e;
      }
    }
  });
  return _requestKeyBackupVersion.apply(this, arguments);
}
export function decryptionKeyMatchesKeyBackupInfo(decryptionKey, keyBackupInfo) {
  var authData = keyBackupInfo.auth_data;
  return authData.public_key === decryptionKey.megolmV1PublicKey.publicKeyBase64;
}

/**
 * Counts the total number of keys present in a key backup.
 * @param keyBackup - The key backup to count the keys from.
 * @returns The total number of keys in the backup.
 */
function countKeysInBackup(keyBackup) {
  var count = 0;
  for (var {
    sessions
  } of Object.values(keyBackup.rooms)) {
    count += Object.keys(sessions).length;
  }
  return count;
}

/**
 * Response from GET `/room_keys/keys` endpoint.
 * See https://spec.matrix.org/latest/client-server-api/#get_matrixclientv3room_keyskeys
 */
//# sourceMappingURL=backup.js.map