import _defineProperty from "@babel/runtime/helpers/defineProperty";
import { logger as loggerInstance } from "../logger.js";
import { TypedEventEmitter } from "./typed-event-emitter.js";
var logger = loggerInstance.getChild("RoomStickyEvents");
export var RoomStickyEventsEvent = /*#__PURE__*/function (RoomStickyEventsEvent) {
  RoomStickyEventsEvent["Update"] = "RoomStickyEvents.Update";
  return RoomStickyEventsEvent;
}({});
/**
 * Tracks sticky events on behalf of one room, and fires an event
 * whenever a sticky event is updated or replaced.
 */
export class RoomStickyEventsStore extends TypedEventEmitter {
  constructor() {
    super(...arguments);
    _defineProperty(this, "stickyEventsMap", new Map());
    // (type -> stickyKey+userId) -> event
    _defineProperty(this, "unkeyedStickyEvents", new Set());
    _defineProperty(this, "stickyEventTimer", void 0);
    _defineProperty(this, "nextStickyEventExpiryTs", Number.MAX_SAFE_INTEGER);
    /**
     * Clean out any expired sticky events.
     */
    _defineProperty(this, "cleanExpiredStickyEvents", () => {
      var now = Date.now();
      var removedEvents = [];

      // We will recalculate this as we check all events.
      this.nextStickyEventExpiryTs = Number.MAX_SAFE_INTEGER;
      for (var [eventType, innerEvents] of this.stickyEventsMap.entries()) {
        var _this$stickyEventsMap;
        for (var [innerMapKey, event] of innerEvents) {
          // we only added items with `sticky` into this map so we can assert non-null here
          if (now >= event.unstableStickyExpiresAt) {
            logger.debug("Expiring sticky event", event.getId());
            removedEvents.push(event);
            this.stickyEventsMap.get(eventType).delete(innerMapKey);
          } else {
            // If not removing the event, check to see if it's the next lowest expiry.
            this.nextStickyEventExpiryTs = Math.min(this.nextStickyEventExpiryTs, event.unstableStickyExpiresAt);
          }
        }
        // Clean up map after use.
        if (((_this$stickyEventsMap = this.stickyEventsMap.get(eventType)) === null || _this$stickyEventsMap === void 0 ? void 0 : _this$stickyEventsMap.size) === 0) {
          this.stickyEventsMap.delete(eventType);
        }
      }
      for (var _event of this.unkeyedStickyEvents) {
        if (now >= _event.unstableStickyExpiresAt) {
          logger.debug("Expiring sticky event", _event.getId());
          this.unkeyedStickyEvents.delete(_event);
          removedEvents.push(_event);
        } else {
          // If not removing the event, check to see if it's the next lowest expiry.
          this.nextStickyEventExpiryTs = Math.min(this.nextStickyEventExpiryTs, _event.unstableStickyExpiresAt);
        }
      }
      if (removedEvents.length) {
        this.emit(RoomStickyEventsEvent.Update, [], [], removedEvents);
      }
      // Finally, schedule the next run.
      this.scheduleStickyTimer();
    });
  }
  /**
   * Get all sticky events that are currently active.
   * @returns An iterable set of events.
   */
  *getStickyEvents() {
    yield* this.unkeyedStickyEvents;
    for (var innerMap of this.stickyEventsMap.values()) {
      yield* innerMap.values();
    }
  }

  /**
   * Get an active sticky event that match the given `type`, `sender`, and `stickyKey`
   * @param type The event `type`.
   * @param sender The sender of the sticky event.
   * @param stickyKey The sticky key used by the event.
   * @returns A matching active sticky event, or undefined.
   */
  getKeyedStickyEvent(sender, type, stickyKey) {
    var _this$stickyEventsMap2;
    return (_this$stickyEventsMap2 = this.stickyEventsMap.get(type)) === null || _this$stickyEventsMap2 === void 0 ? void 0 : _this$stickyEventsMap2.get("".concat(stickyKey).concat(sender));
  }

  /**
   * Get active sticky events without a sticky key that match the given `type` and `sender`.
   * @param type The event `type`.
   * @param sender The sender of the sticky event.
   * @returns An array of matching sticky events.
   */
  getUnkeyedStickyEvent(sender, type) {
    return [...this.unkeyedStickyEvents].filter(ev => ev.getType() === type && ev.getSender() === sender);
  }

  /**
   * Adds a sticky event into the local sticky event map.
   *
   * NOTE: This will not cause `RoomEvent.StickyEvents` to be emitted.
   *
   * @throws If the `event` does not contain valid sticky data.
   * @param event The MatrixEvent that contains sticky data.
   * @returns An object describing whether the event was added to the map,
   *          and the previous event it may have replaced.
   */
  addStickyEvent(event) {
    var stickyKey = event.getContent().msc4354_sticky_key;
    if (typeof stickyKey !== "string" && stickyKey !== undefined) {
      throw new Error("".concat(event.getId(), " is missing msc4354_sticky_key"));
    }

    // With this we have the guarantee, that all events in stickyEventsMap are correctly formatted
    if (event.unstableStickyExpiresAt === undefined) {
      throw new Error("".concat(event.getId(), " is missing msc4354_sticky.duration_ms"));
    }
    var sender = event.getSender();
    var type = event.getType();
    if (!sender) {
      throw new Error("".concat(event.getId(), " is missing a sender"));
    } else if (event.unstableStickyExpiresAt <= Date.now()) {
      logger.info("ignored sticky event with older expiration time than current time", stickyKey);
      return {
        added: false
      };
    }

    // While we fully expect the server to always provide the correct value,
    // this is just insurance to protect against attacks on our Map.
    if (!sender.startsWith("@")) {
      throw new Error("Expected sender to start with @");
    }
    var prevEvent;
    if (stickyKey !== undefined) {
      var _this$stickyEventsMap3, _event$getId, _prevEvent$getId;
      // Why this is safe:
      // A type may contain anything but the *sender* is tightly
      // constrained so that a key will always end with a @<user_id>
      // E.g. Where a malicous event type might be "rtc.member.event@foo:bar" the key becomes:
      // "rtc.member.event.@foo:bar@bar:baz"
      var innerMapKey = "".concat(stickyKey).concat(sender);
      prevEvent = (_this$stickyEventsMap3 = this.stickyEventsMap.get(type)) === null || _this$stickyEventsMap3 === void 0 ? void 0 : _this$stickyEventsMap3.get(innerMapKey);

      // sticky events are not allowed to expire sooner than their predecessor.
      if (prevEvent && event.unstableStickyExpiresAt < prevEvent.unstableStickyExpiresAt) {
        logger.info("ignored sticky event with older expiry time", stickyKey);
        return {
          added: false
        };
      } else if (prevEvent && event.getTs() === prevEvent.getTs() && ((_event$getId = event.getId()) !== null && _event$getId !== void 0 ? _event$getId : "") < ((_prevEvent$getId = prevEvent.getId()) !== null && _prevEvent$getId !== void 0 ? _prevEvent$getId : "")) {
        // This path is unlikely, as it requires both events to have the same TS.
        logger.info("ignored sticky event due to 'id tie break rule' on sticky_key", stickyKey);
        return {
          added: false
        };
      }
      if (!this.stickyEventsMap.has(type)) {
        this.stickyEventsMap.set(type, new Map());
      }
      this.stickyEventsMap.get(type).set(innerMapKey, event);
    } else {
      this.unkeyedStickyEvents.add(event);
    }

    // Recalculate the next expiry time.
    this.nextStickyEventExpiryTs = Math.min(event.unstableStickyExpiresAt, this.nextStickyEventExpiryTs);
    this.scheduleStickyTimer();
    return {
      added: true,
      prevEvent
    };
  }

  /**
   * Add a series of sticky events, emitting `RoomEvent.StickyEvents` if any
   * changes were made.
   * @param events A set of new sticky events.
   */
  addStickyEvents(events) {
    var added = [];
    var updated = [];
    for (var event of events) {
      try {
        var result = this.addStickyEvent(event);
        if (result.added) {
          if (result.prevEvent) {
            // e is validated as a StickyMatrixEvent by virtue of `addStickyEvent` returning added: true.
            updated.push({
              current: event,
              previous: result.prevEvent
            });
          } else {
            added.push(event);
          }
        }
      } catch (ex) {
        logger.warn("ignored invalid sticky event", ex);
      }
    }
    if (added.length || updated.length) this.emit(RoomStickyEventsEvent.Update, added, updated, []);
    this.scheduleStickyTimer();
  }

  /**
   * Schedule the sticky event expiry timer. The timer will
   * run immediately if an event has already expired.
   */
  scheduleStickyTimer() {
    if (this.stickyEventTimer) {
      clearTimeout(this.stickyEventTimer);
      this.stickyEventTimer = undefined;
    }
    if (this.nextStickyEventExpiryTs === Number.MAX_SAFE_INTEGER) {
      // We have no events due to expire.
      return;
    } // otherwise, schedule in the future
    this.stickyEventTimer = setTimeout(this.cleanExpiredStickyEvents, this.nextStickyEventExpiryTs - Date.now());
  }
  /**
   * Clear all events and stop the timer from firing.
   */
  clear() {
    this.stickyEventsMap.clear();
    // Unschedule timer.
    this.nextStickyEventExpiryTs = Number.MAX_SAFE_INTEGER;
    this.scheduleStickyTimer();
  }
}
//# sourceMappingURL=room-sticky-events.js.map