import { type User } from "./user.ts";
import { type MatrixEvent } from "./event.ts";
import { type RoomState } from "./room-state.ts";
import { TypedEventEmitter } from "./typed-event-emitter.ts";
import { type Membership } from "../@types/membership.ts";
export declare enum RoomMemberEvent {
    Membership = "RoomMember.membership",
    Name = "RoomMember.name",
    PowerLevel = "RoomMember.powerLevel",
    Typing = "RoomMember.typing"
}
export type RoomMemberEventHandlerMap = {
    /**
     * Fires whenever any room member's membership state changes.
     * @param event - The matrix event which caused this event to fire.
     * @param member - The member whose RoomMember.membership changed.
     * @param oldMembership - The previous membership state. Null if it's a new member.
     * @example
     * ```
     * matrixClient.on("RoomMember.membership", function(event, member, oldMembership){
     *   var newState = member.membership;
     * });
     * ```
     */
    [RoomMemberEvent.Membership]: (event: MatrixEvent, member: RoomMember, oldMembership?: string) => void;
    /**
     * Fires whenever any room member's name changes.
     * @param event - The matrix event which caused this event to fire.
     * @param member - The member whose RoomMember.name changed.
     * @param oldName - The previous name. Null if the member didn't have a name previously.
     * @example
     * ```
     * matrixClient.on("RoomMember.name", function(event, member){
     *   var newName = member.name;
     * });
     * ```
     */
    [RoomMemberEvent.Name]: (event: MatrixEvent, member: RoomMember, oldName: string | null) => void;
    /**
     * Fires whenever any room member's power level changes.
     * @param event - The matrix event which caused this event to fire.
     * @param member - The member whose RoomMember.powerLevel changed.
     * @example
     * ```
     * matrixClient.on("RoomMember.powerLevel", function(event, member){
     *   var newPowerLevel = member.powerLevel;
     * });
     * ```
     */
    [RoomMemberEvent.PowerLevel]: (event: MatrixEvent, member: RoomMember) => void;
    /**
     * Fires whenever any room member's typing state changes.
     * @param event - The matrix event which caused this event to fire.
     * @param member - The member whose RoomMember.typing changed.
     * @example
     * ```
     * matrixClient.on("RoomMember.typing", function(event, member){
     *   var isTyping = member.typing;
     * });
     * ```
     */
    [RoomMemberEvent.Typing]: (event: MatrixEvent, member: RoomMember) => void;
};
export declare class RoomMember extends TypedEventEmitter<RoomMemberEvent, RoomMemberEventHandlerMap> {
    readonly roomId: string;
    readonly userId: string;
    private _isOutOfBand;
    private modified;
    requestedProfileInfo: boolean;
    /**
     * True if the room member is currently typing.
     */
    typing: boolean;
    /**
     * The human-readable name for this room member. This will be
     * disambiguated with a suffix of " (\@user_id:matrix.org)" if another member shares the
     * same displayname.
     */
    name: string;
    /**
     * The ambiguous displayname of this room member.
     */
    rawDisplayName: string;
    /**
     * The power level for this room member.
     */
    powerLevel: number;
    /**
     * The User object for this room member, if one exists.
     */
    user?: User;
    /**
     * The membership state for this room member e.g. 'join'.
     */
    membership?: Membership;
    /**
     * True if the member's name is disambiguated.
     */
    disambiguate: boolean;
    /**
     * The events describing this RoomMember.
     */
    events: {
        /**
         * The m.room.member event for this RoomMember.
         */
        member?: MatrixEvent;
    };
    /**
     * Construct a new room member.
     *
     * @param roomId - The room ID of the member.
     * @param userId - The user ID of the member.
     */
    constructor(roomId: string, userId: string);
    /**
     * Mark the member as coming from a channel that is not sync
     */
    markOutOfBand(): void;
    /**
     * @returns does the member come from a channel that is not sync?
     * This is used to store the member seperately
     * from the sync state so it available across browser sessions.
     */
    isOutOfBand(): boolean;
    /**
     * Update this room member's membership event. May fire "RoomMember.name" if
     * this event updates this member's name.
     * @param event - The `m.room.member` event
     * @param roomState - Optional. The room state to take into account
     * when calculating (e.g. for disambiguating users with the same name).
     *
     * @remarks
     * Fires {@link RoomMemberEvent.Name}
     * Fires {@link RoomMemberEvent.Membership}
     */
    setMembershipEvent(event: MatrixEvent, roomState?: RoomState): void;
    /**
     * Update this room member's power level event. Will fire
     * "RoomMember.powerLevel" if the new power level is different
     * @param powerLevel - The power level of the room member.
     *
     * @remarks
     * Fires {@link RoomMemberEvent.PowerLevel}
     */
    setPowerLevel(powerLevel: number, powerLevelEvent: MatrixEvent): void;
    /**
     * Update this room member's typing event. May fire "RoomMember.typing" if
     * this event changes this member's typing state.
     * @param event - The typing event
     *
     * @remarks
     * Fires {@link RoomMemberEvent.Typing}
     */
    setTypingEvent(event: MatrixEvent): void;
    /**
     * Update the last modified time to the current time.
     */
    private updateModifiedTime;
    /**
     * Get the timestamp when this RoomMember was last updated. This timestamp is
     * updated when properties on this RoomMember are updated.
     * It is updated <i>before</i> firing events.
     * @returns The timestamp
     */
    getLastModifiedTime(): number;
    isKicked(): boolean;
    /**
     * If this member was invited with the is_direct flag set, return
     * the user that invited this member
     * @returns user id of the inviter
     */
    getDMInviter(): string | undefined;
    /**
     * Get the avatar URL for a room member.
     * @param baseUrl - The base homeserver URL See
     * {@link MatrixClient#getHomeserverUrl}.
     * @param width - The desired width of the thumbnail.
     * @param height - The desired height of the thumbnail.
     * @param resizeMethod - The thumbnail resize method to use, either
     * "crop" or "scale".
     * @param allowDefault - (optional) Passing false causes this method to
     * return null if the user has no avatar image. Otherwise, a default image URL
     * will be returned. Default: true. (Deprecated)
     * @param allowDirectLinks - (optional) If true, the avatar URL will be
     * returned even if it is a direct hyperlink rather than a matrix content URL.
     * If false, any non-matrix content URLs will be ignored. Setting this option to
     * true will expose URLs that, if fetched, will leak information about the user
     * to anyone who they share a room with.
     * @param useAuthentication - (optional) If true, the caller supports authenticated
     * media and wants an authentication-required URL. Note that server support for
     * authenticated media will not be checked - it is the caller's responsibility
     * to do so before calling this function. Note also that useAuthentication
     * implies allowRedirects. Defaults to false (unauthenticated endpoints).
     * @returns the avatar URL or null.
     */
    getAvatarUrl(baseUrl: string, width: number, height: number, resizeMethod: string, allowDefault: boolean | undefined, allowDirectLinks: boolean, useAuthentication?: boolean): string | null;
    /**
     * get the mxc avatar url, either from a state event, or from a lazily loaded member
     * @returns the mxc avatar url
     */
    getMxcAvatarUrl(): string | undefined;
}
export declare const MXID_PATTERN: RegExp;
//# sourceMappingURL=room-member.d.ts.map