import _asyncToGenerator from "@babel/runtime/helpers/asyncToGenerator";
import _defineProperty from "@babel/runtime/helpers/defineProperty";
/*
Copyright 2019, 2021, 2023 The Matrix.org Foundation C.I.C.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

import { EventStatus, MatrixEventEvent } from "./event.js";
import { logger } from "../logger.js";
import { RelationType } from "../@types/event.js";
import { TypedEventEmitter } from "./typed-event-emitter.js";
import { Room } from "./room.js";
export var RelationsEvent = /*#__PURE__*/function (RelationsEvent) {
  RelationsEvent["Add"] = "Relations.add";
  RelationsEvent["Remove"] = "Relations.remove";
  RelationsEvent["Redaction"] = "Relations.redaction";
  return RelationsEvent;
}({});
var matchesEventType = function matchesEventType(eventType, targetEventType) {
  var altTargetEventTypes = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : [];
  return [targetEventType, ...altTargetEventTypes].includes(eventType);
};

/**
 * A container for relation events that supports easy access to common ways of
 * aggregating such events. Each instance holds events that of a single relation
 * type and event type. All of the events also relate to the same original event.
 *
 * The typical way to get one of these containers is via
 * EventTimelineSet#getRelationsForEvent.
 */
export class Relations extends TypedEventEmitter {
  /**
   * @param relationType - The type of relation involved, such as "m.annotation", "m.reference", "m.replace", etc.
   * @param eventType - The relation event's type, such as "m.reaction", etc.
   * @param client - The client which created this instance. For backwards compatibility also accepts a Room.
   * @param altEventTypes - alt event types for relation events, for example to support unstable prefixed event types
   */
  constructor(relationType, eventType, client, altEventTypes) {
    var _this;
    super();
    _this = this;
    this.relationType = relationType;
    this.eventType = eventType;
    this.altEventTypes = altEventTypes;
    _defineProperty(this, "relationEventIds", new Set());
    _defineProperty(this, "relations", new Set());
    _defineProperty(this, "annotationsByKey", {});
    _defineProperty(this, "annotationsBySender", {});
    _defineProperty(this, "sortedAnnotationsByKey", []);
    _defineProperty(this, "targetEvent", null);
    _defineProperty(this, "creationEmitted", false);
    _defineProperty(this, "client", void 0);
    /**
     * Listens for event status changes to remove cancelled events.
     *
     * @param event - The event whose status has changed
     * @param status - The new status
     */
    _defineProperty(this, "onEventStatus", (event, status) => {
      if (!event.isSending()) {
        // Sending is done, so we don't need to listen anymore
        event.removeListener(MatrixEventEvent.Status, this.onEventStatus);
        return;
      }
      if (status !== EventStatus.CANCELLED) {
        return;
      }
      // Event was cancelled, remove from the collection
      event.removeListener(MatrixEventEvent.Status, this.onEventStatus);
      this.removeEvent(event);
    });
    /**
     * For relations that have been redacted, we want to remove them from
     * aggregation data sets and emit an update event.
     *
     * To do so, we listen for `Event.beforeRedaction`, which happens:
     *   - after the server accepted the redaction and remote echoed back to us
     *   - before the original event has been marked redacted in the client
     *
     * @param redactedEvent - The original relation event that is about to be redacted.
     */
    _defineProperty(this, "onBeforeRedaction", /*#__PURE__*/function () {
      var _ref = _asyncToGenerator(function* (redactedEvent) {
        if (!_this.relations.has(redactedEvent)) {
          return;
        }
        _this.relations.delete(redactedEvent);
        if (_this.relationType === RelationType.Annotation) {
          // Remove the redacted annotation from aggregation by key
          _this.removeAnnotationFromAggregation(redactedEvent);
        } else if (_this.relationType === RelationType.Replace && _this.targetEvent && !_this.targetEvent.isState()) {
          var lastReplacement = yield _this.getLastReplacement();
          _this.targetEvent.makeReplaced(lastReplacement);
        }
        redactedEvent.removeListener(MatrixEventEvent.BeforeRedaction, _this.onBeforeRedaction);
        _this.emit(RelationsEvent.Redaction, redactedEvent);
      });
      return function (_x) {
        return _ref.apply(this, arguments);
      };
    }());
    this.client = client instanceof Room ? client.client : client;
  }

  /**
   * Add relation events to this collection.
   *
   * @param event - The new relation event to be added.
   */
  addEvent(event) {
    var _this2 = this;
    return _asyncToGenerator(function* () {
      if (_this2.relationEventIds.has(event.getId())) {
        return;
      }
      var relation = event.getRelation();
      if (!relation) {
        logger.error("Event must have relation info");
        return;
      }
      var relationType = relation.rel_type;
      var eventType = event.getType();
      if (_this2.relationType !== relationType || !matchesEventType(eventType, _this2.eventType, _this2.altEventTypes)) {
        logger.error("Event relation info doesn't match this container");
        return;
      }

      // If the event is in the process of being sent, listen for cancellation
      // so we can remove the event from the collection.
      if (event.isSending()) {
        event.on(MatrixEventEvent.Status, _this2.onEventStatus);
      }
      _this2.relations.add(event);
      _this2.relationEventIds.add(event.getId());
      if (_this2.relationType === RelationType.Annotation) {
        _this2.addAnnotationToAggregation(event);
      } else if (_this2.relationType === RelationType.Replace && _this2.targetEvent && !_this2.targetEvent.isState()) {
        var lastReplacement = yield _this2.getLastReplacement();
        _this2.targetEvent.makeReplaced(lastReplacement);
      }
      event.on(MatrixEventEvent.BeforeRedaction, _this2.onBeforeRedaction);
      _this2.emit(RelationsEvent.Add, event);
      _this2.maybeEmitCreated();
    })();
  }

  /**
   * Remove relation event from this collection.
   *
   * @param event - The relation event to remove.
   */
  removeEvent(event) {
    var _this3 = this;
    return _asyncToGenerator(function* () {
      if (!_this3.relations.has(event)) {
        return;
      }
      _this3.relations.delete(event);
      if (_this3.relationType === RelationType.Annotation) {
        _this3.removeAnnotationFromAggregation(event);
      } else if (_this3.relationType === RelationType.Replace && _this3.targetEvent && !_this3.targetEvent.isState()) {
        var lastReplacement = yield _this3.getLastReplacement();
        _this3.targetEvent.makeReplaced(lastReplacement);
      }
      _this3.emit(RelationsEvent.Remove, event);
    })();
  }
  /**
   * Get all relation events in this collection.
   *
   * These are currently in the order of insertion to this collection, which
   * won't match timeline order in the case of scrollback.
   * TODO: Tweak `addEvent` to insert correctly for scrollback.
   *
   * Relation events in insertion order.
   */
  getRelations() {
    return [...this.relations];
  }
  addAnnotationToAggregation(event) {
    var _event$getRelation;
    var {
      key
    } = (_event$getRelation = event.getRelation()) !== null && _event$getRelation !== void 0 ? _event$getRelation : {};
    if (!key) return;
    var eventsForKey = this.annotationsByKey[key];
    if (!eventsForKey) {
      eventsForKey = this.annotationsByKey[key] = new Set();
      this.sortedAnnotationsByKey.push([key, eventsForKey]);
    }
    // Add the new event to the set for this key
    eventsForKey.add(event);
    // Re-sort the [key, events] pairs in descending order of event count
    this.sortedAnnotationsByKey.sort((a, b) => {
      var aEvents = a[1];
      var bEvents = b[1];
      return bEvents.size - aEvents.size;
    });
    var sender = event.getSender();
    var eventsFromSender = this.annotationsBySender[sender];
    if (!eventsFromSender) {
      eventsFromSender = this.annotationsBySender[sender] = new Set();
    }
    // Add the new event to the set for this sender
    eventsFromSender.add(event);
  }
  removeAnnotationFromAggregation(event) {
    var _event$getRelation2;
    var {
      key
    } = (_event$getRelation2 = event.getRelation()) !== null && _event$getRelation2 !== void 0 ? _event$getRelation2 : {};
    if (!key) return;
    var eventsForKey = this.annotationsByKey[key];
    if (eventsForKey) {
      eventsForKey.delete(event);

      // Re-sort the [key, events] pairs in descending order of event count
      this.sortedAnnotationsByKey.sort((a, b) => {
        var aEvents = a[1];
        var bEvents = b[1];
        return bEvents.size - aEvents.size;
      });
    }
    var sender = event.getSender();
    var eventsFromSender = this.annotationsBySender[sender];
    if (eventsFromSender) {
      eventsFromSender.delete(event);
    }
  }
  /**
   * Get all events in this collection grouped by key and sorted by descending
   * event count in each group.
   *
   * This is currently only supported for the annotation relation type.
   *
   * An array of [key, events] pairs sorted by descending event count.
   * The events are stored in a Set (which preserves insertion order).
   */
  getSortedAnnotationsByKey() {
    if (this.relationType !== RelationType.Annotation) {
      // Other relation types are not grouped currently.
      return null;
    }
    return this.sortedAnnotationsByKey;
  }

  /**
   * Get all events in this collection grouped by sender.
   *
   * This is currently only supported for the annotation relation type.
   *
   * An object with each relation sender as a key and the matching Set of
   * events for that sender as a value.
   */
  getAnnotationsBySender() {
    if (this.relationType !== RelationType.Annotation) {
      // Other relation types are not grouped currently.
      return null;
    }
    return this.annotationsBySender;
  }

  /**
   * Returns the most recent (and allowed) m.replace relation, if any.
   *
   * This is currently only supported for the m.replace relation type,
   * once the target event is known, see `addEvent`.
   */
  getLastReplacement() {
    var _this4 = this;
    return _asyncToGenerator(function* () {
      if (_this4.relationType !== RelationType.Replace) {
        // Aggregating on last only makes sense for this relation type
        return null;
      }
      if (!_this4.targetEvent) {
        // Don't know which replacements to accept yet.
        // This method shouldn't be called before the original
        // event is known anyway.
        return null;
      }

      // the all-knowning server tells us that the event at some point had
      // this timestamp for its replacement, so any following replacement should definitely not be less
      var replaceRelation = _this4.targetEvent.getServerAggregatedRelation(RelationType.Replace);
      var minTs = replaceRelation === null || replaceRelation === void 0 ? void 0 : replaceRelation.origin_server_ts;
      var lastReplacement = _this4.getRelations().reduce((last, event) => {
        if (event.getSender() !== _this4.targetEvent.getSender()) {
          return last;
        }
        if (minTs && minTs > event.getTs()) {
          return last;
        }
        if (last && last.getTs() > event.getTs()) {
          return last;
        }
        return event;
      }, null);
      if (lastReplacement !== null && lastReplacement !== void 0 && lastReplacement.shouldAttemptDecryption() && _this4.client.getCrypto()) {
        // Dirty but we are expecting to pass the cryptoBackend which is not accessible here
        yield lastReplacement.attemptDecryption(_this4.client.getCrypto());
      } else if (lastReplacement !== null && lastReplacement !== void 0 && lastReplacement.isBeingDecrypted()) {
        yield lastReplacement.getDecryptionPromise();
      }
      return lastReplacement;
    })();
  }

  /*
   * @param targetEvent - the event the relations are related to.
   */
  setTargetEvent(event) {
    var _this5 = this;
    return _asyncToGenerator(function* () {
      if (_this5.targetEvent) {
        return;
      }
      _this5.targetEvent = event;
      if (_this5.relationType === RelationType.Replace && !_this5.targetEvent.isState()) {
        var replacement = yield _this5.getLastReplacement();
        // this is the initial update, so only call it if we already have something
        // to not emit Event.replaced needlessly
        if (replacement) {
          _this5.targetEvent.makeReplaced(replacement);
        }
      }
      _this5.maybeEmitCreated();
    })();
  }
  maybeEmitCreated() {
    if (this.creationEmitted) {
      return;
    }
    // Only emit we're "created" once we have a target event instance _and_
    // at least one related event.
    if (!this.targetEvent || !this.relations.size) {
      return;
    }
    this.creationEmitted = true;
    this.targetEvent.emit(MatrixEventEvent.RelationsCreated, this.relationType, this.eventType);
  }
}
//# sourceMappingURL=relations.js.map