/**
 * This is an internal module. See {@link MatrixEvent} and {@link RoomEvent} for
 * the public classes.
 */
import { type ExtensibleEvent, type Optional } from "matrix-events-sdk";
import { EventType, type MsgType, RelationType, UNSIGNED_THREAD_ID_FIELD } from "../@types/event.ts";
import { type RoomMember } from "./room-member.ts";
import { type Thread, ThreadEvent, type ThreadEventHandlerMap } from "./thread.ts";
import { type IActionsObject } from "../pushprocessor.ts";
import { type MatrixError } from "../http-api/index.ts";
import { TypedEventEmitter } from "./typed-event-emitter.ts";
import { type EventStatus } from "./event-status.ts";
import { type CryptoBackend } from "../common-crypto/CryptoBackend.ts";
import { type IAnnotatedPushRule } from "../@types/PushRules.ts";
import { type Room } from "./room.ts";
import { type Membership } from "../@types/membership.ts";
import { DecryptionFailureCode } from "../crypto-api/index.ts";
import { type RoomState } from "./room-state.ts";
import { type EmptyObject } from "../@types/common.ts";
export { EventStatus } from "./event-status.ts";
export interface IContent {
    [key: string]: any;
    "msgtype"?: MsgType | string;
    "membership"?: Membership;
    "avatar_url"?: string;
    "displayname"?: string;
    "m.relates_to"?: IEventRelation;
    "m.mentions"?: IMentions;
}
type StrippedState = Required<Pick<IEvent, "content" | "state_key" | "type" | "sender">>;
export interface IUnsigned {
    [key: string]: any;
    "age"?: number;
    "prev_sender"?: string;
    "prev_content"?: IContent;
    "redacted_because"?: IEvent;
    "replaces_state"?: string;
    "transaction_id"?: string;
    "invite_room_state"?: StrippedState[];
    "m.relations"?: Record<RelationType | string, any>;
    "msc4354_sticky_duration_ttl_ms"?: number;
    [UNSIGNED_THREAD_ID_FIELD.name]?: string;
}
export interface IThreadBundledRelationship {
    latest_event: IEvent;
    count: number;
    current_user_participated?: boolean;
}
export interface IEvent {
    event_id: string;
    type: string;
    content: IContent;
    sender: string;
    room_id?: string;
    origin_server_ts: number;
    txn_id?: string;
    state_key?: string;
    membership?: Membership;
    unsigned: IUnsigned;
    redacts?: string;
    msc4354_sticky?: {
        duration_ms: number;
    };
}
export interface IAggregatedRelation {
    origin_server_ts: number;
    event_id?: string;
    sender?: string;
    type?: string;
    count?: number;
    key?: string;
}
export interface IEventRelation {
    "rel_type"?: RelationType | string;
    "event_id"?: string;
    "is_falling_back"?: boolean;
    "m.in_reply_to"?: {
        event_id?: string;
    };
    "key"?: string;
}
export interface IMentions {
    user_ids?: string[];
    room?: boolean;
}
export interface PushDetails {
    rule?: IAnnotatedPushRule;
    actions?: IActionsObject;
}
/**
 * When an event is a visibility change event, as per MSC3531,
 * the visibility change implied by the event.
 */
export interface IVisibilityChange {
    /**
     * If `true`, the target event should be made visible.
     * Otherwise, it should be hidden.
     */
    visible: boolean;
    /**
     * The event id affected.
     */
    eventId: string;
    /**
     * Optionally, a human-readable reason explaining why
     * the event was hidden. Ignored if the event was made
     * visible.
     */
    reason: string | null;
}
export interface IMarkedUnreadEvent {
    unread: boolean;
}
export interface IClearEvent {
    room_id?: string;
    type: string;
    state_key?: string;
    content: Omit<IContent, "membership" | "avatar_url" | "displayname" | "m.relates_to">;
    unsigned?: IUnsigned;
}
interface IKeyRequestRecipient {
    userId: string;
    deviceId: "*" | string;
}
export interface IDecryptOptions {
    /** Whether to emit {@link MatrixEventEvent.Decrypted} events on successful decryption. Defaults to true.
     */
    emit?: boolean;
    /**
     * True if this is a retry, after receiving an update to the session key. (Enables more logging.)
     *
     * This is only intended for use within the js-sdk.
     *
     * @internal
     */
    isRetry?: boolean;
    /**
     * Whether the message should be re-decrypted if it was previously successfully decrypted with an untrusted key.
     * Defaults to `false`.
     */
    forceRedecryptIfUntrusted?: boolean;
}
/**
 * Message hiding, as specified by https://github.com/matrix-org/matrix-doc/pull/3531.
 */
export type MessageVisibility = IMessageVisibilityHidden | IMessageVisibilityVisible;
/**
 * Variant of `MessageVisibility` for the case in which the message should be displayed.
 */
export interface IMessageVisibilityVisible {
    readonly visible: true;
}
/**
 * Variant of `MessageVisibility` for the case in which the message should be hidden.
 */
export interface IMessageVisibilityHidden {
    readonly visible: false;
    /**
     * Optionally, a human-readable reason to show to the user indicating why the
     * message has been hidden (e.g. "Message Pending Moderation").
     */
    readonly reason: string | null;
}
export declare const MAX_STICKY_DURATION_MS = 3600000;
export declare enum MatrixEventEvent {
    /**
     * An event has been decrypted, or we have failed to decrypt it.
     *
     * The payload consists of:
     *
     *  * `event` - The {@link MatrixEvent} which we attempted to decrypt.
     *
     *  * `err` - The error that occurred during decryption, or `undefined` if no error occurred.
     *     Avoid use of this: {@link MatrixEvent.decryptionFailureReason} is more useful.
     */
    Decrypted = "Event.decrypted",
    BeforeRedaction = "Event.beforeRedaction",
    VisibilityChange = "Event.visibilityChange",
    LocalEventIdReplaced = "Event.localEventIdReplaced",
    Status = "Event.status",
    Replaced = "Event.replaced",
    RelationsCreated = "Event.relationsCreated",
    SentinelUpdated = "Event.sentinelUpdated"
}
export type MatrixEventEmittedEvents = MatrixEventEvent | ThreadEvent.Update;
export type MatrixEventHandlerMap = {
    [MatrixEventEvent.Decrypted]: (event: MatrixEvent, err?: Error) => void;
    [MatrixEventEvent.BeforeRedaction]: (event: MatrixEvent, redactionEvent: MatrixEvent) => void;
    [MatrixEventEvent.VisibilityChange]: (event: MatrixEvent, visible: boolean) => void;
    [MatrixEventEvent.LocalEventIdReplaced]: (event: MatrixEvent) => void;
    [MatrixEventEvent.Status]: (event: MatrixEvent, status: EventStatus | null) => void;
    [MatrixEventEvent.Replaced]: (event: MatrixEvent) => void;
    [MatrixEventEvent.RelationsCreated]: (relationType: string, eventType: string) => void;
    [MatrixEventEvent.SentinelUpdated]: () => void;
} & Pick<ThreadEventHandlerMap, ThreadEvent.Update>;
export declare class MatrixEvent extends TypedEventEmitter<MatrixEventEmittedEvents, MatrixEventHandlerMap> {
    event: Partial<IEvent>;
    private pushDetails;
    private _replacingEvent;
    private _localRedactionEvent;
    private _isCancelled;
    private clearEvent?;
    private visibility;
    private _hasCachedExtEv;
    private _cachedExtEv;
    /** If we failed to decrypt this event, the reason for the failure. Otherwise, `null`. */
    private _decryptionFailureReason;
    private senderCurve25519Key;
    private claimedEd25519Key;
    private forwardingCurve25519KeyChain;
    private untrusted;
    private decryptionPromise;
    private retryDecryption;
    private txnId?;
    /**
     * A reference to the thread this event belongs to
     */
    private thread?;
    private threadId?;
    localTimestamp: number;
    /**
     * The room member who sent this event, or null e.g.
     * this is a presence event. This is only guaranteed to be set for events that
     * appear in a timeline, ie. do not guarantee that it will be set on state
     * events.
     * @privateRemarks
     * Should be read-only
     */
    sender: RoomMember | null;
    /**
     * The room member who is the target of this event, e.g.
     * the invitee, the person being banned, etc.
     * @privateRemarks
     * Should be read-only
     */
    target: RoomMember | null;
    /**
     * Update the sentinels and forwardLooking flag for this event.
     * @param stateContext -  the room state to be queried
     * @param toStartOfTimeline -  if true the event's forwardLooking flag is set false
     * @internal
     */
    setMetadata(stateContext: RoomState, toStartOfTimeline: boolean): void;
    /**
     * The sending status of the event.
     * @privateRemarks
     * Should be read-only
     */
    status: EventStatus | null;
    /**
     * most recent error associated with sending the event, if any
     * @privateRemarks
     * Should be read-only. May not be a MatrixError.
     */
    error: MatrixError | null;
    /**
     * True if this event is 'forward looking', meaning
     * that getDirectionalContent() will return event.content and not event.prev_content.
     * Only state events may be backwards looking
     * Default: true. <strong>This property is experimental and may change.</strong>
     * @privateRemarks
     * Should be read-only
     */
    forwardLooking: boolean;
    private readonly reEmitter;
    /**
     * The timestamp for when this event should expire, in milliseconds.
     * Prefers using the server-provided value, but will fall back to local calculation.
     *
     * This value is **safe** to use, as malicious start time and duration are appropriately capped.
     *
     * If the event is not a sticky event (or not supported by the server),
     * then this returns `undefined`.
     */
    readonly unstableStickyExpiresAt: number | undefined;
    /**
     * Construct a Matrix Event object
     *
     * @param event - The raw (possibly encrypted) event. <b>Do not access
     * this property</b> directly unless you absolutely have to. Prefer the getter
     * methods defined on this class. Using the getter methods shields your app
     * from changes to event JSON between Matrix versions.
     */
    constructor(event?: Partial<IEvent>);
    /**
     * Unstable getter to try and get an extensible event. Note that this might
     * return a falsy value if the event could not be parsed as an extensible
     * event.
     *
     * @deprecated Use stable functions where possible.
     */
    get unstableExtensibleEvent(): Optional<ExtensibleEvent>;
    private invalidateExtensibleEvent;
    /**
     * Gets the event as it would appear if it had been sent unencrypted.
     *
     * If the event is encrypted, we attempt to mock up an event as it would have looked had the sender not encrypted it.
     * If the event is not encrypted, a copy of it is simply returned as-is.
     *
     * @returns A shallow copy of the event, in wire format, as it would have been had it not been encrypted.
     */
    getEffectiveEvent(): IEvent;
    /**
     * Get the event_id for this event.
     * @returns The event ID, e.g. <code>$143350589368169JsLZx:localhost
     * </code>
     */
    getId(): string | undefined;
    /**
     * Get the user_id for this event.
     * @returns The user ID, e.g. `@alice:matrix.org`
     */
    getSender(): string | undefined;
    /**
     * Get the (decrypted, if necessary) type of event.
     *
     * @returns The event type, e.g. `m.room.message`
     */
    getType(): EventType | string;
    /**
     * Get the (possibly encrypted) type of the event that will be sent to the
     * homeserver.
     *
     * @returns The event type.
     */
    getWireType(): EventType | string;
    /**
     * Get the room_id for this event. This will return `undefined`
     * for `m.presence` events.
     * @returns The room ID, e.g. <code>!cURbafjkfsMDVwdRDQ:matrix.org
     * </code>
     */
    getRoomId(): string | undefined;
    /**
     * Get the timestamp of this event.
     * @returns The event timestamp, e.g. `1433502692297`
     */
    getTs(): number;
    /**
     * Get the timestamp of this event, as a Date object.
     * @returns The event date, e.g. `new Date(1433502692297)`
     */
    getDate(): Date | null;
    /**
     * Get a string containing details of this event
     *
     * This is intended for logging, to help trace errors. Example output:
     *
     * @example
     * ```
     * id=$HjnOHV646n0SjLDAqFrgIjim7RCpB7cdMXFrekWYAn type=m.room.encrypted
     * sender=@user:example.com room=!room:example.com ts=2022-10-25T17:30:28.404Z
     * ```
     */
    getDetails(): string;
    /**
     * Get the (decrypted, if necessary) event content JSON, even if the event
     * was replaced by another event.
     *
     * @returns The event content JSON, or an empty object.
     */
    getOriginalContent<T = IContent>(): T;
    /**
     * Get the (decrypted, if necessary) event content JSON,
     * or the content from the replacing event, if any.
     * See `makeReplaced`.
     *
     * @returns The event content JSON, or an empty object.
     */
    getContent<T extends IContent = IContent>(): T;
    /**
     * Get the (possibly encrypted) event content JSON that will be sent to the
     * homeserver.
     *
     * @returns The event content JSON, or an empty object.
     */
    getWireContent(): IContent;
    /**
     * Get the event ID of the thread head
     */
    get threadRootId(): string | undefined;
    /**
     * A helper to check if an event is a thread's head or not
     */
    get isThreadRoot(): boolean;
    get replyEventId(): string | undefined;
    get relationEventId(): string | undefined;
    /**
     * Get the previous event content JSON. This will only return something for
     * state events which exist in the timeline.
     * @returns The previous event content JSON, or an empty object.
     */
    getPrevContent(): IContent;
    /**
     * Get either 'content' or 'prev_content' depending on if this event is
     * 'forward-looking' or not. This can be modified via event.forwardLooking.
     * In practice, this means we get the chronologically earlier content value
     * for this event (this method should surely be called getEarlierContent)
     * <strong>This method is experimental and may change.</strong>
     * @returns event.content if this event is forward-looking, else
     * event.prev_content.
     */
    getDirectionalContent(): IContent;
    /**
     * Get the age of this event. This represents the age of the event when the
     * event arrived at the device, and not the age of the event when this
     * function was called.
     * Can only be returned once the server has echo'ed back
     * @returns The age of this event in milliseconds.
     */
    getAge(): number | undefined;
    /**
     * Get the age of the event when this function was called.
     * This is the 'age' field adjusted according to how long this client has
     * had the event.
     * @returns The age of this event in milliseconds.
     */
    getLocalAge(): number;
    /**
     * Get the event state_key if it has one. If necessary, this will perform
     * string-unpacking on the state key, as per MSC3414. This will return
     * <code>undefined</code> for message events.
     * @returns The event's `state_key`.
     */
    getStateKey(): string | undefined;
    /**
     * Get the raw event state_key if it has one. This may be string-packed as per
     * MSC3414 if the state event is encrypted. This will return <code>undefined
     * </code> for message events.
     * @returns The event's `state_key`.
     */
    getWireStateKey(): string | undefined;
    /**
     * Check if this event is a state event.
     * @returns True if this is a state event.
     */
    isState(): boolean;
    /**
     * Get the user's room membership at the time the event was sent, as reported
     * by the server.  This uses MSC4115.
     *
     * @returns The user's room membership, or `undefined` if the server does
     *   not report it.
     */
    getMembershipAtEvent(): Optional<Membership | string>;
    /**
     * Replace the content of this event with encrypted versions.
     * (This is used when sending an event; it should not be used by applications).
     *
     * @internal
     *
     * @param cryptoType - type of the encrypted event - typically
     * <tt>"m.room.encrypted"</tt>
     *
     * @param cryptoContent - raw 'content' for the encrypted event.
     *
     * @param senderCurve25519Key - curve25519 key to record for the
     *   sender of this event.
     *   See {@link MatrixEvent#getSenderKey}.
     *
     * @param claimedEd25519Key - claimed ed25519 key to record for the
     *   sender if this event.
     *   See {@link MatrixEvent#getClaimedEd25519Key}
     */
    makeEncrypted(cryptoType: string, cryptoContent: object, senderCurve25519Key: string, claimedEd25519Key: string): void;
    /**
     * Check if this event is currently being decrypted.
     *
     * @returns True if this event is currently being decrypted, else false.
     */
    isBeingDecrypted(): boolean;
    getDecryptionPromise(): Promise<void> | null;
    /**
     * Check if this event is an encrypted event which we failed to decrypt
     *
     * (This implies that we might retry decryption at some point in the future)
     *
     * @returns True if this event is an encrypted event which we
     *     couldn't decrypt.
     */
    isDecryptionFailure(): boolean;
    /** If we failed to decrypt this event, the reason for the failure. Otherwise, `null`. */
    get decryptionFailureReason(): DecryptionFailureCode | null;
    shouldAttemptDecryption(): boolean;
    /**
     * Start the process of trying to decrypt this event.
     *
     * (This is used within the SDK: it isn't intended for use by applications)
     *
     * @internal
     *
     * @param crypto - crypto module
     *
     * @returns promise which resolves (to undefined) when the decryption
     * attempt is completed.
     */
    attemptDecryption(crypto: CryptoBackend, options?: IDecryptOptions): Promise<void>;
    /**
     * Calculate the recipients for keyshare requests.
     *
     * @param userId - the user who received this event.
     *
     * @returns array of recipients
     */
    getKeyRequestRecipients(userId: string): IKeyRequestRecipient[];
    private decryptionLoop;
    /**
     * Update the cleartext data on this event.
     *
     * (This is used after decrypting an event; it should not be used by applications).
     *
     * @internal
     *
     * @param decryptionResult - the decryption result, including the plaintext and some key info
     */
    private setClearData;
    /**
     * Update the cleartext data on this event after a decryption failure.
     *
     * @param reason - the textual reason for the failure
     */
    private setClearDataForDecryptionFailure;
    /**
     * Gets the cleartext content for this event. If the event is not encrypted,
     * or encryption has not been completed, this will return null.
     *
     * @returns The cleartext (decrypted) content for the event
     */
    getClearContent(): IContent | null;
    /**
     * Check if the event is encrypted.
     * @returns True if this event is encrypted.
     */
    isEncrypted(): boolean;
    /**
     * The curve25519 key for the device that we think sent this event
     *
     * For an Olm-encrypted event, this is inferred directly from the DH
     * exchange at the start of the session: the curve25519 key is involved in
     * the DH exchange, so only a device which holds the private part of that
     * key can establish such a session.
     *
     * For a megolm-encrypted event, it is inferred from the Olm message which
     * established the megolm session
     */
    getSenderKey(): string | null;
    /**
     * The additional keys the sender of this encrypted event claims to possess.
     *
     * Just a wrapper for #getClaimedEd25519Key (q.v.)
     */
    getKeysClaimed(): Partial<Record<"ed25519", string>>;
    /**
     * Get the ed25519 the sender of this event claims to own.
     *
     * For Olm messages, this claim is encoded directly in the plaintext of the
     * event itself. For megolm messages, it is implied by the m.room_key event
     * which established the megolm session.
     *
     * Until we download the device list of the sender, it's just a claim: the
     * device list gives a proof that the owner of the curve25519 key used for
     * this event (and returned by #getSenderKey) also owns the ed25519 key by
     * signing the public curve25519 key with the ed25519 key.
     *
     * In general, applications should not use this method directly, but should
     * instead use {@link crypto-api!CryptoApi#getEncryptionInfoForEvent}.
     */
    getClaimedEd25519Key(): string | null;
    /**
     * Get the curve25519 keys of the devices which were involved in telling us
     * about the claimedEd25519Key and sender curve25519 key.
     *
     * Normally this will be empty, but in the case of a forwarded megolm
     * session, the sender keys are sent to us by another device (the forwarding
     * device), which we need to trust to do this. In that case, the result will
     * be a list consisting of one entry.
     *
     * If the device that sent us the key (A) got it from another device which
     * it wasn't prepared to vouch for (B), the result will be [A, B]. And so on.
     *
     * @returns base64-encoded curve25519 keys, from oldest to newest.
     */
    getForwardingCurve25519KeyChain(): string[];
    /**
     * Whether the decryption key was obtained from an untrusted source. If so,
     * we cannot verify the authenticity of the message.
     */
    isKeySourceUntrusted(): boolean | undefined;
    getUnsigned(): IUnsigned;
    setUnsigned(unsigned: IUnsigned): void;
    unmarkLocallyRedacted(): boolean;
    markLocallyRedacted(redactionEvent: MatrixEvent): void;
    /**
     * Change the visibility of an event, as per https://github.com/matrix-org/matrix-doc/pull/3531 .
     *
     * @param visibilityChange - event holding a hide/unhide payload, or nothing
     *   if the event is being reset to its original visibility (presumably
     *   by a visibility event being redacted).
     *
     * @remarks
     * Fires {@link MatrixEventEvent.VisibilityChange} if `visibilityEvent`
     *   caused a change in the actual visibility of this event, either by making it
     *   visible (if it was hidden), by making it hidden (if it was visible) or by
     *   changing the reason (if it was hidden).
     */
    applyVisibilityEvent(visibilityChange?: IVisibilityChange): void;
    /**
     * Return instructions to display or hide the message.
     *
     * @returns Instructions determining whether the message
     * should be displayed.
     */
    messageVisibility(): MessageVisibility;
    /**
     * Update the content of an event in the same way it would be by the server
     * if it were redacted before it was sent to us
     *
     * @param redactionEvent - event causing the redaction
     * @param room - the room in which the event exists
     */
    makeRedacted(redactionEvent: MatrixEvent, room: Room): void;
    private moveAllRelatedToMainTimeline;
    private moveToMainTimeline;
    /**
     * Check if this event has been redacted
     *
     * @returns True if this event has been redacted
     */
    isRedacted(): boolean;
    /**
     * Check if this event is a redaction of another event
     *
     * @returns True if this event is a redaction
     */
    isRedaction(): boolean;
    /**
     * Return the visibility change caused by this event,
     * as per https://github.com/matrix-org/matrix-doc/pull/3531.
     *
     * @returns If the event is a well-formed visibility change event,
     * an instance of `IVisibilityChange`, otherwise `null`.
     */
    asVisibilityChange(): IVisibilityChange | null;
    /**
     * Check if this event alters the visibility of another event,
     * as per https://github.com/matrix-org/matrix-doc/pull/3531.
     *
     * @returns True if this event alters the visibility
     * of another event.
     */
    isVisibilityEvent(): boolean;
    /**
     * Get the (decrypted, if necessary) redaction event JSON
     * if event was redacted
     *
     * @returns The redaction event JSON, or an empty object
     */
    getRedactionEvent(): IEvent | EmptyObject | null;
    /**
     * Get the push actions, if known, for this event
     *
     * @returns push actions
     */
    getPushActions(): IActionsObject | null;
    /**
     * Get the push details, if known, for this event
     *
     * @returns push actions
     */
    getPushDetails(): PushDetails;
    /**
     * Set the push details for this event.
     *
     * @param pushActions - push actions
     * @param rule - the executed push rule
     */
    setPushDetails(pushActions?: IActionsObject, rule?: IAnnotatedPushRule): void;
    /**
     * Replace the `event` property and recalculate any properties based on it.
     * @param event - the object to assign to the `event` property
     */
    handleRemoteEcho(event: object): void;
    /**
     * Whether the event is in any phase of sending, send failure, waiting for
     * remote echo, etc.
     */
    isSending(): boolean;
    /**
     * Update the event's sending status and emit an event as well.
     *
     * @param status - The new status
     */
    setStatus(status: EventStatus | null): void;
    replaceLocalEventId(eventId: string): void;
    /**
     * Get whether the event is a relation event, and of a given type if
     * `relType` is passed in. State events cannot be relation events
     *
     * @param relType - if given, checks that the relation is of the
     * given type
     */
    isRelation(relType?: string): boolean;
    /**
     * Get relation info for the event, if any.
     */
    getRelation(): IEventRelation | null;
    /**
     * Set an event that replaces the content of this event, through an m.replace relation.
     *
     * @param newEvent - the event with the replacing content, if any.
     *
     * @remarks
     * Fires {@link MatrixEventEvent.Replaced}
     */
    makeReplaced(newEvent?: MatrixEvent): void;
    /**
     * Returns the status of any associated edit or redaction
     * (not for reactions/annotations as their local echo doesn't affect the original event),
     * or else the status of the event.
     */
    getAssociatedStatus(): EventStatus | null;
    getServerAggregatedRelation<T>(relType: RelationType | string): T | undefined;
    /**
     * Returns the event ID of the event replacing the content of this event, if any.
     */
    replacingEventId(): string | undefined;
    /**
     * Returns the event replacing the content of this event, if any.
     * Replacements are aggregated on the server, so this would only
     * return an event in case it came down the sync, or for local echo of edits.
     */
    replacingEvent(): MatrixEvent | null;
    /**
     * Returns the origin_server_ts of the event replacing the content of this event, if any.
     */
    replacingEventDate(): Date | undefined;
    /**
     * Returns the event that wants to redact this event, but hasn't been sent yet.
     * @returns the event
     */
    localRedactionEvent(): MatrixEvent | null;
    /**
     * For relations and redactions, returns the event_id this event is referring to.
     */
    getAssociatedId(): string | undefined;
    /**
     * Checks if this event is associated with another event. See `getAssociatedId`.
     */
    hasAssociation(): boolean;
    /**
     * Update the related id with a new one.
     *
     * Used to replace a local id with remote one before sending
     * an event with a related id.
     *
     * @param eventId - the new event id
     */
    updateAssociatedId(eventId: string): void;
    /**
     * Flags an event as cancelled due to future conditions. For example, a verification
     * request event in the same sync transaction may be flagged as cancelled to warn
     * listeners that a cancellation event is coming down the same pipe shortly.
     * @param cancelled - Whether the event is to be cancelled or not.
     */
    flagCancelled(cancelled?: boolean): void;
    /**
     * Gets whether or not the event is flagged as cancelled. See flagCancelled() for
     * more information.
     * @returns True if the event is cancelled, false otherwise.
     */
    isCancelled(): boolean;
    /**
     * Get a copy/snapshot of this event. The returned copy will be loosely linked
     * back to this instance, though will have "frozen" event information. Other
     * properties of this MatrixEvent instance will be copied verbatim, which can
     * mean they are in reference to this instance despite being on the copy too.
     * The reference the snapshot uses does not change, however members aside from
     * the underlying event will not be deeply cloned, thus may be mutated internally.
     * For example, the sender profile will be copied over at snapshot time, and
     * the sender profile internally may mutate without notice to the consumer.
     *
     * This is meant to be used to snapshot the event details themselves, not the
     * features (such as sender) surrounding the event.
     * @returns A snapshot of this event.
     */
    toSnapshot(): MatrixEvent;
    /**
     * Determines if this event is equivalent to the given event. This only checks
     * the event object itself, not the other properties of the event. Intended for
     * use with toSnapshot() to identify events changing.
     * @param otherEvent - The other event to check against.
     * @returns True if the events are the same, false otherwise.
     */
    isEquivalentTo(otherEvent?: MatrixEvent): boolean;
    /**
     * Summarise the event as JSON.
     *
     * If encrypted, include both the decrypted and encrypted view of the event.
     *
     * This is named `toJSON` for use with `JSON.stringify` which checks objects
     * for functions named `toJSON` and will call them to customise the output
     * if they are defined.
     *
     * **WARNING** Do not log the result of this method; otherwise, it will end up
     * in rageshakes, leading to a privacy violation.
     *
     * @deprecated Prefer to use {@link MatrixEvent#getEffectiveEvent} or similar.
     * This method will be removed soon; it is too easy to use it accidentally
     * and cause a privacy violation (cf https://github.com/vector-im/element-web/issues/26380).
     * In any case, the value it returns is not a faithful serialization of the object.
     */
    toJSON(): object;
    setTxnId(txnId: string): void;
    getTxnId(): string | undefined;
    /**
     * Set the instance of a thread associated with the current event
     * @param thread - the thread
     */
    setThread(thread?: Thread): void;
    /**
     * Get the instance of the thread associated with the current event
     */
    getThread(): Thread | undefined;
    setThreadId(threadId?: string): void;
    /**
     * Unstable getter to try and get the sticky information for the event.
     * If the event is not a sticky event (or not supported by the server),
     * then this returns `undefined`.
     *
     * `duration_ms` is safely bounded to a hour.
     */
    get unstableStickyInfo(): {
        duration_ms: number;
        duration_ttl_ms?: number;
    } | undefined;
}
//# sourceMappingURL=event.d.ts.map