import _asyncToGenerator from "@babel/runtime/helpers/asyncToGenerator";
import _defineProperty from "@babel/runtime/helpers/defineProperty";
function ownKeys(e, r) { var t = Object.keys(e); if (Object.getOwnPropertySymbols) { var o = Object.getOwnPropertySymbols(e); r && (o = o.filter(function (r) { return Object.getOwnPropertyDescriptor(e, r).enumerable; })), t.push.apply(t, o); } return t; }
function _objectSpread(e) { for (var r = 1; r < arguments.length; r++) { var t = null != arguments[r] ? arguments[r] : {}; r % 2 ? ownKeys(Object(t), !0).forEach(function (r) { _defineProperty(e, r, t[r]); }) : Object.getOwnPropertyDescriptors ? Object.defineProperties(e, Object.getOwnPropertyDescriptors(t)) : ownKeys(Object(t)).forEach(function (r) { Object.defineProperty(e, r, Object.getOwnPropertyDescriptor(t, r)); }); } return e; }
/*
Copyright 2021 The Matrix.org Foundation C.I.C.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

import { EventType, MsgType, UNSTABLE_MSC3089_BRANCH, UNSTABLE_MSC3089_LEAF } from "../@types/event.js";
import { logger } from "../logger.js";
import { averageBetweenStrings, DEFAULT_ALPHABET, lexicographicCompare, nextString, prevString, simpleRetryOperation } from "../utils.js";
import { MSC3089Branch } from "./MSC3089Branch.js";
import { MatrixError } from "../http-api/index.js";
import { KnownMembership } from "../@types/membership.js";
/**
 * The recommended defaults for a tree space's power levels. Note that this
 * is UNSTABLE and subject to breaking changes without notice.
 */
export var DEFAULT_TREE_POWER_LEVELS_TEMPLATE = {
  // Owner
  invite: 100,
  kick: 100,
  ban: 100,
  // Editor
  redact: 50,
  state_default: 50,
  events_default: 50,
  // Viewer
  users_default: 0,
  // Mixed
  events: {
    [EventType.RoomPowerLevels]: 100,
    [EventType.RoomHistoryVisibility]: 100,
    [EventType.RoomTombstone]: 100,
    [EventType.RoomEncryption]: 100,
    [EventType.RoomName]: 50,
    [EventType.RoomMessage]: 50,
    [EventType.RoomMessageEncrypted]: 50,
    [EventType.Sticker]: 50
  },
  users: {} // defined by calling code
};

/**
 * Ease-of-use representation for power levels represented as simple roles.
 * Note that this is UNSTABLE and subject to breaking changes without notice.
 */
export var TreePermissions = /*#__PURE__*/function (TreePermissions) {
  TreePermissions["Viewer"] = "viewer";
  // Default
  TreePermissions["Editor"] = "editor";
  // "Moderator" or ~PL50
  TreePermissions["Owner"] = "owner"; // "Admin" or PL100
  return TreePermissions;
}({});
/**
 * Represents a [MSC3089](https://github.com/matrix-org/matrix-doc/pull/3089)
 * file tree Space. Note that this is UNSTABLE and subject to breaking changes
 * without notice.
 */
export class MSC3089TreeSpace {
  constructor(client, roomId) {
    this.client = client;
    this.roomId = roomId;
    _defineProperty(this, "room", void 0);
    this.room = this.client.getRoom(this.roomId);
    if (!this.room) throw new Error("Unknown room");
  }

  /**
   * Syntactic sugar for room ID of the Space.
   */
  get id() {
    return this.roomId;
  }

  /**
   * Whether or not this is a top level space.
   */
  get isTopLevel() {
    // XXX: This is absolutely not how you find out if the space is top level
    // but is safe for a managed usecase like we offer in the SDK.
    var parentEvents = this.room.currentState.getStateEvents(EventType.SpaceParent);
    if (!(parentEvents !== null && parentEvents !== void 0 && parentEvents.length)) return true;
    return parentEvents.every(e => {
      var _e$getContent;
      return !((_e$getContent = e.getContent()) !== null && _e$getContent !== void 0 && _e$getContent["via"]);
    });
  }

  /**
   * Sets the name of the tree space.
   * @param name - The new name for the space.
   * @returns Promise which resolves when complete.
   */
  setName(name) {
    var _this = this;
    return _asyncToGenerator(function* () {
      yield _this.client.sendStateEvent(_this.roomId, EventType.RoomName, {
        name
      }, "");
    })();
  }

  /**
   * Invites a user to the tree space. They will be given the default Viewer
   * permission level unless specified elsewhere.
   * @param userId - The user ID to invite.
   * @param andSubspaces - True (default) to invite the user to all
   * directories/subspaces too, recursively.
   * @returns Promise which resolves when complete.
   */
  invite(userId) {
    var _arguments = arguments,
      _this2 = this;
    return _asyncToGenerator(function* () {
      var andSubspaces = _arguments.length > 1 && _arguments[1] !== undefined ? _arguments[1] : true;
      var promises = [_this2.retryInvite(userId)];
      if (andSubspaces) {
        promises.push(..._this2.getDirectories().map(d => d.invite(userId, andSubspaces)));
      }
      yield Promise.all(promises);
    })();
  }
  retryInvite(userId) {
    var _this3 = this;
    return _asyncToGenerator(function* () {
      yield simpleRetryOperation(() => _this3.client.invite(_this3.roomId, userId), e => {
        // We don't want to retry permission errors forever...
        if (e instanceof MatrixError && e.errcode === "M_FORBIDDEN") {
          return false;
        }
        return true;
      });
    })();
  }

  /**
   * Sets the permissions of a user to the given role. Note that if setting a user
   * to Owner then they will NOT be able to be demoted. If the user does not have
   * permission to change the power level of the target, an error will be thrown.
   * @param userId - The user ID to change the role of.
   * @param role - The role to assign.
   * @returns Promise which resolves when complete.
   */
  setPermissions(userId, role) {
    var _this4 = this;
    return _asyncToGenerator(function* () {
      var _pls$events;
      var currentPls = _this4.room.currentState.getStateEvents(EventType.RoomPowerLevels, "");
      if (Array.isArray(currentPls)) throw new Error("Unexpected return type for power levels");
      var pls = (currentPls === null || currentPls === void 0 ? void 0 : currentPls.getContent()) || {};
      var viewLevel = pls["users_default"] || 0;
      var editLevel = pls["events_default"] || 50;
      var adminLevel = ((_pls$events = pls["events"]) === null || _pls$events === void 0 ? void 0 : _pls$events[EventType.RoomPowerLevels]) || 100;
      var users = pls["users"] || {};
      switch (role) {
        case TreePermissions.Viewer:
          users[userId] = viewLevel;
          break;
        case TreePermissions.Editor:
          users[userId] = editLevel;
          break;
        case TreePermissions.Owner:
          users[userId] = adminLevel;
          break;
        default:
          throw new Error("Invalid role: " + role);
      }
      pls["users"] = users;
      yield _this4.client.sendStateEvent(_this4.roomId, EventType.RoomPowerLevels, pls, "");
    })();
  }

  /**
   * Gets the current permissions of a user. Note that any users missing explicit permissions (or not
   * in the space) will be considered Viewers. Appropriate membership checks need to be performed
   * elsewhere.
   * @param userId - The user ID to check permissions of.
   * @returns The permissions for the user, defaulting to Viewer.
   */
  getPermissions(userId) {
    var _pls$events2, _pls$users;
    var currentPls = this.room.currentState.getStateEvents(EventType.RoomPowerLevels, "");
    if (Array.isArray(currentPls)) throw new Error("Unexpected return type for power levels");
    var pls = (currentPls === null || currentPls === void 0 ? void 0 : currentPls.getContent()) || {};
    var viewLevel = pls["users_default"] || 0;
    var editLevel = pls["events_default"] || 50;
    var adminLevel = ((_pls$events2 = pls["events"]) === null || _pls$events2 === void 0 ? void 0 : _pls$events2[EventType.RoomPowerLevels]) || 100;
    var userLevel = ((_pls$users = pls["users"]) === null || _pls$users === void 0 ? void 0 : _pls$users[userId]) || viewLevel;
    if (userLevel >= adminLevel) return TreePermissions.Owner;
    if (userLevel >= editLevel) return TreePermissions.Editor;
    return TreePermissions.Viewer;
  }

  /**
   * Creates a directory under this tree space, represented as another tree space.
   * @param name - The name for the directory.
   * @returns Promise which resolves to the created directory.
   */
  createDirectory(name) {
    var _this5 = this;
    return _asyncToGenerator(function* () {
      var directory = yield _this5.client.unstableCreateFileTree(name);
      yield _this5.client.sendStateEvent(_this5.roomId, EventType.SpaceChild, {
        via: [_this5.client.getDomain()]
      }, directory.roomId);
      yield _this5.client.sendStateEvent(directory.roomId, EventType.SpaceParent, {
        via: [_this5.client.getDomain()]
      }, _this5.roomId);
      return directory;
    })();
  }

  /**
   * Gets a list of all known immediate subdirectories to this tree space.
   * @returns The tree spaces (directories). May be empty, but not null.
   */
  getDirectories() {
    var trees = [];
    var children = this.room.currentState.getStateEvents(EventType.SpaceChild);
    for (var child of children) {
      try {
        var stateKey = child.getStateKey();
        if (stateKey) {
          var tree = this.client.unstableGetFileTreeSpace(stateKey);
          if (tree) trees.push(tree);
        }
      } catch (e) {
        logger.warn("Unable to create tree space instance for listing. Are we joined?", e);
      }
    }
    return trees;
  }

  /**
   * Gets a subdirectory of a given ID under this tree space. Note that this will not recurse
   * into children and instead only look one level deep.
   * @param roomId - The room ID (directory ID) to find.
   * @returns The directory, or undefined if not found.
   */
  getDirectory(roomId) {
    return this.getDirectories().find(r => r.roomId === roomId);
  }

  /**
   * Deletes the tree, kicking all members and deleting **all subdirectories**.
   * @returns Promise which resolves when complete.
   */
  delete() {
    var _this6 = this;
    return _asyncToGenerator(function* () {
      var subdirectories = _this6.getDirectories();
      for (var dir of subdirectories) {
        yield dir.delete();
      }
      var kickMemberships = [KnownMembership.Invite, KnownMembership.Knock, KnownMembership.Join];
      var members = _this6.room.currentState.getStateEvents(EventType.RoomMember);
      for (var member of members) {
        var isNotUs = member.getStateKey() !== _this6.client.getUserId();
        if (isNotUs && kickMemberships.includes(member.getContent().membership)) {
          var stateKey = member.getStateKey();
          if (!stateKey) {
            throw new Error("State key not found for branch");
          }
          yield _this6.client.kick(_this6.roomId, stateKey, "Room deleted");
        }
      }
      yield _this6.client.leave(_this6.roomId);
    })();
  }
  getOrderedChildren(children) {
    var ordered = children.map(c => ({
      roomId: c.getStateKey(),
      order: c.getContent()["order"]
    })).filter(c => c.roomId);
    ordered.sort((a, b) => {
      if (a.order && !b.order) {
        return -1;
      } else if (!a.order && b.order) {
        return 1;
      } else if (!a.order && !b.order) {
        var _roomA$currentState$g, _roomA$currentState$g2, _roomB$currentState$g, _roomB$currentState$g2;
        var roomA = this.client.getRoom(a.roomId);
        var roomB = this.client.getRoom(b.roomId);
        if (!roomA || !roomB) {
          // just don't bother trying to do more partial sorting
          return lexicographicCompare(a.roomId, b.roomId);
        }
        var createTsA = (_roomA$currentState$g = (_roomA$currentState$g2 = roomA.currentState.getStateEvents(EventType.RoomCreate, "")) === null || _roomA$currentState$g2 === void 0 ? void 0 : _roomA$currentState$g2.getTs()) !== null && _roomA$currentState$g !== void 0 ? _roomA$currentState$g : 0;
        var createTsB = (_roomB$currentState$g = (_roomB$currentState$g2 = roomB.currentState.getStateEvents(EventType.RoomCreate, "")) === null || _roomB$currentState$g2 === void 0 ? void 0 : _roomB$currentState$g2.getTs()) !== null && _roomB$currentState$g !== void 0 ? _roomB$currentState$g : 0;
        if (createTsA === createTsB) {
          return lexicographicCompare(a.roomId, b.roomId);
        }
        return createTsA - createTsB;
      } else {
        // both not-null orders
        return lexicographicCompare(a.order, b.order);
      }
    });
    return ordered;
  }
  getParentRoom() {
    var parents = this.room.currentState.getStateEvents(EventType.SpaceParent);
    var parent = parents[0]; // XXX: Wild assumption
    if (!parent) throw new Error("Expected to have a parent in a non-top level space");

    // XXX: We are assuming the parent is a valid tree space.
    // We probably don't need to validate the parent room state for this usecase though.
    var stateKey = parent.getStateKey();
    if (!stateKey) throw new Error("No state key found for parent");
    var parentRoom = this.client.getRoom(stateKey);
    if (!parentRoom) throw new Error("Unable to locate room for parent");
    return parentRoom;
  }

  /**
   * Gets the current order index for this directory. Note that if this is the top level space
   * then -1 will be returned.
   * @returns The order index of this space.
   */
  getOrder() {
    if (this.isTopLevel) return -1;
    var parentRoom = this.getParentRoom();
    var children = parentRoom.currentState.getStateEvents(EventType.SpaceChild);
    var ordered = this.getOrderedChildren(children);
    return ordered.findIndex(c => c.roomId === this.roomId);
  }

  /**
   * Sets the order index for this directory within its parent. Note that if this is a top level
   * space then an error will be thrown. -1 can be used to move the child to the start, and numbers
   * larger than the number of children can be used to move the child to the end.
   * @param index - The new order index for this space.
   * @returns Promise which resolves when complete.
   * @throws Throws if this is a top level space.
   */
  setOrder(index) {
    var _this7 = this;
    return _asyncToGenerator(function* () {
      var _currentChild$getCont2;
      if (_this7.isTopLevel) throw new Error("Cannot set order of top level spaces currently");
      var parentRoom = _this7.getParentRoom();
      var children = parentRoom.currentState.getStateEvents(EventType.SpaceChild);
      var ordered = _this7.getOrderedChildren(children);
      index = Math.max(Math.min(index, ordered.length - 1), 0);
      var currentIndex = _this7.getOrder();
      var movingUp = currentIndex < index;
      if (movingUp && index === ordered.length - 1) {
        index--;
      } else if (!movingUp && index === 0) {
        index++;
      }
      var prev = ordered[movingUp ? index : index - 1];
      var next = ordered[movingUp ? index + 1 : index];
      var newOrder = DEFAULT_ALPHABET[0];
      var ensureBeforeIsSane = false;
      if (!prev) {
        // Move to front
        if (next !== null && next !== void 0 && next.order) {
          newOrder = prevString(next.order);
        }
      } else if (index === ordered.length - 1) {
        // Move to back
        if (next !== null && next !== void 0 && next.order) {
          newOrder = nextString(next.order);
        }
      } else {
        // Move somewhere in the middle
        var startOrder = prev === null || prev === void 0 ? void 0 : prev.order;
        var endOrder = next === null || next === void 0 ? void 0 : next.order;
        if (startOrder && endOrder) {
          if (startOrder === endOrder) {
            // Error case: just move +1 to break out of awful math
            newOrder = nextString(startOrder);
          } else {
            newOrder = averageBetweenStrings(startOrder, endOrder);
          }
        } else {
          if (startOrder) {
            // We're at the end (endOrder is null, so no explicit order)
            newOrder = nextString(startOrder);
          } else if (endOrder) {
            // We're at the start (startOrder is null, so nothing before us)
            newOrder = prevString(endOrder);
          } else {
            // Both points are unknown. We're likely in a range where all the children
            // don't have particular order values, so we may need to update them too.
            // The other possibility is there's only us as a child, but we should have
            // shown up in the other states.
            ensureBeforeIsSane = true;
          }
        }
      }
      if (ensureBeforeIsSane) {
        // We were asked by the order algorithm to prepare the moving space for a landing
        // in the undefined order part of the order array, which means we need to update the
        // spaces that come before it with a stable order value.
        var lastOrder;
        for (var i = 0; i <= index; i++) {
          var target = ordered[i];
          if (i === 0) {
            lastOrder = target.order;
          }
          if (!target.order) {
            var _currentChild$getCont;
            // XXX: We should be creating gaps to avoid conflicts
            lastOrder = lastOrder ? nextString(lastOrder) : DEFAULT_ALPHABET[0];
            var _currentChild = parentRoom.currentState.getStateEvents(EventType.SpaceChild, target.roomId);
            var _content = (_currentChild$getCont = _currentChild === null || _currentChild === void 0 ? void 0 : _currentChild.getContent()) !== null && _currentChild$getCont !== void 0 ? _currentChild$getCont : {
              via: [_this7.client.getDomain()]
            };
            yield _this7.client.sendStateEvent(parentRoom.roomId, EventType.SpaceChild, _objectSpread(_objectSpread({}, _content), {}, {
              order: lastOrder
            }), target.roomId);
          } else {
            lastOrder = target.order;
          }
        }
        if (lastOrder) {
          newOrder = nextString(lastOrder);
        }
      }

      // TODO: Deal with order conflicts by reordering

      // Now we can finally update our own order state
      var currentChild = parentRoom.currentState.getStateEvents(EventType.SpaceChild, _this7.roomId);
      var content = (_currentChild$getCont2 = currentChild === null || currentChild === void 0 ? void 0 : currentChild.getContent()) !== null && _currentChild$getCont2 !== void 0 ? _currentChild$getCont2 : {
        via: [_this7.client.getDomain()]
      };
      yield _this7.client.sendStateEvent(parentRoom.roomId, EventType.SpaceChild, _objectSpread(_objectSpread({}, content), {}, {
        // TODO: Safely constrain to 50 character limit required by spaces.
        order: newOrder
      }), _this7.roomId);
    })();
  }

  /**
   * Creates (uploads) a new file to this tree. The file must have already been encrypted for the room.
   * The file contents are in a type that is compatible with MatrixClient.uploadContent().
   * @param name - The name of the file.
   * @param encryptedContents - The encrypted contents.
   * @param info - The encrypted file information.
   * @param additionalContent - Optional event content fields to include in the message.
   * @returns Promise which resolves to the file event's sent response.
   */
  createFile(name, encryptedContents, info, additionalContent) {
    var _this8 = this;
    return _asyncToGenerator(function* () {
      var {
        content_uri: mxc
      } = yield _this8.client.uploadContent(encryptedContents, {
        includeFilename: false
      });
      info.url = mxc;
      var fileContent = {
        msgtype: MsgType.File,
        body: name,
        url: mxc,
        file: info
      };
      additionalContent = additionalContent !== null && additionalContent !== void 0 ? additionalContent : {};
      if (additionalContent["m.new_content"]) {
        // We do the right thing according to the spec, but due to how relations are
        // handled we also end up duplicating this information to the regular `content`
        // as well.
        additionalContent["m.new_content"] = fileContent;
      }
      var res = yield _this8.client.sendMessage(_this8.roomId, _objectSpread(_objectSpread(_objectSpread({}, additionalContent), fileContent), {}, {
        [UNSTABLE_MSC3089_LEAF.name]: {}
      }));
      yield _this8.client.sendStateEvent(_this8.roomId, UNSTABLE_MSC3089_BRANCH.name, {
        active: true,
        name: name
      }, res["event_id"]);
      return res;
    })();
  }

  /**
   * Retrieves a file from the tree.
   * @param fileEventId - The event ID of the file.
   * @returns The file, or null if not found.
   */
  getFile(fileEventId) {
    var branch = this.room.currentState.getStateEvents(UNSTABLE_MSC3089_BRANCH.name, fileEventId);
    return branch ? new MSC3089Branch(this.client, branch, this) : null;
  }

  /**
   * Gets an array of all known files for the tree.
   * @returns The known files. May be empty, but not null.
   */
  listFiles() {
    return this.listAllFiles().filter(b => b.isActive);
  }

  /**
   * Gets an array of all known files for the tree, including inactive/invalid ones.
   * @returns The known files. May be empty, but not null.
   */
  listAllFiles() {
    var _this$room$currentSta;
    var branches = (_this$room$currentSta = this.room.currentState.getStateEvents(UNSTABLE_MSC3089_BRANCH.name)) !== null && _this$room$currentSta !== void 0 ? _this$room$currentSta : [];
    return branches.map(e => new MSC3089Branch(this.client, e, this));
  }
}
//# sourceMappingURL=MSC3089TreeSpace.js.map