import _asyncToGenerator from "@babel/runtime/helpers/asyncToGenerator";
import _defineProperty from "@babel/runtime/helpers/defineProperty";
function ownKeys(e, r) { var t = Object.keys(e); if (Object.getOwnPropertySymbols) { var o = Object.getOwnPropertySymbols(e); r && (o = o.filter(function (r) { return Object.getOwnPropertyDescriptor(e, r).enumerable; })), t.push.apply(t, o); } return t; }
function _objectSpread(e) { for (var r = 1; r < arguments.length; r++) { var t = null != arguments[r] ? arguments[r] : {}; r % 2 ? ownKeys(Object(t), !0).forEach(function (r) { _defineProperty(e, r, t[r]); }) : Object.getOwnPropertyDescriptors ? Object.defineProperties(e, Object.getOwnPropertyDescriptors(t)) : ownKeys(Object(t)).forEach(function (r) { Object.defineProperty(e, r, Object.getOwnPropertyDescriptor(t, r)); }); } return e; }
/*
Copyright 2025 The Matrix.org Foundation C.I.C.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
import { AbortError } from "p-retry";
import { EventType, RelationType } from "../@types/event.js";
import { UpdateDelayedEventAction } from "../@types/requests.js";
import { ConnectionError, HTTPError, MatrixError } from "../http-api/errors.js";
import { logger as rootLogger } from "../logger.js";
import { DEFAULT_EXPIRE_DURATION } from "./CallMembership.js";
import { isMyMembership, Status } from "./types.js";
import { slotDescriptionToId } from "./MatrixRTCSession.js";
import { ActionScheduler } from "./MembershipManagerActionScheduler.js";
import { TypedEventEmitter } from "../models/typed-event-emitter.js";
import { UnsupportedDelayedEventsEndpointError } from "../errors.js";
import { MembershipManagerEvent } from "./IMembershipManager.js";

/* MembershipActionTypes:
On Join:  ───────────────┐   ┌───────────────(1)───────────┐
                         ▼   ▼                             │
                   ┌────────────────┐                      │
                   │SendDelayedEvent│ ──────(2)───┐        │
                   └────────────────┘             │        │
                           │(3)                   │        │
                           ▼                      │        │
                    ┌─────────────┐               │        │
       ┌──────(4)───│SendJoinEvent│────(4)─────┐  │        │
       │            └─────────────┘            │  │        │
       │  ┌─────┐                  ┌──────┐    │  │        │
       ▼  ▼     │                  │      ▼    ▼  ▼        │
┌────────────┐  │                  │ ┌───────────────────┐ │
│UpdateExpiry│ (s)                (s)|RestartDelayedEvent│ │
└────────────┘  │                  │ └───────────────────┘ │
          │     │                  │      │        │       │
          └─────┘                  └──────┘        └───────┘

On Leave: ─────────  STOP ALL ABOVE
                           ▼
            ┌────────────────────────────────┐
            │ SendScheduledDelayedLeaveEvent │
            └────────────────────────────────┘
                           │(5)
                           ▼
                    ┌──────────────┐
                    │SendLeaveEvent│
                    └──────────────┘
(1) [Not found error] results in resending the delayed event
(2) [hasMemberEvent = true] Sending the delayed event if we
    already have a call member event results jumping to the
    RestartDelayedEvent loop directly
(3) [hasMemberEvent = false] if there is not call member event
    sending it is the next step
(4) Both (UpdateExpiry and RestartDelayedEvent) actions are
    scheduled when successfully sending the state event
(5) Only if delayed event sending failed (fallback)
(s) Successful restart/resend
*/

/**
 * The different types of actions the MembershipManager can take.
 * @internal
 */
export var MembershipActionType = /*#__PURE__*/function (MembershipActionType) {
  MembershipActionType["SendDelayedEvent"] = "SendDelayedEvent";
  //  -> MembershipActionType.SendJoinEvent if successful
  //  -> DelayedLeaveActionType.SendDelayedEvent on error, retry sending the first delayed event.
  //  -> DelayedLeaveActionType.RestartDelayedEvent on success start updating the delayed event
  MembershipActionType["SendJoinEvent"] = "SendJoinEvent";
  //  -> MembershipActionType.SendJoinEvent if we run into a rate limit and need to retry
  //  -> MembershipActionType.Update if we successfully send the join event then schedule the expire event update
  //  -> DelayedLeaveActionType.RestartDelayedEvent to recheck the delayed event
  MembershipActionType["RestartDelayedEvent"] = "RestartDelayedEvent";
  //  -> DelayedLeaveActionType.SendMainDelayedEvent on missing delay id but there is a rtc state event
  //  -> DelayedLeaveActionType.SendDelayedEvent on missing delay id and there is no state event
  //  -> DelayedLeaveActionType.RestartDelayedEvent on success we schedule the next restart
  MembershipActionType["UpdateExpiry"] = "UpdateExpiry";
  //  -> MembershipActionType.Update if the timeout has passed so the next update is required.
  MembershipActionType["SendScheduledDelayedLeaveEvent"] = "SendScheduledDelayedLeaveEvent";
  //  -> MembershipActionType.SendLeaveEvent on failure (not found) we need to send the leave manually and cannot use the scheduled delayed event
  //  -> DelayedLeaveActionType.SendScheduledDelayedLeaveEvent on error we try again.
  MembershipActionType["SendLeaveEvent"] = "SendLeaveEvent"; // -> MembershipActionType.SendLeaveEvent
  return MembershipActionType;
}({});

/**
 * @internal
 */

/**
 * This class is responsible for sending all events relating to the own membership of a matrixRTC call.
 * It has the following tasks:
 *  - Send the users leave delayed event before sending the membership
 *  - Send the users membership if the state machine is started
 *  - Check if the delayed event was canceled due to sending the membership
 *  - update the delayed event (`restart`)
 *  - Update the state event every ~5h = `DEFAULT_EXPIRE_DURATION` (so it does not get treated as expired)
 *  - When the state machine is stopped:
 *   - Disconnect the member
 *   - Stop the timer for the delay refresh
 *   - Stop the timer for updating the state event
 */
export class MembershipManager extends TypedEventEmitter {
  isActivated() {
    return this.activated;
  }
  // DEPRECATED use isActivated
  isJoined() {
    return this.isActivated();
  }

  /**
   * Puts the MembershipManager in a state where it tries to be joined.
   * It will send delayed events and membership events
   * @param fociPreferred the list of preferred foci to use in the joined RTC membership event.
   * If multiSfuFocus is set, this is only needed if this client wants to publish to multiple transports simultaneously.
   * @param multiSfuFocus the active focus to use in the joined RTC membership event. Setting this implies the
   * membership manager will operate in a multi-SFU connection mode. If `undefined`, an `oldest_membership`
   * transport selection will be used instead.
   * @param onError This will be called once the membership manager encounters an unrecoverable error.
   * This should bubble up the the frontend to communicate that the call does not work in the current environment.
   */
  join(fociPreferred, multiSfuFocus, onError) {
    if (this.scheduler.running) {
      this.logger.error("MembershipManager is already running. Ignoring join request.");
      return;
    }
    this.fociPreferred = fociPreferred;
    this.rtcTransport = multiSfuFocus;
    this.leavePromiseResolvers = undefined;
    this.activated = true;
    this.oldStatus = this.status;
    this.state = MembershipManager.defaultState;
    this.scheduler.startWithJoin().catch(e => {
      this.logger.error("MembershipManager stopped because: ", e);
      onError === null || onError === void 0 || onError(e);
    }).finally(() => {
      // Should already be set to false when calling `leave` in non error cases.
      this.activated = false;
      // Here the scheduler is not running anymore so we the `membershipLoopHandler` is not called to emit.
      if (this.oldStatus && this.oldStatus !== this.status) {
        this.emit(MembershipManagerEvent.StatusChanged, this.oldStatus, this.status);
      }
      if (!this.scheduler.running) {
        var _this$leavePromiseRes;
        (_this$leavePromiseRes = this.leavePromiseResolvers) === null || _this$leavePromiseRes === void 0 || _this$leavePromiseRes.resolve(true);
        this.leavePromiseResolvers = undefined;
      }
    });
  }

  /**
   * Leave from the call (Send an rtc session event with content: `{}`)
   * @param timeout the maximum duration this promise will take to resolve
   * @returns true if it managed to leave and false if the timeout condition happened.
   */
  leave(timeout) {
    if (!this.scheduler.running) {
      this.logger.warn("Called MembershipManager.leave() even though the MembershipManager is not running");
      return Promise.resolve(true);
    }

    // We use the promise to track if we already scheduled a leave event
    // So we do not check scheduler.actions/scheduler.insertions
    if (!this.leavePromiseResolvers) {
      // reset scheduled actions so we will not do any new actions.
      this.leavePromiseResolvers = Promise.withResolvers();
      this.activated = false;
      this.scheduler.initiateLeave();
      if (timeout) setTimeout(() => {
        var _this$leavePromiseRes2;
        return (_this$leavePromiseRes2 = this.leavePromiseResolvers) === null || _this$leavePromiseRes2 === void 0 ? void 0 : _this$leavePromiseRes2.resolve(false);
      }, timeout);
    }
    return this.leavePromiseResolvers.promise;
  }
  onRTCSessionMemberUpdate(memberships) {
    if (!this.isActivated()) {
      return Promise.resolve();
    }
    var userId = this.client.getUserId();
    var deviceId = this.client.getDeviceId();
    if (!userId || !deviceId) {
      this.logger.error("MembershipManager.onRTCSessionMemberUpdate called without user or device id");
      return Promise.resolve();
    }
    this._ownMembership = memberships.find(m => isMyMembership(m, userId, deviceId));
    if (!this._ownMembership) {
      // If one of these actions are scheduled or are getting inserted in the next iteration, we should already
      // take care of our missing membership.
      var sendingMembershipActions = [MembershipActionType.SendDelayedEvent, MembershipActionType.SendJoinEvent];
      this.logger.warn("Missing own membership: force re-join");
      this.state.hasMemberStateEvent = false;
      if (this.scheduler.actions.some(a => sendingMembershipActions.includes(a.type))) {
        this.logger.error("tried adding another `SendDelayedEvent` actions even though we already have one in the Queue\nActionQueueOnMemberUpdate:", this.scheduler.actions);
      } else {
        // Only react to our own membership missing if we have not already scheduled sending a new membership DirectMembershipManagerAction.Join
        this.scheduler.initiateJoin();
      }
    }
    return Promise.resolve();
  }
  updateCallIntent(callIntent) {
    var _this = this;
    return _asyncToGenerator(function* () {
      if (!_this.activated || !_this.ownMembership) {
        throw Error("You cannot update your intent before joining the call");
      }
      if (_this.ownMembership.callIntent === callIntent) {
        return; // No-op
      }
      _this.callIntent = callIntent;
      // Kick off a new membership event as a result.
      yield _this.sendJoinEvent();
    })();
  }

  /**
   * @throws if the client does not return user or device id.
   * @param joinConfig
   * @param room
   * @param client
   */
  constructor(joinConfig, room, client, slotDescription, parentLogger) {
    super();
    this.joinConfig = joinConfig;
    this.room = room;
    this.client = client;
    this.slotDescription = slotDescription;
    _defineProperty(this, "activated", false);
    _defineProperty(this, "logger", void 0);
    _defineProperty(this, "callIntent", void 0);
    _defineProperty(this, "leavePromiseResolvers", void 0);
    _defineProperty(this, "_ownMembership", void 0);
    // scheduler
    _defineProperty(this, "oldStatus", void 0);
    _defineProperty(this, "scheduler", void 0);
    // MembershipManager mutable state.
    _defineProperty(this, "state", void 0);
    // Membership Event static parameters:
    _defineProperty(this, "deviceId", void 0);
    _defineProperty(this, "memberId", void 0);
    /** @deprecated This will be removed in favor or rtcTransport becoming a list of actively used transports */
    _defineProperty(this, "fociPreferred", void 0);
    _defineProperty(this, "rtcTransport", void 0);
    // Config:
    _defineProperty(this, "delayedLeaveEventDelayMsOverride", void 0);
    this.logger = (parentLogger !== null && parentLogger !== void 0 ? parentLogger : rootLogger).getChild("[MembershipManager]");
    var [userId, deviceId] = [this.client.getUserId(), this.client.getDeviceId()];
    if (userId === null) throw Error("Missing userId in client");
    if (deviceId === null) throw Error("Missing deviceId in client");
    this.deviceId = deviceId;
    // this needs to become a uuid so that consecutive join/leaves result in a key rotation.
    // we keep it as a string for now for backwards compatibility.
    this.memberId = this.makeMembershipStateKey(userId, deviceId);
    this.state = MembershipManager.defaultState;
    this.callIntent = joinConfig === null || joinConfig === void 0 ? void 0 : joinConfig.callIntent;
    this.scheduler = new ActionScheduler(type => {
      if (this.oldStatus) {
        // we put this at the beginning of the actions scheduler loop handle callback since it is a loop this
        // is equivalent to running it at the end of the loop. (just after applying the status/action list changes)
        // This order is required because this method needs to return the action updates.
        this.logger.debug("MembershipManager applied action changes. Status: ".concat(this.oldStatus, " -> ").concat(this.status));
        if (this.oldStatus !== this.status) {
          this.emit(MembershipManagerEvent.StatusChanged, this.oldStatus, this.status);
        }
      }
      this.oldStatus = this.status;
      this.logger.debug("MembershipManager before processing action. status=".concat(this.oldStatus));
      return this.membershipLoopHandler(type);
    }, this.logger);
  }
  get ownMembership() {
    return this._ownMembership;
  }
  static get defaultState() {
    return {
      hasMemberStateEvent: false,
      delayId: undefined,
      startTime: 0,
      rateLimitRetries: new Map(),
      networkErrorRetries: new Map(),
      expireUpdateIterations: 1,
      probablyLeft: false
    };
  }
  get networkErrorRetryMs() {
    var _this$joinConfig$netw, _this$joinConfig;
    return (_this$joinConfig$netw = (_this$joinConfig = this.joinConfig) === null || _this$joinConfig === void 0 ? void 0 : _this$joinConfig.networkErrorRetryMs) !== null && _this$joinConfig$netw !== void 0 ? _this$joinConfig$netw : 3000;
  }
  get membershipEventExpiryMs() {
    var _this$joinConfig$memb, _this$joinConfig2;
    return (_this$joinConfig$memb = (_this$joinConfig2 = this.joinConfig) === null || _this$joinConfig2 === void 0 ? void 0 : _this$joinConfig2.membershipEventExpiryMs) !== null && _this$joinConfig$memb !== void 0 ? _this$joinConfig$memb : DEFAULT_EXPIRE_DURATION;
  }
  get membershipEventExpiryHeadroomMs() {
    var _this$joinConfig$memb2, _this$joinConfig3;
    return (_this$joinConfig$memb2 = (_this$joinConfig3 = this.joinConfig) === null || _this$joinConfig3 === void 0 ? void 0 : _this$joinConfig3.membershipEventExpiryHeadroomMs) !== null && _this$joinConfig$memb2 !== void 0 ? _this$joinConfig$memb2 : 5000;
  }
  computeNextExpiryActionTs(iteration) {
    return this.state.startTime + this.membershipEventExpiryMs * iteration - this.membershipEventExpiryHeadroomMs;
  }
  get delayedLeaveEventDelayMs() {
    var _ref, _this$delayedLeaveEve, _this$joinConfig4;
    return (_ref = (_this$delayedLeaveEve = this.delayedLeaveEventDelayMsOverride) !== null && _this$delayedLeaveEve !== void 0 ? _this$delayedLeaveEve : (_this$joinConfig4 = this.joinConfig) === null || _this$joinConfig4 === void 0 ? void 0 : _this$joinConfig4.delayedLeaveEventDelayMs) !== null && _ref !== void 0 ? _ref : 8000;
  }
  get delayedLeaveEventRestartMs() {
    var _this$joinConfig$dela, _this$joinConfig5;
    return (_this$joinConfig$dela = (_this$joinConfig5 = this.joinConfig) === null || _this$joinConfig5 === void 0 ? void 0 : _this$joinConfig5.delayedLeaveEventRestartMs) !== null && _this$joinConfig$dela !== void 0 ? _this$joinConfig$dela : 5000;
  }
  get maximumRateLimitRetryCount() {
    var _this$joinConfig$maxi, _this$joinConfig6;
    return (_this$joinConfig$maxi = (_this$joinConfig6 = this.joinConfig) === null || _this$joinConfig6 === void 0 ? void 0 : _this$joinConfig6.maximumRateLimitRetryCount) !== null && _this$joinConfig$maxi !== void 0 ? _this$joinConfig$maxi : 10;
  }
  get maximumNetworkErrorRetryCount() {
    var _this$joinConfig$maxi2, _this$joinConfig7;
    return (_this$joinConfig$maxi2 = (_this$joinConfig7 = this.joinConfig) === null || _this$joinConfig7 === void 0 ? void 0 : _this$joinConfig7.maximumNetworkErrorRetryCount) !== null && _this$joinConfig$maxi2 !== void 0 ? _this$joinConfig$maxi2 : 10;
  }
  get delayedLeaveEventRestartLocalTimeoutMs() {
    var _this$joinConfig$dela2, _this$joinConfig8;
    return (_this$joinConfig$dela2 = (_this$joinConfig8 = this.joinConfig) === null || _this$joinConfig8 === void 0 ? void 0 : _this$joinConfig8.delayedLeaveEventRestartLocalTimeoutMs) !== null && _this$joinConfig$dela2 !== void 0 ? _this$joinConfig$dela2 : 2000;
  }
  get useRtcMemberFormat() {
    var _this$joinConfig$useR, _this$joinConfig9;
    return (_this$joinConfig$useR = (_this$joinConfig9 = this.joinConfig) === null || _this$joinConfig9 === void 0 ? void 0 : _this$joinConfig9.useRtcMemberFormat) !== null && _this$joinConfig$useR !== void 0 ? _this$joinConfig$useR : false;
  }
  // LOOP HANDLER:
  membershipLoopHandler(type) {
    var _this2 = this;
    return _asyncToGenerator(function* () {
      switch (type) {
        case MembershipActionType.SendDelayedEvent:
          {
            // Before we start we check if we come from a state where we have a delay id.
            if (!_this2.state.delayId) {
              return _this2.sendOrResendDelayedLeaveEvent(); // Normal case without any previous delayed id.
            } else {
              // This can happen if someone else (or another client) removes our own membership event.
              // It will trigger `onRTCSessionMemberUpdate` queue `MembershipActionType.SendDelayedEvent`.
              // We might still have our delayed event from the previous participation and dependent on the server this might not
              // get removed automatically if the state changes. Hence, it would remove our membership unexpectedly shortly after the rejoin.
              //
              // In this block we will try to cancel this delayed event before setting up a new one.

              return _this2.cancelKnownDelayIdBeforeSendDelayedEvent(_this2.state.delayId);
            }
          }
        case MembershipActionType.RestartDelayedEvent:
          {
            if (!_this2.state.delayId) {
              // Delay id got reset. This action was used to check if the hs canceled the delayed event when the join state got sent.
              return createInsertActionUpdate(MembershipActionType.SendDelayedEvent);
            }
            return _this2.restartDelayedEvent(_this2.state.delayId);
          }
        case MembershipActionType.SendScheduledDelayedLeaveEvent:
          {
            // We are already good
            if (!_this2.state.hasMemberStateEvent) {
              return {
                replace: []
              };
            }
            if (_this2.state.delayId) {
              return _this2.sendScheduledDelayedLeaveEventOrFallbackToSendLeaveEvent(_this2.state.delayId);
            } else {
              return createInsertActionUpdate(MembershipActionType.SendLeaveEvent);
            }
          }
        case MembershipActionType.SendJoinEvent:
          {
            return _this2.sendJoinEvent();
          }
        case MembershipActionType.UpdateExpiry:
          {
            return _this2.updateExpiryOnJoinedEvent();
          }
        case MembershipActionType.SendLeaveEvent:
          {
            // We are good already
            if (!_this2.state.hasMemberStateEvent) {
              return {
                replace: []
              };
            }
            // This is only a fallback in case we do not have working delayed events support.
            // first we should try to just send the scheduled leave event
            return _this2.sendFallbackLeaveEvent();
          }
      }
    })();
  }

  // HANDLERS (used in the membershipLoopHandler)
  sendOrResendDelayedLeaveEvent() {
    var _this3 = this;
    return _asyncToGenerator(function* () {
      // We can reach this at the start of a call (where we do not yet have a membership: state.hasMemberStateEvent=false)
      // or during a call if the state event canceled our delayed event or caused by an unexpected error that removed our delayed event.
      // (Another client could have canceled it, the homeserver might have removed/lost it due to a restart, ...)
      // In the `then` and `catch` block we treat both cases differently. "if (this.state.hasMemberStateEvent) {} else {}"
      return yield _this3.client._unstable_sendDelayedStateEvent(_this3.room.roomId, {
        delay: _this3.delayedLeaveEventDelayMs
      }, _this3.useRtcMemberFormat ? EventType.RTCMembership : EventType.GroupCallMemberPrefix, {},
      // leave event
      _this3.memberId).then(response => {
        _this3.state.expectedServerDelayLeaveTs = Date.now() + _this3.delayedLeaveEventDelayMs;
        _this3.setAndEmitProbablyLeft(false);
        // On success we reset retries and set delayId.
        _this3.resetRateLimitCounter(MembershipActionType.SendDelayedEvent);
        _this3.state.delayId = response.delay_id;
        if (_this3.state.hasMemberStateEvent) {
          // This action was scheduled because the previous delayed event was cancelled
          // due to lack of https://github.com/element-hq/synapse/pull/17810
          return createInsertActionUpdate(MembershipActionType.RestartDelayedEvent, _this3.delayedLeaveEventRestartMs);
        } else {
          // This action was scheduled because we are in the process of joining
          return createInsertActionUpdate(MembershipActionType.SendJoinEvent);
        }
      }).catch(e => {
        var repeatActionType = MembershipActionType.SendDelayedEvent;
        if (_this3.manageMaxDelayExceededSituation(e)) {
          return createInsertActionUpdate(repeatActionType);
        }
        var update = _this3.actionUpdateFromErrors(e, repeatActionType, "sendDelayedStateEvent");
        if (update) return update;
        if (_this3.state.hasMemberStateEvent) {
          // This action was scheduled because the previous delayed event was cancelled
          // due to lack of https://github.com/element-hq/synapse/pull/17810

          // Don't do any other delayed event work if its not supported.
          if (_this3.isUnsupportedDelayedEndpoint(e)) return {};
          throw Error("Could not send delayed event, even though delayed events are supported. " + e);
        } else {
          // This action was scheduled because we are in the process of joining
          // log and fall through
          if (_this3.isUnsupportedDelayedEndpoint(e)) {
            _this3.logger.info("Not using delayed event because the endpoint is not supported");
          } else {
            _this3.logger.info("Not using delayed event because: " + e);
          }
          // On any other error we fall back to not using delayed events and send the join state event immediately
          return createInsertActionUpdate(MembershipActionType.SendJoinEvent);
        }
      });
    })();
  }
  cancelKnownDelayIdBeforeSendDelayedEvent(delayId) {
    var _this4 = this;
    return _asyncToGenerator(function* () {
      // Remove all running updates and restarts
      return yield _this4.client._unstable_updateDelayedEvent(delayId, UpdateDelayedEventAction.Cancel).then(() => {
        _this4.state.delayId = undefined;
        _this4.resetRateLimitCounter(MembershipActionType.SendDelayedEvent);
        return createReplaceActionUpdate(MembershipActionType.SendDelayedEvent);
      }).catch(e => {
        var repeatActionType = MembershipActionType.SendDelayedEvent;
        var update = _this4.actionUpdateFromErrors(e, repeatActionType, "updateDelayedEvent");
        if (update) return update;
        if (_this4.isNotFoundError(e)) {
          // If we get a M_NOT_FOUND we know that the delayed event got already removed.
          // This means we are good and can set it to undefined and run this again.
          _this4.state.delayId = undefined;
          return createReplaceActionUpdate(repeatActionType);
        }
        if (_this4.isUnsupportedDelayedEndpoint(e)) {
          return createReplaceActionUpdate(MembershipActionType.SendJoinEvent);
        }
        // We do not just ignore and log this error since we would also need to reset the delayId.

        // This becomes an unrecoverable error case since something is significantly off if we don't hit any of the above cases
        // when state.delayId !== undefined
        // We do not just ignore and log this error since we would also need to reset the delayId.
        // It is cleaner if we, the frontend, rejoins instead of resetting the delayId here and behaving like in the success case.
        throw Error("We failed to cancel a delayed event where we already had a delay id with an error we cannot automatically handle");
      });
    })();
  }
  setAndEmitProbablyLeft(probablyLeft) {
    if (this.state.probablyLeft === probablyLeft) {
      return;
    }
    this.state.probablyLeft = probablyLeft;
    this.emit(MembershipManagerEvent.ProbablyLeft, this.state.probablyLeft);
  }
  restartDelayedEvent(delayId) {
    var _this5 = this;
    return _asyncToGenerator(function* () {
      // Compute the duration until we expect the server to send the delayed leave event.
      var durationUntilServerDelayedLeave = _this5.state.expectedServerDelayLeaveTs ? _this5.state.expectedServerDelayLeaveTs - Date.now() : undefined;
      var abortPromise = new Promise((_, reject) => {
        setTimeout(() => {
          reject(new AbortError("Restart delayed event timed out before the HS responded"));
        },
        // We abort immediately at the time where we expect the server to send the delayed leave event.
        // At this point we want the catch block to run and set the `probablyLeft` state.
        //
        // While we are already in probablyLeft state, we use the unaltered delayedLeaveEventRestartLocalTimeoutMs.
        durationUntilServerDelayedLeave !== undefined && !_this5.state.probablyLeft ? Math.min(_this5.delayedLeaveEventRestartLocalTimeoutMs, durationUntilServerDelayedLeave) : _this5.delayedLeaveEventRestartLocalTimeoutMs);
      });

      // The obvious choice here would be to use the `IRequestOpts` to set the timeout. Since this call might be forwarded
      // to the widget driver this information would get lost. That is why we mimic the AbortError using the race.
      return yield Promise.race([_this5.client._unstable_updateDelayedEvent(delayId, UpdateDelayedEventAction.Restart), abortPromise]).then(() => {
        // Whenever we successfully restart the delayed event we update the `state.expectedServerDelayLeaveTs`
        // which stores the predicted timestamp at which the server will send the delayed leave event if there wont be any further
        // successful restart requests.
        _this5.state.expectedServerDelayLeaveTs = Date.now() + _this5.delayedLeaveEventDelayMs;
        _this5.resetRateLimitCounter(MembershipActionType.RestartDelayedEvent);
        _this5.setAndEmitProbablyLeft(false);
        return createInsertActionUpdate(MembershipActionType.RestartDelayedEvent, _this5.delayedLeaveEventRestartMs);
      }).catch(e => {
        if (_this5.state.expectedServerDelayLeaveTs && _this5.state.expectedServerDelayLeaveTs <= Date.now()) {
          // Once we reach this point it's likely that the server is sending the delayed leave event so we emit `probablyLeft = true`.
          // It will emit `probablyLeft = false` once we notice about our leave through sync and successfully setup a new state event.
          _this5.setAndEmitProbablyLeft(true);
        }
        var repeatActionType = MembershipActionType.RestartDelayedEvent;
        if (_this5.isNotFoundError(e)) {
          _this5.state.delayId = undefined;
          return createInsertActionUpdate(MembershipActionType.SendDelayedEvent);
        }
        // If the HS does not support delayed events we wont reschedule.
        if (_this5.isUnsupportedDelayedEndpoint(e)) return {};

        // TODO this also needs a test: get rate limit while checking id delayed event is scheduled
        var update = _this5.actionUpdateFromErrors(e, repeatActionType, "updateDelayedEvent");
        if (update) return update;

        // In other error cases we have no idea what is happening
        throw Error("Could not restart delayed event, even though delayed events are supported. " + e);
      });
    })();
  }
  sendScheduledDelayedLeaveEventOrFallbackToSendLeaveEvent(delayId) {
    var _this6 = this;
    return _asyncToGenerator(function* () {
      return yield _this6.client._unstable_updateDelayedEvent(delayId, UpdateDelayedEventAction.Send).then(() => {
        _this6.state.hasMemberStateEvent = false;
        _this6.resetRateLimitCounter(MembershipActionType.SendScheduledDelayedLeaveEvent);
        return {
          replace: []
        };
      }).catch(e => {
        var repeatActionType = MembershipActionType.SendLeaveEvent;
        if (_this6.isUnsupportedDelayedEndpoint(e)) return {};
        if (_this6.isNotFoundError(e)) {
          _this6.state.delayId = undefined;
          return createInsertActionUpdate(repeatActionType);
        }
        var update = _this6.actionUpdateFromErrors(e, repeatActionType, "updateDelayedEvent");
        if (update) return update;

        // On any other error we fall back to SendLeaveEvent (this includes hard errors from rate limiting)
        _this6.logger.warn("Encountered unexpected error during SendScheduledDelayedLeaveEvent. Falling back to SendLeaveEvent", e);
        return createInsertActionUpdate(repeatActionType);
      });
    })();
  }
  sendJoinEvent() {
    var _this7 = this;
    return _asyncToGenerator(function* () {
      return yield _this7.client.sendStateEvent(_this7.room.roomId, _this7.useRtcMemberFormat ? EventType.RTCMembership : EventType.GroupCallMemberPrefix, _this7.makeMyMembership(_this7.membershipEventExpiryMs), _this7.memberId).then(() => {
        _this7.setAndEmitProbablyLeft(false);
        _this7.state.startTime = Date.now();
        // The next update should already use twice the membershipEventExpiryTimeout
        _this7.state.expireUpdateIterations = 1;
        _this7.state.hasMemberStateEvent = true;
        _this7.resetRateLimitCounter(MembershipActionType.SendJoinEvent);
        // An UpdateExpiry action might be left over from a previous join event.
        // We can reach sendJoinEvent when the delayed leave event gets send by the HS.
        // The branch where we might have a leftover UpdateExpiry action is:
        // RestartDelayedEvent (cannot find it, server removed it)
        // -> SendDelayedEvent (send new delayed event)
        // -> SendJoinEvent (here with a still scheduled UpdateExpiry action)
        var actionsWithoutUpdateExpiry = _this7.scheduler.actions.filter(a => a.type !== MembershipActionType.UpdateExpiry &&
        // A new UpdateExpiry action with an updated will be scheduled,
        a.type !== MembershipActionType.SendJoinEvent // Manually remove the SendJoinEvent action,
        );
        return {
          replace: [...actionsWithoutUpdateExpiry,
          // To check if the delayed event is still there or got removed by inserting the stateEvent, we need to restart it.
          {
            ts: Date.now(),
            type: MembershipActionType.RestartDelayedEvent
          }, {
            ts: _this7.computeNextExpiryActionTs(_this7.state.expireUpdateIterations),
            type: MembershipActionType.UpdateExpiry
          }]
        };
      }).catch(e => {
        var update = _this7.actionUpdateFromErrors(e, MembershipActionType.SendJoinEvent, "sendStateEvent");
        if (update) return update;
        throw e;
      });
    })();
  }
  updateExpiryOnJoinedEvent() {
    var _this8 = this;
    return _asyncToGenerator(function* () {
      var nextExpireUpdateIteration = _this8.state.expireUpdateIterations + 1;
      return yield _this8.client.sendStateEvent(_this8.room.roomId, _this8.useRtcMemberFormat ? EventType.RTCMembership : EventType.GroupCallMemberPrefix, _this8.makeMyMembership(_this8.membershipEventExpiryMs * nextExpireUpdateIteration), _this8.memberId).then(() => {
        // Success, we reset retries and schedule update.
        _this8.resetRateLimitCounter(MembershipActionType.UpdateExpiry);
        _this8.state.expireUpdateIterations = nextExpireUpdateIteration;
        return {
          insert: [{
            ts: _this8.computeNextExpiryActionTs(nextExpireUpdateIteration),
            type: MembershipActionType.UpdateExpiry
          }]
        };
      }).catch(e => {
        var update = _this8.actionUpdateFromErrors(e, MembershipActionType.UpdateExpiry, "sendStateEvent");
        if (update) return update;
        throw e;
      });
    })();
  }
  sendFallbackLeaveEvent() {
    var _this9 = this;
    return _asyncToGenerator(function* () {
      return yield _this9.client.sendStateEvent(_this9.room.roomId, _this9.useRtcMemberFormat ? EventType.RTCMembership : EventType.GroupCallMemberPrefix, {}, _this9.memberId).then(() => {
        _this9.resetRateLimitCounter(MembershipActionType.SendLeaveEvent);
        _this9.state.hasMemberStateEvent = false;
        return {
          replace: []
        };
      }).catch(e => {
        var update = _this9.actionUpdateFromErrors(e, MembershipActionType.SendLeaveEvent, "sendStateEvent");
        if (update) return update;
        throw e;
      });
    })();
  }

  // HELPERS
  makeMembershipStateKey(localUserId, localDeviceId) {
    var stateKey = "".concat(localUserId, "_").concat(localDeviceId, "_").concat(this.slotDescription.application).concat(this.slotDescription.id);
    if (/^org\.matrix\.msc(3757|3779)\b/.exec(this.room.getVersion())) {
      return stateKey;
    } else {
      return "_".concat(stateKey);
    }
  }

  /**
   * Constructs our own membership
   */
  makeMyMembership(expires) {
    var ownMembership = this.ownMembership;
    if (this.useRtcMemberFormat) {
      var relationObject = ownMembership !== null && ownMembership !== void 0 && ownMembership.eventId ? {
        "m.relation": {
          rel_type: RelationType.Reference,
          event_id: ownMembership === null || ownMembership === void 0 ? void 0 : ownMembership.eventId
        }
      } : {};
      return _objectSpread({
        application: _objectSpread({
          type: this.slotDescription.application
        }, this.callIntent ? {
          "m.call.intent": this.callIntent
        } : {}),
        slot_id: slotDescriptionToId(this.slotDescription),
        rtc_transports: this.rtcTransport ? [this.rtcTransport] : [],
        member: {
          device_id: this.deviceId,
          user_id: this.client.getUserId(),
          id: this.memberId
        },
        versions: []
      }, relationObject);
    } else {
      var _this$fociPreferred, _this$fociPreferred2;
      var focusObjects = this.rtcTransport === undefined ? {
        focus_active: {
          type: "livekit",
          focus_selection: "oldest_membership"
        },
        foci_preferred: (_this$fociPreferred = this.fociPreferred) !== null && _this$fociPreferred !== void 0 ? _this$fociPreferred : []
      } : {
        focus_active: {
          type: "livekit",
          focus_selection: "multi_sfu"
        },
        foci_preferred: [this.rtcTransport, ...((_this$fociPreferred2 = this.fociPreferred) !== null && _this$fociPreferred2 !== void 0 ? _this$fociPreferred2 : [])]
      };
      return _objectSpread(_objectSpread({
        "application": this.slotDescription.application,
        "call_id": this.slotDescription.id,
        "scope": "m.room",
        "device_id": this.deviceId,
        expires,
        "m.call.intent": this.callIntent
      }, focusObjects), ownMembership !== undefined ? {
        created_ts: ownMembership.createdTs()
      } : undefined);
    }
  }

  // Error checks and handlers

  /**
   * Check if its a NOT_FOUND error
   * @param error the error causing this handler check/execution
   * @returns true if its a not found error
   */
  isNotFoundError(error) {
    return error instanceof MatrixError && error.errcode === "M_NOT_FOUND";
  }

  /**
   * Check if this is a DelayExceeded timeout and update the TimeoutOverride for the next try
   * @param error the error causing this handler check/execution
   * @returns true if its a delay exceeded error and we updated the local TimeoutOverride
   */
  manageMaxDelayExceededSituation(error) {
    if (error instanceof MatrixError && error.errcode === "M_UNKNOWN" && error.data["org.matrix.msc4140.errcode"] === "M_MAX_DELAY_EXCEEDED") {
      var maxDelayAllowed = error.data["org.matrix.msc4140.max_delay"];
      if (typeof maxDelayAllowed === "number" && this.delayedLeaveEventDelayMs > maxDelayAllowed) {
        this.delayedLeaveEventDelayMsOverride = maxDelayAllowed;
      }
      this.logger.warn("Retry sending delayed disconnection event due to server timeout limitations:", error);
      return true;
    }
    return false;
  }
  actionUpdateFromErrors(error, type, method) {
    var updateLimit = this.actionUpdateFromRateLimitError(error, method, type);
    if (updateLimit) return updateLimit;
    var updateNetwork = this.actionUpdateFromNetworkErrorRetry(error, type);
    if (updateNetwork) return updateNetwork;
  }
  /**
   * Check if we have a rate limit error and schedule the same action again if we dont exceed the rate limit retry count yet.
   * @param error the error causing this handler check/execution
   * @param method the method used for the throw message
   * @param type which MembershipActionType we reschedule because of a rate limit.
   * @throws If it is a rate limit error and the retry count got exceeded
   * @returns Returns true if we handled the error by rescheduling the correct next action.
   * Returns false if it is not a network error.
   */
  actionUpdateFromRateLimitError(error, method, type) {
    var _this$state$rateLimit;
    // "Is rate limit"-boundary
    if (!((error instanceof HTTPError || error instanceof MatrixError) && error.isRateLimitError())) {
      return undefined;
    }

    // retry boundary
    var rateLimitRetries = (_this$state$rateLimit = this.state.rateLimitRetries.get(type)) !== null && _this$state$rateLimit !== void 0 ? _this$state$rateLimit : 0;
    if (rateLimitRetries < this.maximumRateLimitRetryCount) {
      var resendDelay;
      var defaultMs = 5000;
      try {
        var _error$getRetryAfterM;
        resendDelay = (_error$getRetryAfterM = error.getRetryAfterMs()) !== null && _error$getRetryAfterM !== void 0 ? _error$getRetryAfterM : defaultMs;
        this.logger.info("Rate limited by server, retrying in ".concat(resendDelay, "ms"));
      } catch (e) {
        this.logger.warn("Error while retrieving a rate-limit retry delay, retrying after default delay of ".concat(defaultMs), e);
        resendDelay = defaultMs;
      }
      this.state.rateLimitRetries.set(type, rateLimitRetries + 1);
      return createInsertActionUpdate(type, resendDelay);
    }
    throw Error("Exceeded maximum retries for " + type + " attempts (client." + method + "): " + error);
  }

  /**
   * FIXME Don't Check the error and retry the same MembershipAction again in the configured time and for the configured retry count.
   * @param error the error causing this handler check/execution
   * @param type the action type that we need to repeat because of the error
   * @throws If it is a network error and the retry count got exceeded
   * @returns
   * Returns true if we handled the error by rescheduling the correct next action.
   * Returns false if it is not a network error.
   */
  actionUpdateFromNetworkErrorRetry(error, type) {
    var _this$state$networkEr;
    // "Is a network error"-boundary
    var retries = (_this$state$networkEr = this.state.networkErrorRetries.get(type)) !== null && _this$state$networkEr !== void 0 ? _this$state$networkEr : 0;

    // Strings for error logging
    var retryDurationString = this.networkErrorRetryMs / 1000 + "s";
    var retryCounterString = "(" + retries + "/" + this.maximumNetworkErrorRetryCount + ")";

    // Variables for scheduling the new event
    var retryDuration = this.networkErrorRetryMs;
    if (error instanceof Error && error.name === "AbortError") {
      // We do not wait for the timeout on local timeouts.
      retryDuration = 0;
      this.logger.warn("Network local timeout error while sending event, immediate retry (" + retryCounterString + ")", error);
    } else if (error instanceof Error && error.message.includes("updating delayed event")) {
      // TODO: We do not want error message matching here but instead the error should be a typed HTTPError
      // and be handled below automatically (the same as in the SPA case).
      //
      // The error originates because of https://github.com/matrix-org/matrix-widget-api/blob/5d81d4a26ff69e4bd3ddc79a884c9527999fb2f4/src/ClientWidgetApi.ts#L698-L701
      // uses `e` instance of HttpError (and not MatrixError)
      // The element web widget driver (only checks for MatrixError) is then failing to process (`processError`) it as a typed error: https://github.com/element-hq/element-web/blob/471712cbf06a067e5499bd5d2d7a75f693d9a12d/src/stores/widgets/StopGapWidgetDriver.ts#L711-L715
      // So it will not call: `error.asWidgetApiErrorData()` which is also missing for `HttpError`
      //
      // A proper fix would be to either find a place to convert the `HttpError` into a `MatrixError` and the `processError`
      // method to handle it as expected or to adjust `processError` to also process `HttpError`'s.
      this.logger.warn("delayed event update timeout error, retrying in " + retryDurationString + " " + retryCounterString, error);
    } else if (error instanceof ConnectionError) {
      this.logger.warn("Network connection error while sending event, retrying in " + retryDurationString + " " + retryCounterString, error);
    } else if ((error instanceof HTTPError || error instanceof MatrixError) && typeof error.httpStatus === "number" && error.httpStatus >= 500 && error.httpStatus < 600) {
      this.logger.warn("Server error while sending event, retrying in " + retryDurationString + " " + retryCounterString, error);
    } else {
      return undefined;
    }

    // retry boundary
    if (retries < this.maximumNetworkErrorRetryCount) {
      this.state.networkErrorRetries.set(type, retries + 1);
      return createInsertActionUpdate(type, retryDuration);
    }

    // Failure
    throw Error("Reached maximum (" + this.maximumNetworkErrorRetryCount + ") retries cause by: " + error);
  }

  /**
   * Check if its an UnsupportedDelayedEventsEndpointError and which implies that we cannot do any delayed event logic
   * @param error The error to check
   * @returns true it its an UnsupportedDelayedEventsEndpointError
   */
  isUnsupportedDelayedEndpoint(error) {
    return error instanceof UnsupportedDelayedEventsEndpointError;
  }
  resetRateLimitCounter(type) {
    this.state.rateLimitRetries.set(type, 0);
    this.state.networkErrorRetries.set(type, 0);
  }
  get status() {
    var actions = this.scheduler.actions;
    if (actions.length === 1) {
      var {
        type
      } = actions[0];
      switch (type) {
        case MembershipActionType.SendDelayedEvent:
        case MembershipActionType.SendJoinEvent:
          return Status.Connecting;
        case MembershipActionType.UpdateExpiry:
          // where no delayed events
          return Status.Connected;
        case MembershipActionType.SendScheduledDelayedLeaveEvent:
        case MembershipActionType.SendLeaveEvent:
          return Status.Disconnecting;
        default:
        // pass through as not expected
      }
    } else if (actions.length === 2) {
      var types = actions.map(a => a.type);
      // normal state for connected with delayed events
      if ((types.includes(MembershipActionType.RestartDelayedEvent) || types.includes(MembershipActionType.SendDelayedEvent) && this.state.hasMemberStateEvent) && types.includes(MembershipActionType.UpdateExpiry)) {
        return Status.Connected;
      }
    } else if (actions.length === 3) {
      var _types = actions.map(a => a.type);
      // It is a correct connected state if we already schedule the next Restart but have not yet cleaned up
      // the current restart.
      if (_types.filter(t => t === MembershipActionType.RestartDelayedEvent).length === 2 && _types.includes(MembershipActionType.UpdateExpiry)) {
        return Status.Connected;
      }
    }
    if (!this.scheduler.running) {
      return Status.Disconnected;
    }
    this.logger.error("MembershipManager has an unknown state. Actions: ", actions);
    return Status.Unknown;
  }
  get probablyLeft() {
    return this.state.probablyLeft;
  }
}
function createInsertActionUpdate(type, offset) {
  return {
    insert: [{
      ts: Date.now() + (offset !== null && offset !== void 0 ? offset : 0),
      type
    }]
  };
}
function createReplaceActionUpdate(type, offset) {
  return {
    replace: [{
      ts: Date.now() + (offset !== null && offset !== void 0 ? offset : 0),
      type
    }]
  };
}
//# sourceMappingURL=MembershipManager.js.map