import _asyncToGenerator from "@babel/runtime/helpers/asyncToGenerator";
import _defineProperty from "@babel/runtime/helpers/defineProperty";
import { logger as rootLogger } from "../logger.js";
import { secureRandomBase64Url } from "../randomstring.js";
import { decodeBase64, encodeUnpaddedBase64 } from "../base64.js";
import { safeGetRetryAfterMs } from "../http-api/errors.js";
import { KeyTransportEvents } from "./IKeyTransport.js";
import { isMyMembership } from "./types.js";
import { getParticipantId } from "./utils.js";
import { RoomAndToDeviceEvents, RoomAndToDeviceTransport } from "./RoomAndToDeviceKeyTransport.js";

/**
 * This interface is for testing and for making it possible to interchange the encryption manager.
 * @internal
 */

/**
 * This class implements the IEncryptionManager interface,
 * and takes care of managing the encryption keys of all rtc members:
 *  - generate new keys for the local user and send them to other participants
 *  - track all keys of all other members and update livekit.
 *
 * @internal
 */
export class EncryptionManager {
  get updateEncryptionKeyThrottle() {
    var _this$joinConfig$upda, _this$joinConfig;
    return (_this$joinConfig$upda = (_this$joinConfig = this.joinConfig) === null || _this$joinConfig === void 0 ? void 0 : _this$joinConfig.updateEncryptionKeyThrottle) !== null && _this$joinConfig$upda !== void 0 ? _this$joinConfig$upda : 3000;
  }
  get makeKeyDelay() {
    var _this$joinConfig$make, _this$joinConfig2;
    return (_this$joinConfig$make = (_this$joinConfig2 = this.joinConfig) === null || _this$joinConfig2 === void 0 ? void 0 : _this$joinConfig2.makeKeyDelay) !== null && _this$joinConfig$make !== void 0 ? _this$joinConfig$make : 3000;
  }
  get useKeyDelay() {
    var _this$joinConfig$useK, _this$joinConfig3;
    return (_this$joinConfig$useK = (_this$joinConfig3 = this.joinConfig) === null || _this$joinConfig3 === void 0 ? void 0 : _this$joinConfig3.useKeyDelay) !== null && _this$joinConfig$useK !== void 0 ? _this$joinConfig$useK : 5000;
  }
  constructor(userId, deviceId, getMemberships, transport, statistics, onEncryptionKeysChanged, parentLogger) {
    var _this = this;
    this.userId = userId;
    this.deviceId = deviceId;
    this.getMemberships = getMemberships;
    this.transport = transport;
    this.statistics = statistics;
    this.onEncryptionKeysChanged = onEncryptionKeysChanged;
    _defineProperty(this, "manageMediaKeys", false);
    _defineProperty(this, "keysEventUpdateTimeout", void 0);
    _defineProperty(this, "makeNewKeyTimeout", void 0);
    _defineProperty(this, "setNewKeyTimeouts", new Set());
    _defineProperty(this, "encryptionKeys", new Map());
    _defineProperty(this, "lastEncryptionKeyUpdateRequest", void 0);
    // We use this to store the last membership fingerprints we saw, so we can proactively re-send encryption keys
    // if it looks like a membership has been updated.
    _defineProperty(this, "lastMembershipFingerprints", void 0);
    _defineProperty(this, "latestGeneratedKeyIndex", -1);
    _defineProperty(this, "joinConfig", void 0);
    _defineProperty(this, "logger", void 0);
    _defineProperty(this, "joined", false);
    /**
     * Re-sends the encryption keys room event
     */
    _defineProperty(this, "sendEncryptionKeysEvent", /*#__PURE__*/function () {
      var _ref = _asyncToGenerator(function* (indexToSend) {
        if (_this.keysEventUpdateTimeout !== undefined) {
          clearTimeout(_this.keysEventUpdateTimeout);
          _this.keysEventUpdateTimeout = undefined;
        }
        _this.lastEncryptionKeyUpdateRequest = Date.now();
        if (!_this.joined) return;
        var myKeys = _this.getKeysForParticipant(_this.userId, _this.deviceId);
        if (!myKeys) {
          _this.logger.warn("Tried to send encryption keys event but no keys found!");
          return;
        }
        if (typeof indexToSend !== "number" && _this.latestGeneratedKeyIndex === -1) {
          _this.logger.warn("Tried to send encryption keys event but no current key index found!");
          return;
        }
        var keyIndexToSend = indexToSend !== null && indexToSend !== void 0 ? indexToSend : _this.latestGeneratedKeyIndex;
        _this.logger.info("Try sending encryption keys event. keyIndexToSend=".concat(keyIndexToSend, " (method parameter: ").concat(indexToSend, ")"));
        var keyToSend = myKeys[keyIndexToSend];
        try {
          _this.statistics.counters.roomEventEncryptionKeysSent += 1;
          var targets = _this.getMemberships().filter(membership => {
            return membership.sender != undefined;
          }).map(membership => {
            return {
              userId: membership.sender,
              deviceId: membership.deviceId,
              membershipTs: membership.createdTs()
            };
          });
          yield _this.transport.sendKey(encodeUnpaddedBase64(keyToSend), keyIndexToSend, targets);
          _this.logger.debug("sendEncryptionKeysEvent participantId=".concat(_this.userId, ":").concat(_this.deviceId, " numKeys=").concat(myKeys.length, " currentKeyIndex=").concat(_this.latestGeneratedKeyIndex, " keyIndexToSend=").concat(keyIndexToSend));
        } catch (error) {
          if (_this.keysEventUpdateTimeout === undefined) {
            var resendDelay = safeGetRetryAfterMs(error, 5000);
            _this.logger.warn("Failed to send m.call.encryption_key, retrying in ".concat(resendDelay), error);
            _this.keysEventUpdateTimeout = setTimeout(() => void _this.sendEncryptionKeysEvent(), resendDelay);
          } else {
            _this.logger.info("Not scheduling key resend as another re-send is already pending");
          }
        }
      });
      return function (_x) {
        return _ref.apply(this, arguments);
      };
    }());
    _defineProperty(this, "onTransportChanged", () => {
      this.requestSendCurrentKey();
    });
    _defineProperty(this, "onNewKeyReceived", (userId, deviceId, keyBase64Encoded, index, timestamp) => {
      this.logger.debug("Received key over key transport ".concat(userId, ":").concat(deviceId, " at index ").concat(index));
      this.setEncryptionKey(userId, deviceId, index, keyBase64Encoded, timestamp);
    });
    _defineProperty(this, "onRotateKeyTimeout", () => {
      if (!this.manageMediaKeys) return;
      this.makeNewKeyTimeout = undefined;
      this.logger.info("Making new sender key for key rotation");
      var newKeyIndex = this.makeNewSenderKey(true);
      // send immediately: if we're about to start sending with a new key, it's
      // important we get it out to others as soon as we can.
      void this.sendEncryptionKeysEvent(newKeyIndex);
    });
    this.logger = (parentLogger !== null && parentLogger !== void 0 ? parentLogger : rootLogger).getChild("[EncryptionManager]");
  }
  getEncryptionKeys() {
    var keysMap = new Map();
    for (var [_userId, userKeys] of this.encryptionKeys) {
      var keys = userKeys.map((entry, index) => ({
        key: entry.key,
        keyIndex: index
      }));
      keysMap.set(_userId, keys);
    }
    return keysMap;
  }
  join(joinConfig) {
    var _this$joinConfig$mana, _this$joinConfig4, _this$joinConfig5;
    this.joinConfig = joinConfig;
    this.joined = true;
    this.manageMediaKeys = (_this$joinConfig$mana = (_this$joinConfig4 = this.joinConfig) === null || _this$joinConfig4 === void 0 ? void 0 : _this$joinConfig4.manageMediaKeys) !== null && _this$joinConfig$mana !== void 0 ? _this$joinConfig$mana : this.manageMediaKeys;
    this.transport.on(KeyTransportEvents.ReceivedKeys, this.onNewKeyReceived);
    // Deprecate RoomKeyTransport: this can get removed.
    if (this.transport instanceof RoomAndToDeviceTransport) {
      this.transport.on(RoomAndToDeviceEvents.EnabledTransportsChanged, this.onTransportChanged);
    }
    this.transport.start();
    if ((_this$joinConfig5 = this.joinConfig) !== null && _this$joinConfig5 !== void 0 && _this$joinConfig5.manageMediaKeys) {
      this.makeNewSenderKey();
      this.requestSendCurrentKey();
    }
  }
  leave() {
    // clear our encryption keys as we're done with them now (we'll
    // make new keys if we rejoin). We leave keys for other participants
    // as they may still be using the same ones.
    this.encryptionKeys.set(getParticipantId(this.userId, this.deviceId), []);
    this.transport.off(KeyTransportEvents.ReceivedKeys, this.onNewKeyReceived);
    this.transport.stop();
    if (this.makeNewKeyTimeout !== undefined) {
      clearTimeout(this.makeNewKeyTimeout);
      this.makeNewKeyTimeout = undefined;
    }
    for (var t of this.setNewKeyTimeouts) {
      clearTimeout(t);
    }
    this.setNewKeyTimeouts.clear();
    this.manageMediaKeys = false;
    this.joined = false;
  }
  onMembershipsUpdate(oldMemberships) {
    if (this.manageMediaKeys && this.joined) {
      var oldMembershipIds = new Set(oldMemberships.filter(m => !isMyMembership(m, this.userId, this.deviceId)).map(getParticipantIdFromMembership));
      var newMembershipIds = new Set(this.getMemberships().filter(m => !isMyMembership(m, this.userId, this.deviceId)).map(getParticipantIdFromMembership));

      // We can use https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Set/symmetricDifference
      // for this once available
      var anyLeft = Array.from(oldMembershipIds).some(x => !newMembershipIds.has(x));
      var anyJoined = Array.from(newMembershipIds).some(x => !oldMembershipIds.has(x));
      var oldFingerprints = this.lastMembershipFingerprints;
      // always store the fingerprints of these latest memberships
      this.storeLastMembershipFingerprints();
      if (anyLeft) {
        if (this.makeNewKeyTimeout) {
          // existing rotation in progress, so let it complete
        } else {
          this.logger.debug("Member(s) have left: queueing sender key rotation");
          this.makeNewKeyTimeout = setTimeout(this.onRotateKeyTimeout, this.makeKeyDelay);
        }
      } else if (anyJoined) {
        this.logger.debug("New member(s) have joined: re-sending keys");
        this.requestSendCurrentKey();
      } else if (oldFingerprints) {
        // does it look like any of the members have updated their memberships?
        var newFingerprints = this.lastMembershipFingerprints;

        // We can use https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Set/symmetricDifference
        // for this once available
        var candidateUpdates = Array.from(oldFingerprints).some(x => !newFingerprints.has(x)) || Array.from(newFingerprints).some(x => !oldFingerprints.has(x));
        if (candidateUpdates) {
          this.logger.debug("Member(s) have updated/reconnected: re-sending keys to everyone");
          this.requestSendCurrentKey();
        }
      }
    }
  }

  /**
   * Generate a new sender key and add it at the next available index
   * @param delayBeforeUse - If true, wait for a short period before setting the key for the
   *                         media encryptor to use. If false, set the key immediately.
   * @returns The index of the new key
   */
  makeNewSenderKey() {
    var delayBeforeUse = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : false;
    var encryptionKey = secureRandomBase64Url(16);
    var encryptionKeyIndex = this.getNewEncryptionKeyIndex();
    this.logger.info("Generated new key at index " + encryptionKeyIndex);
    this.setEncryptionKey(this.userId, this.deviceId, encryptionKeyIndex, encryptionKey, Date.now(), delayBeforeUse);
    return encryptionKeyIndex;
  }

  /**
   * Requests that we resend our current keys to the room. May send a keys event immediately
   * or queue for alter if one has already been sent recently.
   */
  requestSendCurrentKey() {
    if (!this.manageMediaKeys) return;
    if (this.lastEncryptionKeyUpdateRequest && this.lastEncryptionKeyUpdateRequest + this.updateEncryptionKeyThrottle > Date.now()) {
      this.logger.info("Last encryption key event sent too recently: postponing");
      if (this.keysEventUpdateTimeout === undefined) {
        this.keysEventUpdateTimeout = setTimeout(() => void this.sendEncryptionKeysEvent(), this.updateEncryptionKeyThrottle);
      }
      return;
    }
    void this.sendEncryptionKeysEvent();
  }

  /**
   * Get the known encryption keys for a given participant device.
   *
   * @param userId the user ID of the participant
   * @param deviceId the device ID of the participant
   * @returns The encryption keys for the given participant, or undefined if they are not known.
   */
  getKeysForParticipant(userId, deviceId) {
    var _this$encryptionKeys$;
    return (_this$encryptionKeys$ = this.encryptionKeys.get(getParticipantId(userId, deviceId))) === null || _this$encryptionKeys$ === void 0 ? void 0 : _this$encryptionKeys$.map(entry => entry.key);
  }
  storeLastMembershipFingerprints() {
    this.lastMembershipFingerprints = new Set(this.getMemberships().filter(m => !isMyMembership(m, this.userId, this.deviceId)).map(m => "".concat(getParticipantIdFromMembership(m), ":").concat(m.createdTs())));
  }
  getNewEncryptionKeyIndex() {
    if (this.latestGeneratedKeyIndex === -1) {
      return 0;
    }

    // maximum key index is 255
    return (this.latestGeneratedKeyIndex + 1) % 256;
  }

  /**
   * Sets an encryption key at a specified index for a participant.
   * The encryption keys for the local participant are also stored here under the
   * user and device ID of the local participant.
   * If the key is older than the existing key at the index, it will be ignored.
   * @param userId - The user ID of the participant
   * @param deviceId - Device ID of the participant
   * @param encryptionKeyIndex - The index of the key to set
   * @param encryptionKeyString - The string representation of the key to set in base64
   * @param timestamp - The timestamp of the key. We assume that these are monotonic for each participant device.
   * @param delayBeforeUse - If true, delay before emitting a key changed event. Useful when setting
   *                         encryption keys for the local participant to allow time for the key to
   *                         be distributed.
   */
  setEncryptionKey(userId, deviceId, encryptionKeyIndex, encryptionKeyString, timestamp) {
    var delayBeforeUse = arguments.length > 5 && arguments[5] !== undefined ? arguments[5] : false;
    this.logger.debug("Setting encryption key for ".concat(userId, ":").concat(deviceId, " at index ").concat(encryptionKeyIndex));
    var keyBin = decodeBase64(encryptionKeyString);
    var participantId = getParticipantId(userId, deviceId);
    if (!this.encryptionKeys.has(participantId)) {
      this.encryptionKeys.set(participantId, []);
    }
    var participantKeys = this.encryptionKeys.get(participantId);
    var existingKeyAtIndex = participantKeys[encryptionKeyIndex];
    if (existingKeyAtIndex) {
      if (existingKeyAtIndex.timestamp > timestamp) {
        this.logger.info("Ignoring new key at index ".concat(encryptionKeyIndex, " for ").concat(participantId, " as it is older than existing known key"));
        return;
      }
      if (keysEqual(existingKeyAtIndex.key, keyBin)) {
        existingKeyAtIndex.timestamp = timestamp;
        return;
      }
    }
    if (userId === this.userId && deviceId === this.deviceId) {
      // It is important to already update the latestGeneratedKeyIndex here
      // NOT IN THE `delayBeforeUse` `setTimeout`.
      // Even though this is where we call onEncryptionKeysChanged and set the key in EC (and livekit).
      // It needs to happen here because we will send the key before the timeout has passed and sending
      // the key will use latestGeneratedKeyIndex as the index. if we update it in the `setTimeout` callback
      // it will use the wrong index (index - 1)!
      this.latestGeneratedKeyIndex = encryptionKeyIndex;
    }
    participantKeys[encryptionKeyIndex] = {
      key: keyBin,
      timestamp
    };
    if (delayBeforeUse) {
      var useKeyTimeout = setTimeout(() => {
        this.setNewKeyTimeouts.delete(useKeyTimeout);
        this.logger.info("Delayed-emitting key changed event for ".concat(participantId, " index ").concat(encryptionKeyIndex));
        this.onEncryptionKeysChanged(keyBin, encryptionKeyIndex, participantId);
      }, this.useKeyDelay);
      this.setNewKeyTimeouts.add(useKeyTimeout);
    } else {
      this.onEncryptionKeysChanged(keyBin, encryptionKeyIndex, participantId);
    }
  }
}
function keysEqual(a, b) {
  if (a === b) return true;
  return !!a && !!b && a.length === b.length && a.every((x, i) => x === b[i]);
}
var getParticipantIdFromMembership = m => getParticipantId(m.sender, m.deviceId);
//# sourceMappingURL=EncryptionManager.js.map