import { type Logger } from "../logger.ts";
import { type EncryptionConfig } from "./MatrixRTCSession.ts";
import { type CallMembership } from "./CallMembership.ts";
import { type KeyTransportEventListener, type IKeyTransport } from "./IKeyTransport.ts";
import { type ParticipantId, type Statistics } from "./types.ts";
/**
 * This interface is for testing and for making it possible to interchange the encryption manager.
 * @internal
 */
export interface IEncryptionManager {
    /**
     * Joins the encryption manager with the provided configuration.
     *
     * @param joinConfig - The configuration for joining encryption, or undefined
     * if no specific configuration is provided.
     */
    join(joinConfig: EncryptionConfig | undefined): void;
    /**
     * Leaves the encryption manager, cleaning up any associated resources.
     */
    leave(): void;
    /**
     * Called from the MatrixRTCSession when the memberships in this session updated.
     *
     * @param oldMemberships - The previous state of call memberships before the update.
     */
    onMembershipsUpdate(oldMemberships: CallMembership[]): void;
    /**
     * Retrieves the encryption keys currently managed by the encryption manager.
     *
     * @returns A map of participant IDs to their encryption keys.
     */
    getEncryptionKeys(): ReadonlyMap<ParticipantId, ReadonlyArray<{
        key: Uint8Array;
        keyIndex: number;
    }>>;
}
/**
 * This class implements the IEncryptionManager interface,
 * and takes care of managing the encryption keys of all rtc members:
 *  - generate new keys for the local user and send them to other participants
 *  - track all keys of all other members and update livekit.
 *
 * @internal
 */
export declare class EncryptionManager implements IEncryptionManager {
    private userId;
    private deviceId;
    private getMemberships;
    private transport;
    private statistics;
    private onEncryptionKeysChanged;
    private manageMediaKeys;
    private keysEventUpdateTimeout?;
    private makeNewKeyTimeout?;
    private setNewKeyTimeouts;
    private get updateEncryptionKeyThrottle();
    private get makeKeyDelay();
    private get useKeyDelay();
    private encryptionKeys;
    private lastEncryptionKeyUpdateRequest?;
    private lastMembershipFingerprints;
    private latestGeneratedKeyIndex;
    private joinConfig;
    private logger;
    constructor(userId: string, deviceId: string, getMemberships: () => CallMembership[], transport: IKeyTransport, statistics: Statistics, onEncryptionKeysChanged: (keyBin: Uint8Array, encryptionKeyIndex: number, participantId: string) => void, parentLogger?: Logger);
    getEncryptionKeys(): ReadonlyMap<ParticipantId, ReadonlyArray<{
        key: Uint8Array;
        keyIndex: number;
    }>>;
    private joined;
    join(joinConfig: EncryptionConfig): void;
    leave(): void;
    onMembershipsUpdate(oldMemberships: CallMembership[]): void;
    /**
     * Generate a new sender key and add it at the next available index
     * @param delayBeforeUse - If true, wait for a short period before setting the key for the
     *                         media encryptor to use. If false, set the key immediately.
     * @returns The index of the new key
     */
    private makeNewSenderKey;
    /**
     * Requests that we resend our current keys to the room. May send a keys event immediately
     * or queue for alter if one has already been sent recently.
     */
    private requestSendCurrentKey;
    /**
     * Get the known encryption keys for a given participant device.
     *
     * @param userId the user ID of the participant
     * @param deviceId the device ID of the participant
     * @returns The encryption keys for the given participant, or undefined if they are not known.
     */
    private getKeysForParticipant;
    /**
     * Re-sends the encryption keys room event
     */
    private sendEncryptionKeysEvent;
    private onTransportChanged;
    onNewKeyReceived: KeyTransportEventListener;
    private storeLastMembershipFingerprints;
    private getNewEncryptionKeyIndex;
    /**
     * Sets an encryption key at a specified index for a participant.
     * The encryption keys for the local participant are also stored here under the
     * user and device ID of the local participant.
     * If the key is older than the existing key at the index, it will be ignored.
     * @param userId - The user ID of the participant
     * @param deviceId - Device ID of the participant
     * @param encryptionKeyIndex - The index of the key to set
     * @param encryptionKeyString - The string representation of the key to set in base64
     * @param timestamp - The timestamp of the key. We assume that these are monotonic for each participant device.
     * @param delayBeforeUse - If true, delay before emitting a key changed event. Useful when setting
     *                         encryption keys for the local participant to allow time for the key to
     *                         be distributed.
     */
    private setEncryptionKey;
    private onRotateKeyTimeout;
}
//# sourceMappingURL=EncryptionManager.d.ts.map