import _asyncToGenerator from "@babel/runtime/helpers/asyncToGenerator";
import _defineProperty from "@babel/runtime/helpers/defineProperty";
function ownKeys(e, r) { var t = Object.keys(e); if (Object.getOwnPropertySymbols) { var o = Object.getOwnPropertySymbols(e); r && (o = o.filter(function (r) { return Object.getOwnPropertyDescriptor(e, r).enumerable; })), t.push.apply(t, o); } return t; }
function _objectSpread(e) { for (var r = 1; r < arguments.length; r++) { var t = null != arguments[r] ? arguments[r] : {}; r % 2 ? ownKeys(Object(t), !0).forEach(function (r) { _defineProperty(e, r, t[r]); }) : Object.getOwnPropertyDescriptors ? Object.defineProperties(e, Object.getOwnPropertyDescriptors(t)) : ownKeys(Object(t)).forEach(function (r) { Object.defineProperty(e, r, Object.getOwnPropertyDescriptor(t, r)); }); } return e; }
/*
Copyright 2022 The Matrix.org Foundation C.I.C.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

/**
 * This is an internal module. See {@link MatrixHttpApi} for the public class.
 */

import { checkObjectHasKeys, deepCopy, encodeParams } from "../utils.js";
import { Method } from "./method.js";
import { ConnectionError, MatrixError, TokenRefreshError } from "./errors.js";
import { HttpApiEvent } from "./interface.js";
import { anySignal, parseErrorResponse, timeoutSignal } from "./utils.js";
import { TokenRefresher, TokenRefreshOutcome } from "./refresh.js";
export class FetchHttpApi {
  constructor(eventEmitter, opts) {
    var _opts$useAuthorizatio;
    this.eventEmitter = eventEmitter;
    this.opts = opts;
    _defineProperty(this, "abortController", new AbortController());
    _defineProperty(this, "tokenRefresher", void 0);
    checkObjectHasKeys(opts, ["baseUrl", "prefix"]);
    if (!opts.onlyData) {
      throw new Error("Constructing FetchHttpApi without `onlyData=true` is no longer supported.");
    }
    opts.useAuthorizationHeader = (_opts$useAuthorizatio = opts.useAuthorizationHeader) !== null && _opts$useAuthorizatio !== void 0 ? _opts$useAuthorizatio : true;
    this.tokenRefresher = new TokenRefresher(opts);
  }
  abort() {
    this.abortController.abort();
    this.abortController = new AbortController();
  }
  fetch(resource, options) {
    if (this.opts.fetchFn) {
      return this.opts.fetchFn(resource, options);
    }
    return globalThis.fetch(resource, options);
  }

  /**
   * Sets the base URL for the identity server
   * @param url - The new base url
   */
  setIdBaseUrl(url) {
    this.opts.idBaseUrl = url;
  }
  idServerRequest(method, path, params, prefix, accessToken) {
    if (!this.opts.idBaseUrl) {
      throw new Error("No identity server base URL set");
    }
    var queryParams = undefined;
    var body = undefined;
    if (method === Method.Get) {
      queryParams = params;
    } else {
      body = params;
    }
    var fullUri = this.getUrl(path, queryParams, prefix, this.opts.idBaseUrl);
    var opts = {
      json: true,
      headers: {}
    };
    if (accessToken) {
      opts.headers.Authorization = "Bearer ".concat(accessToken);
    }
    return this.requestOtherUrl(method, fullUri, body, opts);
  }

  /**
   * Perform an authorised request to the homeserver.
   * @param method - The HTTP method e.g. "GET".
   * @param path - The HTTP path <b>after</b> the supplied prefix e.g.
   * "/createRoom".
   *
   * @param queryParams - A dict of query params (these will NOT be
   * urlencoded). If unspecified, there will be no query params.
   *
   * @param body - The HTTP JSON body.
   *
   * @param paramOpts - additional options.
   * When `paramOpts.doNotAttemptTokenRefresh` is true, token refresh will not be attempted
   * when an expired token is encountered. Used to only attempt token refresh once.
   *
   * @returns The parsed response.
   * @throws Error if a problem occurred. This includes network problems and Matrix-specific error JSON.
   */
  authedRequest(method, path) {
    var queryParams = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
    var body = arguments.length > 3 ? arguments[3] : undefined;
    var paramOpts = arguments.length > 4 && arguments[4] !== undefined ? arguments[4] : {};
    return this.doAuthedRequest(1, method, path, queryParams, body, paramOpts);
  }

  // Wrapper around public method authedRequest to allow for tracking retry attempt counts
  doAuthedRequest(attempt, method, path, queryParams, body) {
    var _arguments = arguments,
      _this = this;
    return _asyncToGenerator(function* () {
      var paramOpts = _arguments.length > 5 && _arguments[5] !== undefined ? _arguments[5] : {};
      // avoid mutating paramOpts so they can be used on retry
      var opts = deepCopy(paramOpts);
      // we have to manually copy the abortSignal over as it is not a plain object
      opts.abortSignal = paramOpts.abortSignal;

      // Take a snapshot of the current token state before we start the request so we can reference it if we error
      var requestSnapshot = yield _this.tokenRefresher.prepareForRequest();
      if (requestSnapshot.accessToken) {
        if (_this.opts.useAuthorizationHeader) {
          if (!opts.headers) {
            opts.headers = {};
          }
          if (!opts.headers.Authorization) {
            opts.headers.Authorization = "Bearer ".concat(requestSnapshot.accessToken);
          }
          if (queryParams.access_token) {
            delete queryParams.access_token;
          }
        } else if (!queryParams.access_token) {
          queryParams.access_token = requestSnapshot.accessToken;
        }
      }
      try {
        var response = yield _this.request(method, path, queryParams, body, opts);
        return response;
      } catch (error) {
        if (!(error instanceof MatrixError)) {
          throw error;
        }
        if (error.errcode === "M_UNKNOWN_TOKEN") {
          var outcome = yield _this.tokenRefresher.handleUnknownToken(requestSnapshot, attempt);
          if (outcome === TokenRefreshOutcome.Success) {
            // if we got a new token retry the request
            return _this.doAuthedRequest(attempt + 1, method, path, queryParams, body, paramOpts);
          }
          if (outcome === TokenRefreshOutcome.Failure) {
            throw new TokenRefreshError(error);
          }
          if (!(opts !== null && opts !== void 0 && opts.inhibitLogoutEmit)) {
            _this.eventEmitter.emit(HttpApiEvent.SessionLoggedOut, error);
          }
        } else if (error.errcode == "M_CONSENT_NOT_GIVEN") {
          _this.eventEmitter.emit(HttpApiEvent.NoConsent, error.message, error.data.consent_uri);
        }
        throw error;
      }
    })();
  }

  /**
   * Perform a request to the homeserver without any credentials.
   * @param method - The HTTP method e.g. "GET".
   * @param path - The HTTP path <b>after</b> the supplied prefix e.g.
   * "/createRoom".
   *
   * @param queryParams - A dict of query params (these will NOT be
   * urlencoded). If unspecified, there will be no query params.
   *
   * @param body - The HTTP JSON body.
   *
   * @param opts - additional options
   *
   * @returns The parsed response.
   * @throws Error if a problem occurred. This includes network problems and Matrix-specific error JSON.
   */
  request(method, path, queryParams, body, opts) {
    var fullUri = this.getUrl(path, queryParams, opts === null || opts === void 0 ? void 0 : opts.prefix, opts === null || opts === void 0 ? void 0 : opts.baseUrl);
    return this.requestOtherUrl(method, fullUri, body, opts);
  }

  /**
   * Perform a request to an arbitrary URL.
   * @param method - The HTTP method e.g. "GET".
   * @param url - The HTTP URL object.
   *
   * @param body - The HTTP JSON body.
   *
   * @param opts - additional options
   *
   * @returns The parsed response.
   * @throws Error if a problem occurred. This includes network problems and Matrix-specific error JSON.
   */
  requestOtherUrl(method, url, body) {
    var _arguments2 = arguments,
      _this2 = this;
    return _asyncToGenerator(function* () {
      var _this2$opts$logger, _opts$localTimeoutMs, _opts$keepAlive, _body$constructor;
      var opts = _arguments2.length > 3 && _arguments2[3] !== undefined ? _arguments2[3] : {};
      if (opts.json !== undefined && opts.rawResponseBody !== undefined) {
        throw new Error("Invalid call to `FetchHttpApi` sets both `opts.json` and `opts.rawResponseBody`");
      }
      var urlForLogs = _this2.sanitizeUrlForLogs(url);
      (_this2$opts$logger = _this2.opts.logger) === null || _this2$opts$logger === void 0 || _this2$opts$logger.debug("FetchHttpApi: --> ".concat(method, " ").concat(urlForLogs));
      var headers = Object.assign({}, opts.headers || {});
      var jsonResponse = !opts.rawResponseBody && opts.json !== false;
      if (jsonResponse) {
        if (!headers["Accept"]) {
          headers["Accept"] = "application/json";
        }
      }
      var timeout = (_opts$localTimeoutMs = opts.localTimeoutMs) !== null && _opts$localTimeoutMs !== void 0 ? _opts$localTimeoutMs : _this2.opts.localTimeoutMs;
      var keepAlive = (_opts$keepAlive = opts.keepAlive) !== null && _opts$keepAlive !== void 0 ? _opts$keepAlive : false;
      var signals = [_this2.abortController.signal];
      if (timeout !== undefined) {
        signals.push(timeoutSignal(timeout));
      }
      if (opts.abortSignal) {
        signals.push(opts.abortSignal);
      }

      // If the body is an object, encode it as JSON and set the `Content-Type` header,
      // unless that has been explicitly inhibited by setting `opts.json: false`.
      // We can't use getPrototypeOf here as objects made in other contexts e.g. over postMessage won't have same ref
      var data;
      if (opts.json !== false && (body === null || body === void 0 || (_body$constructor = body.constructor) === null || _body$constructor === void 0 ? void 0 : _body$constructor.name) === Object.name) {
        data = JSON.stringify(body);
        if (!headers["Content-Type"]) {
          headers["Content-Type"] = "application/json";
        }
      } else {
        data = body;
      }
      var {
        signal,
        cleanup
      } = anySignal(signals);
      var res;
      var start = Date.now();
      try {
        var _this2$opts$logger2;
        res = yield _this2.fetch(url, {
          signal,
          method,
          body: data,
          headers,
          mode: "cors",
          redirect: "follow",
          referrer: "",
          referrerPolicy: "no-referrer",
          cache: "no-cache",
          credentials: "omit",
          // we send credentials via headers
          keepalive: keepAlive,
          priority: opts.priority
        });
        (_this2$opts$logger2 = _this2.opts.logger) === null || _this2$opts$logger2 === void 0 || _this2$opts$logger2.debug("FetchHttpApi: <-- ".concat(method, " ").concat(urlForLogs, " [").concat(Date.now() - start, "ms ").concat(res.status, "]"));
      } catch (e) {
        var _this2$opts$logger3;
        (_this2$opts$logger3 = _this2.opts.logger) === null || _this2$opts$logger3 === void 0 || _this2$opts$logger3.debug("FetchHttpApi: <-- ".concat(method, " ").concat(urlForLogs, " [").concat(Date.now() - start, "ms ").concat(e, "]"));
        if (e.name === "AbortError") {
          throw e;
        }
        throw new ConnectionError("fetch failed", e);
      } finally {
        cleanup();
      }
      if (!res.ok) {
        throw parseErrorResponse(res, yield res.text());
      }
      if (opts.rawResponseBody) {
        return yield res.blob();
      } else if (jsonResponse) {
        return yield res.json();
      } else {
        return yield res.text();
      }
    })();
  }
  sanitizeUrlForLogs(url) {
    try {
      var asUrl;
      if (typeof url === "string") {
        asUrl = new URL(url);
      } else {
        asUrl = url;
      }
      // Remove the values of any URL params that could contain potential secrets
      var sanitizedQs = new URLSearchParams();
      for (var key of asUrl.searchParams.keys()) {
        sanitizedQs.append(key, "xxx");
      }
      var sanitizedQsString = sanitizedQs.toString();
      var sanitizedQsUrlPiece = sanitizedQsString ? "?".concat(sanitizedQsString) : "";
      return asUrl.origin + asUrl.pathname + sanitizedQsUrlPiece;
    } catch (_unused) {
      // defensive coding for malformed url
      return "??";
    }
  }
  /**
   * Form and return a homeserver request URL based on the given path params and prefix.
   * @param path - The HTTP path <b>after</b> the supplied prefix e.g. "/createRoom".
   * @param queryParams - A dict of query params (these will NOT be urlencoded).
   * @param prefix - The full prefix to use e.g. "/_matrix/client/v2_alpha", defaulting to this.opts.prefix.
   * @param baseUrl - The baseUrl to use e.g. "https://matrix.org", defaulting to this.opts.baseUrl.
   * @returns URL
   */
  getUrl(path, queryParams, prefix, baseUrl) {
    var baseUrlWithFallback = baseUrl !== null && baseUrl !== void 0 ? baseUrl : this.opts.baseUrl;
    var baseUrlWithoutTrailingSlash = baseUrlWithFallback.endsWith("/") ? baseUrlWithFallback.slice(0, -1) : baseUrlWithFallback;
    var url = new URL(baseUrlWithoutTrailingSlash + (prefix !== null && prefix !== void 0 ? prefix : this.opts.prefix) + path);
    // If there are any params, encode and append them to the URL.
    if (this.opts.extraParams || queryParams) {
      var mergedParams = _objectSpread(_objectSpread({}, this.opts.extraParams), queryParams);
      encodeParams(mergedParams, url.searchParams);
    }
    return url;
  }
}
//# sourceMappingURL=fetch.js.map