import _defineProperty from "@babel/runtime/helpers/defineProperty";
/*
Copyright 2023 The Matrix.org Foundation C.I.C.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

/**
 * `matrix-js-sdk/lib/crypto-api`: End-to-end encryption support.
 *
 * The most important type is {@link CryptoApi}, an instance of which can be retrieved via
 * {@link MatrixClient.getCrypto}.
 *
 * @packageDocumentation
 */

/**
 * The options to start device dehydration.
 */

/**
 * Public interface to the cryptography parts of the js-sdk
 *
 * @remarks Currently, this is a work-in-progress. In time, more methods will be added here.
 */

/** A reason code for a failure to decrypt an event. */
export var DecryptionFailureCode = /*#__PURE__*/function (DecryptionFailureCode) {
  /** Message was encrypted with a Megolm session whose keys have not been shared with us. */
  DecryptionFailureCode["MEGOLM_UNKNOWN_INBOUND_SESSION_ID"] = "MEGOLM_UNKNOWN_INBOUND_SESSION_ID";
  /** A special case of {@link MEGOLM_UNKNOWN_INBOUND_SESSION_ID}: the sender has told us it is withholding the key. */
  DecryptionFailureCode["MEGOLM_KEY_WITHHELD"] = "MEGOLM_KEY_WITHHELD";
  /** A special case of {@link MEGOLM_KEY_WITHHELD}: the sender has told us it is withholding the key, because the current device is unverified. */
  DecryptionFailureCode["MEGOLM_KEY_WITHHELD_FOR_UNVERIFIED_DEVICE"] = "MEGOLM_KEY_WITHHELD_FOR_UNVERIFIED_DEVICE";
  /** Message was encrypted with a Megolm session which has been shared with us, but in a later ratchet state. */
  DecryptionFailureCode["OLM_UNKNOWN_MESSAGE_INDEX"] = "OLM_UNKNOWN_MESSAGE_INDEX";
  /**
   * Message was sent before the current device was created; there is no key backup on the server, so this
   * decryption failure is expected.
   */
  DecryptionFailureCode["HISTORICAL_MESSAGE_NO_KEY_BACKUP"] = "HISTORICAL_MESSAGE_NO_KEY_BACKUP";
  /**
   * Message was sent before the current device was created; there was a key backup on the server, but we don't
   * seem to have access to the backup. (Probably we don't have the right key.)
   */
  DecryptionFailureCode["HISTORICAL_MESSAGE_BACKUP_UNCONFIGURED"] = "HISTORICAL_MESSAGE_BACKUP_UNCONFIGURED";
  /**
   * Message was sent before the current device was created; there was a (usable) key backup on the server, but we
   * still can't decrypt. (Either the session isn't in the backup, or we just haven't gotten around to checking yet.)
   */
  DecryptionFailureCode["HISTORICAL_MESSAGE_WORKING_BACKUP"] = "HISTORICAL_MESSAGE_WORKING_BACKUP";
  /**
   * Message was sent when the user was not a member of the room.
   */
  DecryptionFailureCode["HISTORICAL_MESSAGE_USER_NOT_JOINED"] = "HISTORICAL_MESSAGE_USER_NOT_JOINED";
  /**
   * The sender's identity is not verified, but was previously verified.
   */
  DecryptionFailureCode["SENDER_IDENTITY_PREVIOUSLY_VERIFIED"] = "SENDER_IDENTITY_PREVIOUSLY_VERIFIED";
  /**
   * The sender device is not cross-signed.  This will only be used if the
   * device isolation mode is set to `OnlySignedDevicesIsolationMode`.
   */
  DecryptionFailureCode["UNSIGNED_SENDER_DEVICE"] = "UNSIGNED_SENDER_DEVICE";
  /**
   * We weren't able to link the message back to any known device.  This will
   * only be used if the device isolation mode is set to `OnlySignedDevicesIsolationMode`.
   */
  DecryptionFailureCode["UNKNOWN_SENDER_DEVICE"] = "UNKNOWN_SENDER_DEVICE";
  /** Unknown or unclassified error. */
  DecryptionFailureCode["UNKNOWN_ERROR"] = "UNKNOWN_ERROR";
  return DecryptionFailureCode;
}({});

/** Base {@link DeviceIsolationMode} kind. */
export var DeviceIsolationModeKind = /*#__PURE__*/function (DeviceIsolationModeKind) {
  DeviceIsolationModeKind[DeviceIsolationModeKind["AllDevicesIsolationMode"] = 0] = "AllDevicesIsolationMode";
  DeviceIsolationModeKind[DeviceIsolationModeKind["OnlySignedDevicesIsolationMode"] = 1] = "OnlySignedDevicesIsolationMode";
  return DeviceIsolationModeKind;
}({});

/**
 * A type of {@link DeviceIsolationMode}.
 *
 * Message encryption keys are shared with all devices in the room, except in case of
 * verified user problems (see {@link errorOnVerifiedUserProblems}).
 *
 * Events from all senders are always decrypted (and should be decorated with message shields in case
 * of authenticity warnings, see {@link EventEncryptionInfo}).
 */
export class AllDevicesIsolationMode {
  /**
   *
   * @param errorOnVerifiedUserProblems - Behavior when sharing keys to remote devices.
   *
   * If set to `true`, sharing keys will fail (i.e. message sending will fail) with an error if:
   *   - The user was previously verified but is not anymore, or:
   *   - A verified user has some unverified devices (not cross-signed).
   *
   * If `false`, the keys will be distributed as usual. In this case, the client UX should display
   * warnings to inform the user about problematic devices/users, and stop them hitting this case.
   */
  constructor(errorOnVerifiedUserProblems) {
    this.errorOnVerifiedUserProblems = errorOnVerifiedUserProblems;
    _defineProperty(this, "kind", DeviceIsolationModeKind.AllDevicesIsolationMode);
  }
}

/**
 * A type of {@link DeviceIsolationMode}.
 *
 * Message encryption keys are only shared with devices that have been cross-signed by their owner.
 * Encryption will throw an error if a verified user replaces their identity.
 *
 * Events are decrypted only if they come from a cross-signed device. Other events will result in a decryption
 * failure. (To access the failure reason, see {@link MatrixEvent.decryptionFailureReason}.)
 */
export class OnlySignedDevicesIsolationMode {
  constructor() {
    _defineProperty(this, "kind", DeviceIsolationModeKind.OnlySignedDevicesIsolationMode);
  }
}

/**
 * DeviceIsolationMode represents the mode of device isolation used when encrypting or decrypting messages.
 * It can be one of two types: {@link AllDevicesIsolationMode} or {@link OnlySignedDevicesIsolationMode}.
 *
 * Only supported by rust Crypto.
 */

/**
 * Options object for `CryptoApi.bootstrapCrossSigning`.
 */

/**
 * Represents the ways in which we trust a user
 */
export class UserVerificationStatus {
  constructor(crossSigningVerified, crossSigningVerifiedBefore, tofu) {
    var needsUserApproval = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : false;
    this.crossSigningVerified = crossSigningVerified;
    this.crossSigningVerifiedBefore = crossSigningVerifiedBefore;
    this.tofu = tofu;
    /**
     * Indicates if the identity has changed in a way that needs user approval.
     *
     * This happens if the identity has changed since we first saw it, *unless* the new identity has also been verified
     * by our user (eg via an interactive verification).
     *
     * To rectify this, either:
     *
     *  * Conduct a verification of the new identity via {@link CryptoApi.requestVerificationDM}.
     *  * Pin the new identity, via {@link CryptoApi.pinCurrentUserIdentity}.
     *
     * @returns true if the identity has changed in a way that needs user approval.
     */
    _defineProperty(this, "needsUserApproval", void 0);
    this.needsUserApproval = needsUserApproval;
  }

  /**
   * @returns true if this user is verified via any means
   */
  isVerified() {
    return this.isCrossSigningVerified();
  }

  /**
   * @returns true if this user is verified via cross signing
   */
  isCrossSigningVerified() {
    return this.crossSigningVerified;
  }

  /**
   * @returns true if we ever verified this user before (at least for
   * the history of verifications observed by this device).
   */
  wasCrossSigningVerified() {
    return this.crossSigningVerifiedBefore;
  }

  /**
   * @returns true if this user's key is trusted on first use
   *
   * @deprecated No longer supported, with the Rust crypto stack.
   */
  isTofu() {
    return this.tofu;
  }
}
export class DeviceVerificationStatus {
  constructor(opts) {
    var _opts$signedByOwner, _opts$crossSigningVer, _opts$tofu, _opts$localVerified, _opts$trustCrossSigne;
    /**
     * True if this device has been signed by its owner (and that signature verified).
     *
     * This doesn't necessarily mean that we have verified the device, since we may not have verified the
     * owner's cross-signing key.
     */
    _defineProperty(this, "signedByOwner", void 0);
    /**
     * True if this device has been verified via cross signing.
     *
     * This does *not* take into account `trustCrossSignedDevices`.
     */
    _defineProperty(this, "crossSigningVerified", void 0);
    /**
     * TODO: tofu magic wtf does this do?
     */
    _defineProperty(this, "tofu", void 0);
    /**
     * True if the device has been marked as locally verified.
     */
    _defineProperty(this, "localVerified", void 0);
    /**
     * True if the client has been configured to trust cross-signed devices via {@link CryptoApi#setTrustCrossSignedDevices}.
     */
    _defineProperty(this, "trustCrossSignedDevices", void 0);
    this.signedByOwner = (_opts$signedByOwner = opts.signedByOwner) !== null && _opts$signedByOwner !== void 0 ? _opts$signedByOwner : false;
    this.crossSigningVerified = (_opts$crossSigningVer = opts.crossSigningVerified) !== null && _opts$crossSigningVer !== void 0 ? _opts$crossSigningVer : false;
    this.tofu = (_opts$tofu = opts.tofu) !== null && _opts$tofu !== void 0 ? _opts$tofu : false;
    this.localVerified = (_opts$localVerified = opts.localVerified) !== null && _opts$localVerified !== void 0 ? _opts$localVerified : false;
    this.trustCrossSignedDevices = (_opts$trustCrossSigne = opts.trustCrossSignedDevices) !== null && _opts$trustCrossSigne !== void 0 ? _opts$trustCrossSigne : false;
  }

  /**
   * Check if we should consider this device "verified".
   *
   * A device is "verified" if either:
   *  * it has been manually marked as such via {@link CryptoApi.setDeviceVerified}.
   *  * it has been cross-signed with a verified signing key, **and** the client has been configured to trust
   *    cross-signed devices via {@link CryptoApi.setTrustCrossSignedDevices}.
   *
   * @returns true if this device is verified via any means.
   */
  isVerified() {
    return this.localVerified || this.trustCrossSignedDevices && this.crossSigningVerified;
  }
}

/**
 * Enum representing the different stages of importing room keys.
 *
 * This is the type of the `stage` property of {@link ImportRoomKeyProgressData}.
 */
export var ImportRoomKeyStage = /*#__PURE__*/function (ImportRoomKeyStage) {
  /**
   * The stage where room keys are being fetched.
   *
   * @see {@link ImportRoomKeyFetchProgress}.
   */
  ImportRoomKeyStage["Fetch"] = "fetch";
  /**
   * The stage where room keys are being loaded.
   *
   * @see {@link ImportRoomKeyLoadProgress}.
   */
  ImportRoomKeyStage["LoadKeys"] = "load_keys";
  return ImportRoomKeyStage;
}({});

/**
 * Type representing the progress during the 'fetch' stage of the room key import process.
 *
 * @see {@link ImportRoomKeyProgressData}.
 */

/**
 * Type representing the progress during the 'load_keys' stage of the room key import process.
 *
 * @see {@link ImportRoomKeyProgressData}.
 */

/**
 * Room key import progress report.
 * Used when calling {@link CryptoApi#importRoomKeys},
 * {@link CryptoApi#importRoomKeysAsJson} or {@link CryptoApi#restoreKeyBackup} as the parameter of
 * the progressCallback. Used to display feedback.
 */

/**
 * Options object for {@link CryptoApi#importRoomKeys} and
 * {@link CryptoApi#importRoomKeysAsJson}.
 */

/**
 * The result of a call to {@link CryptoApi.getCrossSigningStatus}.
 */

/**
 * Crypto callbacks provided by the application
 */

/**
 * Parameter of {@link CryptoApi#bootstrapSecretStorage}
 */

/** Types of cross-signing key */
export var CrossSigningKey = /*#__PURE__*/function (CrossSigningKey) {
  CrossSigningKey["Master"] = "master";
  CrossSigningKey["SelfSigning"] = "self_signing";
  CrossSigningKey["UserSigning"] = "user_signing";
  return CrossSigningKey;
}({});

/**
 * Information on one of the cross-signing keys.
 * @see https://spec.matrix.org/v1.7/client-server-api/#post_matrixclientv3keysdevice_signingupload
 */

/**
 * Recovery key created by {@link CryptoApi#createRecoveryKeyFromPassphrase} or {@link CreateSecretStorageOpts#createSecretStorageKey}.
 */

/**
 *  Result type of {@link CryptoApi#getEncryptionInfoForEvent}.
 */

/**
 * Types of shield to be shown for {@link EventEncryptionInfo#shieldColour}.
 */
export var EventShieldColour = /*#__PURE__*/function (EventShieldColour) {
  EventShieldColour[EventShieldColour["NONE"] = 0] = "NONE";
  EventShieldColour[EventShieldColour["GREY"] = 1] = "GREY";
  EventShieldColour[EventShieldColour["RED"] = 2] = "RED";
  return EventShieldColour;
}({});

/**
 * Reason codes for {@link EventEncryptionInfo#shieldReason}.
 */
export var EventShieldReason = /*#__PURE__*/function (EventShieldReason) {
  /** An unknown reason from the crypto library (if you see this, it is a bug in matrix-js-sdk). */
  EventShieldReason[EventShieldReason["UNKNOWN"] = 0] = "UNKNOWN";
  /** "Encrypted by an unverified user." */
  EventShieldReason[EventShieldReason["UNVERIFIED_IDENTITY"] = 1] = "UNVERIFIED_IDENTITY";
  /** "Encrypted by a device not verified by its owner." */
  EventShieldReason[EventShieldReason["UNSIGNED_DEVICE"] = 2] = "UNSIGNED_DEVICE";
  /** "Encrypted by an unknown or deleted device." */
  EventShieldReason[EventShieldReason["UNKNOWN_DEVICE"] = 3] = "UNKNOWN_DEVICE";
  /**
   * "The authenticity of this encrypted message can't be guaranteed on this device."
   *
   * ie: the key has been forwarded, or retrieved from an insecure backup.
   */
  EventShieldReason[EventShieldReason["AUTHENTICITY_NOT_GUARANTEED"] = 4] = "AUTHENTICITY_NOT_GUARANTEED";
  /**
   * The (deprecated) sender_key field in the event does not match the Ed25519 key of the device that sent us the
   * decryption keys.
   */
  EventShieldReason[EventShieldReason["MISMATCHED_SENDER_KEY"] = 5] = "MISMATCHED_SENDER_KEY";
  /**
   * The event was sent unencrypted in an encrypted room.
   */
  EventShieldReason[EventShieldReason["SENT_IN_CLEAR"] = 6] = "SENT_IN_CLEAR";
  /**
   * The sender was previously verified but changed their identity.
   */
  EventShieldReason[EventShieldReason["VERIFICATION_VIOLATION"] = 7] = "VERIFICATION_VIOLATION";
  /**
   * The `sender` field on the event does not match the owner of the device
   * that established the Megolm session.
   */
  EventShieldReason[EventShieldReason["MISMATCHED_SENDER"] = 8] = "MISMATCHED_SENDER";
  return EventShieldReason;
}({});

/** The result of a call to {@link CryptoApi.getOwnDeviceKeys} */

/**
 * Information about the encryption of a successfully decrypted to-device message.
 */

export * from "./verification.js";
export * from "./recovery-key.js";
export * from "./key-passphrase.js";
export * from "./CryptoEvent.js";
//# sourceMappingURL=index.js.map