/**
 * This is an internal module. See {@link MatrixClient} for the public class.
 */
import { type Optional } from "matrix-events-sdk";
import type { IDeviceKeys, IOneTimeKey } from "./@types/crypto.ts";
import { type ISyncStateData, type SetPresence, SyncApi, type SyncApiOptions, SyncState } from "./sync.ts";
import { EventStatus, type IDecryptOptions, type IEvent, MatrixEvent, MatrixEventEvent, type MatrixEventHandlerMap, type PushDetails } from "./models/event.ts";
import { type CallEvent, type CallEventHandlerMap, type MatrixCall } from "./webrtc/call.ts";
import { Filter, type IFilterDefinition } from "./filter.ts";
import { CallEventHandler, type CallEventHandlerEvent, type CallEventHandlerEventHandlerMap } from "./webrtc/callEventHandler.ts";
import { GroupCallEventHandler, type GroupCallEventHandlerEvent, type GroupCallEventHandlerEventHandlerMap } from "./webrtc/groupCallEventHandler.ts";
import { type QueryDict } from "./utils.ts";
import { Direction, EventTimeline } from "./models/event-timeline.ts";
import { type IActionsObject, PushProcessor } from "./pushprocessor.ts";
import { type AutoDiscoveryAction } from "./autodiscovery.ts";
import { TypedReEmitter } from "./ReEmitter.ts";
import { type Logger } from "./logger.ts";
import { SERVICE_TYPES } from "./service-types.ts";
import { type FileType, type HttpApiEvent, type HttpApiEventHandlerMap, type IHttpOpts, type IRequestOpts, MatrixError, MatrixHttpApi, type TokenRefreshFunction, type Upload, type UploadOpts, type UploadResponse } from "./http-api/index.ts";
import { User, UserEvent, type UserEventHandlerMap } from "./models/user.ts";
import { type IIdentityServerProvider } from "./@types/IIdentityServerProvider.ts";
import { type MatrixScheduler } from "./scheduler.ts";
import { type BeaconEvent, type BeaconEventHandlerMap } from "./models/beacon.ts";
import { type AuthDict } from "./interactive-auth.ts";
import { type IMinimalEvent, type IRoomEvent, type IStateEvent, type ReceivedToDeviceMessage } from "./sync-accumulator.ts";
import type { EventTimelineSet } from "./models/event-timeline-set.ts";
import { type Room, type RoomEvent, type RoomEventHandlerMap, type RoomNameState } from "./models/room.ts";
import { RoomMemberEvent, type RoomMemberEventHandlerMap } from "./models/room-member.ts";
import { type RoomStateEvent, type RoomStateEventHandlerMap } from "./models/room-state.ts";
import { type DelayedEventInfo, type IAddThreePidOnlyBody, type IBindThreePidBody, type ICreateRoomOpts, type IEventSearchOpts, type IGuestAccessOpts, type IJoinRoomOpts, type InviteOpts, type IPaginateOpts, type IPresenceOpts, type IRedactOpts, type IRelationsRequestOpts, type IRelationsResponse, type IRoomDirectoryOptions, type ISearchOpts, type ISendEventResponse, type IStatusResponse, type ITagsResponse, type KnockRoomOpts, type SendDelayedEventRequestOpts, type SendDelayedEventResponse, type UpdateDelayedEventAction } from "./@types/requests.ts";
import { type AccountDataEvents, EventType, RelationType, RoomType, type StateEvents, type TimelineEvents } from "./@types/event.ts";
import { type IdServerUnbindResult, type JoinRule, Preset, type Terms, type Visibility } from "./@types/partials.ts";
import { type EventMapper, type MapperOpts } from "./event-mapper.ts";
import { MSC3089TreeSpace } from "./models/MSC3089TreeSpace.ts";
import { type ISignatures } from "./@types/signed.ts";
import { type IStore } from "./store/index.ts";
import { type IEventWithRoomId, type ISearchRequestBody, type ISearchResponse, type ISearchResults, type IStateEventWithRoomId } from "./@types/search.ts";
import { type ISynapseAdminDeactivateResponse, type ISynapseAdminWhoisResponse } from "./@types/synapse.ts";
import { type IHierarchyRoom } from "./@types/spaces.ts";
import { type IPusher, type IPusherRequest, type IPushRule, type IPushRules, type PushRuleAction, PushRuleKind, type RuleId } from "./@types/PushRules.ts";
import { type IThreepid } from "./@types/threepids.ts";
import { type CryptoStore } from "./crypto/store/base.ts";
import { GroupCall, type GroupCallIntent, type GroupCallType, type IGroupCallDataChannelOptions } from "./webrtc/groupCall.ts";
import { MediaHandler } from "./webrtc/mediaHandler.ts";
import { type ILoginFlowsResponse, type IRefreshTokenResponse, type LoginRequest, type LoginResponse, type LoginTokenPostResponse, type SSOAction } from "./@types/auth.ts";
import { TypedEventEmitter } from "./models/typed-event-emitter.ts";
import { ReceiptType } from "./@types/read_receipts.ts";
import { type MSC3575SlidingSyncRequest, type MSC3575SlidingSyncResponse, type SlidingSync } from "./sliding-sync.ts";
import { SlidingSyncSdk } from "./sliding-sync-sdk.ts";
import { FeatureSupport, ThreadFilterType } from "./models/thread.ts";
import { type MBeaconInfoEventContent } from "./@types/beacon.ts";
import { NamespacedValue, UnstableValue } from "./NamespacedValue.ts";
import { type ToDeviceBatch, type ToDevicePayload } from "./models/ToDeviceMessage.ts";
import { IgnoredInvites } from "./models/invites-ignorer.ts";
import { type LocalNotificationSettings } from "./@types/local_notifications.ts";
import { Feature, ServerSupport } from "./feature.ts";
import { type CrossSigningKeyInfo, type CryptoApi, type CryptoCallbacks, CryptoEvent, type CryptoEventHandlerMap } from "./crypto-api/index.ts";
import { type SecretStorageKeyDescription, type ServerSideSecretStorage } from "./secret-storage.ts";
import { type RegisterRequest, type RegisterResponse } from "./@types/registration.ts";
import { MatrixRTCSessionManager } from "./matrixrtc/MatrixRTCSessionManager.ts";
import { type Membership } from "./@types/membership.ts";
import { type RoomMessageEventContent } from "./@types/events.ts";
import { type ImageInfo } from "./@types/media.ts";
import { type Capabilities } from "./serverCapabilities.ts";
import { type OidcClientConfig } from "./oidc/index.ts";
import { type EmptyObject } from "./@types/common.ts";
export type Store = IStore;
export type ResetTimelineCallback = (roomId: string) => boolean;
export declare const UNSTABLE_MSC3852_LAST_SEEN_UA: UnstableValue<"last_seen_user_agent", "org.matrix.msc3852.last_seen_user_agent">;
export interface IKeysUploadResponse {
    one_time_key_counts: {
        [algorithm: string]: number;
    };
}
export interface ICreateClientOpts {
    baseUrl: string;
    idBaseUrl?: string;
    /**
     * The data store used for sync data from the homeserver. If not specified,
     * this client will not store any HTTP responses. The `createClient` helper
     * will create a default store if needed.
     */
    store?: Store;
    /**
     * A store to be used for end-to-end crypto session data.
     * The `createClient` helper will create a default store if needed. Calls the factory supplied to
     * {@link setCryptoStoreFactory} if unspecified; or if no factory has been
     * specified, uses a default implementation (indexeddb in the browser,
     * in-memory otherwise).
     *
     * This is only used for the legacy crypto implementation,
     * but if you use the rust crypto implementation ({@link MatrixClient#initRustCrypto}) and the device
     * previously used legacy crypto (so must be migrated), then this must still be provided, so that the
     * data can be migrated from the legacy store.
     */
    cryptoStore?: CryptoStore;
    /**
     * The scheduler to use. If not
     * specified, this client will not retry requests on failure. This client
     * will supply its own processing function to
     * {@link MatrixScheduler#setProcessFunction}.
     */
    scheduler?: MatrixScheduler;
    /**
     * The function to invoke for HTTP requests.
     * Most supported environments have a global `fetch` registered to which this will fall back.
     */
    fetchFn?: typeof globalThis.fetch;
    userId?: string;
    /**
     * A unique identifier for this device; used for tracking things like crypto
     * keys and access tokens. If not specified, end-to-end encryption will be
     * disabled.
     */
    deviceId?: string;
    accessToken?: string;
    refreshToken?: string;
    /**
     * Function used to attempt refreshing access and refresh tokens
     * Called by http-api when a possibly expired token is encountered
     * and a refreshToken is found
     */
    tokenRefreshFunction?: TokenRefreshFunction;
    /**
     * Identity server provider to retrieve the user's access token when accessing
     * the identity server. See also https://github.com/vector-im/element-web/issues/10615
     * which seeks to replace the previous approach of manual access tokens params
     * with this callback throughout the SDK.
     */
    identityServer?: IIdentityServerProvider;
    /**
     * The default maximum amount of
     * time to wait before timing out HTTP requests. If not specified, there is no timeout.
     */
    localTimeoutMs?: number;
    /**
     * Set to false to send the access token to the server via a query parameter rather
     * than the Authorization HTTP header.
     *
     * Note that as of v1.11 of the Matrix spec, sending the access token via a query
     * is deprecated.
     *
     * Default true.
     */
    useAuthorizationHeader?: boolean;
    /**
     * Set to true to enable
     * improved timeline support, see {@link MatrixClient#getEventTimeline}.
     * It is disabled by default for compatibility with older clients - in particular to
     * maintain support for back-paginating the live timeline after a '/sync'
     * result with a gap.
     */
    timelineSupport?: boolean;
    /**
     * Extra query parameters to append
     * to all requests with this client. Useful for application services which require
     * `?user_id=`.
     */
    queryParams?: QueryDict;
    /**
     * Encryption key used for encrypting sensitive data (such as e2ee keys) in {@link ICreateClientOpts#cryptoStore}.
     *
     * This must be set to the same value every time the client is initialised for the same device.
     *
     * This is only used for the legacy crypto implementation,
     * but if you use the rust crypto implementation ({@link MatrixClient#initRustCrypto}) and the device
     * previously used legacy crypto (so must be migrated), then this must still be provided, so that the
     * data can be migrated from the legacy store.
     */
    pickleKey?: string;
    /**
     * Verification methods we should offer to the other side when performing an interactive verification.
     * If unset, we will offer all known methods. Currently these are: showing a QR code, scanning a QR code, and SAS
     * (aka "emojis").
     *
     * See {@link types.VerificationMethod} for a set of useful constants for this parameter.
     */
    verificationMethods?: Array<string>;
    /**
     * Whether relaying calls through a TURN server should be forced. Default false.
     */
    forceTURN?: boolean;
    /**
     * Up to this many ICE candidates will be gathered when an incoming call arrives.
     * Gathering does not send data to the caller, but will communicate with the configured TURN
     * server. Default 0.
     */
    iceCandidatePoolSize?: number;
    /**
     * True to advertise support for call transfers to other parties on Matrix calls. Default false.
     */
    supportsCallTransfer?: boolean;
    /**
     * Whether to allow a fallback ICE server should be used for negotiating a
     * WebRTC connection if the homeserver doesn't provide any servers. Defaults to false.
     */
    fallbackICEServerAllowed?: boolean;
    /**
     * If true, to-device signalling for group calls will be encrypted
     * with Olm. Default: true.
     */
    useE2eForGroupCall?: boolean;
    livekitServiceURL?: string;
    /**
     * Crypto callbacks provided by the application
     */
    cryptoCallbacks?: CryptoCallbacks;
    /**
     * Enable encrypted state events.
     */
    enableEncryptedStateEvents?: boolean;
    /**
     * Method to generate room names for empty rooms and rooms names based on membership.
     * Defaults to a built-in English handler with basic pluralisation.
     */
    roomNameGenerator?: (roomId: string, state: RoomNameState) => string | null;
    /**
     * If true, participant can join group call without video and audio this has to be allowed. By default, a local
     * media stream is needed to establish a group call.
     * Default: false.
     */
    isVoipWithNoMediaAllowed?: boolean;
    /**
     * Disable VoIP support (prevents fetching TURN servers, etc.)
     * Default: false (VoIP enabled)
     */
    disableVoip?: boolean;
    /**
     * If true, group calls will not establish media connectivity and only create the signaling events,
     * so that livekit media can be used in the application layer (js-sdk contains no livekit code).
     */
    useLivekitForGroupCalls?: boolean;
    /**
     * A logger to associate with this MatrixClient.
     * Defaults to the built-in global logger; see {@link DebugLogger} for an alternative.
     */
    logger?: Logger;
}
export interface IMatrixClientCreateOpts extends ICreateClientOpts {
    /**
     * Whether to allow sending messages to encrypted rooms when encryption
     * is not available internally within this SDK. This is useful if you are using an external
     * E2E proxy, for example. Defaults to false.
     */
    usingExternalCrypto?: boolean;
}
export declare enum PendingEventOrdering {
    Chronological = "chronological",
    Detached = "detached"
}
export interface IStartClientOpts {
    /**
     * The event `limit=` to apply to initial sync. Default: 8.
     */
    initialSyncLimit?: number;
    /**
     * True to put `archived=true</code> on the <code>/initialSync` request. Default: false.
     */
    includeArchivedRooms?: boolean;
    /**
     * True to do /profile requests on every invite event if the displayname/avatar_url is not known for this user ID. Default: false.
     */
    resolveInvitesToProfiles?: boolean;
    /**
     * Controls where pending messages appear in a room's timeline. If "<b>chronological</b>", messages will
     * appear in the timeline when the call to `sendEvent` was made. If "<b>detached</b>",
     * pending messages will appear in a separate list, accessible via {@link Room#getPendingEvents}.
     * Default: "chronological".
     */
    pendingEventOrdering?: PendingEventOrdering;
    /**
     * The number of milliseconds to wait on /sync. Default: 30000 (30 seconds).
     */
    pollTimeout?: number;
    /**
     * The filter to apply to /sync calls.
     */
    filter?: Filter;
    /**
     * True to perform syncing without automatically updating presence.
     */
    disablePresence?: boolean;
    /**
     * True to not load all membership events during initial sync but fetch them when needed by calling
     * `loadOutOfBandMembers` This will override the filter option at this moment.
     */
    lazyLoadMembers?: boolean;
    /**
     * The number of seconds between polls to /.well-known/matrix/client, undefined to disable.
     * This should be in the order of hours. Default: undefined.
     */
    clientWellKnownPollPeriod?: number;
    /**
     * Will organises events in threaded conversations when
     * a thread relation is encountered
     */
    threadSupport?: boolean;
    /**
     * @experimental
     */
    slidingSync?: SlidingSync;
}
export interface IStoredClientOpts extends IStartClientOpts {
}
export declare const GET_LOGIN_TOKEN_CAPABILITY: NamespacedValue<"m.get_login_token", "org.matrix.msc3882.get_login_token">;
export declare const UNSTABLE_MSC2666_SHARED_ROOMS = "uk.half-shot.msc2666";
export declare const UNSTABLE_MSC2666_MUTUAL_ROOMS = "uk.half-shot.msc2666.mutual_rooms";
export declare const UNSTABLE_MSC2666_QUERY_MUTUAL_ROOMS = "uk.half-shot.msc2666.query_mutual_rooms";
export declare const UNSTABLE_MSC4140_DELAYED_EVENTS = "org.matrix.msc4140";
export declare const UNSTABLE_MSC4354_STICKY_EVENTS = "org.matrix.msc4354";
export declare const UNSTABLE_MSC4133_EXTENDED_PROFILES = "uk.tcpip.msc4133";
export declare const STABLE_MSC4133_EXTENDED_PROFILES = "uk.tcpip.msc4133.stable";
declare enum CrossSigningKeyType {
    MasterKey = "master_key",
    SelfSigningKey = "self_signing_key",
    UserSigningKey = "user_signing_key"
}
export type CrossSigningKeys = Record<CrossSigningKeyType, CrossSigningKeyInfo>;
export type SendToDeviceContentMap = Map<string, Map<string, Record<string, any>>>;
export interface ISignedKey {
    keys: Record<string, string>;
    signatures: ISignatures;
    user_id: string;
    algorithms: string[];
    device_id: string;
}
export type KeySignatures = Record<string, Record<string, CrossSigningKeyInfo | ISignedKey>>;
export interface IUploadKeySignaturesResponse {
    failures: Record<string, Record<string, {
        errcode: string;
        error: string;
    }>>;
}
export interface IPreviewUrlResponse {
    [key: string]: undefined | string | number;
    "og:title": string;
    "og:type": string;
    "og:url": string;
    "og:image"?: string;
    "og:image:type"?: string;
    "og:image:height"?: number;
    "og:image:width"?: number;
    "og:description"?: string;
    "matrix:image:size"?: number;
}
export interface ITurnServerResponse {
    uris: string[];
    username: string;
    password: string;
    ttl: number;
}
export interface ITurnServer {
    urls: string[];
    username: string;
    credential: string;
}
export interface IServerVersions {
    versions: string[];
    unstable_features: Record<string, boolean>;
}
export interface IClientWellKnown {
    [key: string]: any;
    "m.homeserver"?: IWellKnownConfig;
    "m.identity_server"?: IWellKnownConfig;
}
export interface IWellKnownConfig<T = IClientWellKnown> {
    raw?: T;
    action?: AutoDiscoveryAction;
    reason?: string;
    error?: Error | string;
    base_url?: string | null;
    server_name?: string;
}
interface IMediaConfig {
    [key: string]: any;
    "m.upload.size"?: number;
}
interface ITagMetadata {
    [key: string]: any;
    order?: number;
}
interface IMessagesResponse {
    start?: string;
    end?: string;
    chunk: IRoomEvent[];
    state?: IStateEvent[];
}
export interface IRequestTokenResponse {
    sid: string;
    submit_url?: string;
}
export interface IRequestMsisdnTokenResponse extends IRequestTokenResponse {
    msisdn: string;
    success: boolean;
    intl_fmt: string;
}
export interface IUploadKeysRequest {
    "device_keys"?: Required<IDeviceKeys>;
    "one_time_keys"?: Record<string, IOneTimeKey>;
    "org.matrix.msc2732.fallback_keys"?: Record<string, IOneTimeKey>;
}
export interface IQueryKeysRequest {
    device_keys: {
        [userId: string]: string[];
    };
    timeout?: number;
    token?: string;
}
export interface IClaimKeysRequest {
    one_time_keys: {
        [userId: string]: {
            [deviceId: string]: string;
        };
    };
    timeout?: number;
}
export interface IOpenIDToken {
    access_token: string;
    token_type: "Bearer" | string;
    matrix_server_name: string;
    expires_in: number;
}
interface IRoomInitialSyncResponse {
    room_id: string;
    membership: Membership;
    messages?: {
        start?: string;
        end?: string;
        chunk: IEventWithRoomId[];
    };
    state?: IStateEventWithRoomId[];
    visibility: Visibility;
    account_data?: IMinimalEvent[];
    presence: Partial<IEvent>;
}
interface IJoinedRoomsResponse {
    joined_rooms: string[];
}
interface IJoinedMembersResponse {
    joined: {
        [userId: string]: {
            display_name: string;
            avatar_url: string;
        };
    };
}
export type IRegisterRequestParams = RegisterRequest;
export interface IPublicRoomsChunkRoom {
    room_id: string;
    name?: string;
    avatar_url?: string;
    topic?: string;
    canonical_alias?: string;
    aliases?: string[];
    world_readable: boolean;
    guest_can_join: boolean;
    num_joined_members: number;
    room_type?: RoomType | string;
    join_rule?: JoinRule.Knock | JoinRule.Public;
}
interface IPublicRoomsResponse {
    chunk: IPublicRoomsChunkRoom[];
    next_batch?: string;
    prev_batch?: string;
    total_room_count_estimate?: number;
}
interface IUserDirectoryResponse {
    results: {
        user_id: string;
        display_name?: string;
        avatar_url?: string;
    }[];
    limited: boolean;
}
export interface IMyDevice {
    "device_id": string;
    "display_name"?: string;
    "last_seen_ip"?: string;
    "last_seen_ts"?: number;
    "last_seen_user_agent"?: string;
    "org.matrix.msc3852.last_seen_user_agent"?: string;
}
export interface Keys {
    keys: {
        [keyId: string]: string;
    };
    usage: string[];
    user_id: string;
}
export interface SigningKeys extends Keys {
    signatures: ISignatures;
}
export interface DeviceKeys {
    [deviceId: string]: IDeviceKeys & {
        unsigned?: {
            device_display_name: string;
        };
    };
}
export interface IDownloadKeyResult {
    failures: {
        [serverName: string]: object;
    };
    device_keys: {
        [userId: string]: DeviceKeys;
    };
    master_keys?: {
        [userId: string]: Keys;
    };
    self_signing_keys?: {
        [userId: string]: SigningKeys;
    };
    user_signing_keys?: {
        [userId: string]: SigningKeys;
    };
}
export interface IClaimOTKsResult {
    failures: {
        [serverName: string]: object;
    };
    one_time_keys: {
        [userId: string]: {
            [deviceId: string]: {
                [keyId: string]: {
                    key: string;
                    signatures: ISignatures;
                };
            };
        };
    };
}
export interface IFieldType {
    regexp: string;
    placeholder: string;
}
export interface IInstance {
    desc: string;
    icon?: string;
    fields: object;
    network_id: string;
    instance_id: string;
}
export interface IProtocol {
    user_fields: string[];
    location_fields: string[];
    icon: string;
    field_types: Record<string, IFieldType>;
    instances: IInstance[];
}
interface IThirdPartyLocation {
    alias: string;
    protocol: string;
    fields: object;
}
interface IThirdPartyUser {
    userid: string;
    protocol: string;
    fields: object;
}
/**
 * The summary of a room as defined by an initial version of MSC3266 and implemented in Synapse
 * Proposed at https://github.com/matrix-org/matrix-doc/pull/3266
 */
export interface RoomSummary extends Omit<IPublicRoomsChunkRoom, "canonical_alias" | "aliases"> {
    /**
     * The current membership of this user in the room.
     * Usually "leave" if the room is fetched over federation.
     */
    "membership"?: Membership;
    /**
     * Version of the room.
     */
    "im.nheko.summary.room_version"?: string;
    /**
     * The encryption algorithm used for this room, if the room is encrypted.
     */
    "im.nheko.summary.encryption"?: string;
}
interface IRoomHierarchy {
    rooms: IHierarchyRoom[];
    next_batch?: string;
}
export interface TimestampToEventResponse {
    event_id: string;
    origin_server_ts: number;
}
interface IWhoamiResponse {
    user_id: string;
    device_id?: string;
    is_guest?: boolean;
}
export declare enum ClientEvent {
    /**
     * Fires whenever the SDK's syncing state is updated. The state can be one of:
     * <ul>
     *
     * <li>PREPARED: The client has synced with the server at least once and is
     * ready for methods to be called on it. This will be immediately followed by
     * a state of SYNCING. <i>This is the equivalent of "syncComplete" in the
     * previous API.</i></li>
     *
     * <li>CATCHUP: The client has detected the connection to the server might be
     * available again and will now try to do a sync again. As this sync might take
     * a long time (depending how long ago was last synced, and general server
     * performance) the client is put in this mode so the UI can reflect trying
     * to catch up with the server after losing connection.</li>
     *
     * <li>SYNCING : The client is currently polling for new events from the server.
     * This will be called <i>after</i> processing latest events from a sync.</li>
     *
     * <li>ERROR : The client has had a problem syncing with the server. If this is
     * called <i>before</i> PREPARED then there was a problem performing the initial
     * sync. If this is called <i>after</i> PREPARED then there was a problem polling
     * the server for updates. This may be called multiple times even if the state is
     * already ERROR. <i>This is the equivalent of "syncError" in the previous
     * API.</i></li>
     *
     * <li>RECONNECTING: The sync connection has dropped, but not (yet) in a way that
     * should be considered erroneous.
     * </li>
     *
     * <li>STOPPED: The client has stopped syncing with server due to stopClient
     * being called.
     * </li>
     * </ul>
     * State transition diagram:
     * ```
     *                                          +---->STOPPED
     *                                          |
     *              +----->PREPARED -------> SYNCING <--+
     *              |                        ^  |  ^    |
     *              |      CATCHUP ----------+  |  |    |
     *              |        ^                  V  |    |
     *   null ------+        |  +------- RECONNECTING   |
     *              |        V  V                       |
     *              +------->ERROR ---------------------+
     *
     * NB: 'null' will never be emitted by this event.
     *
     * ```
     * Transitions:
     * <ul>
     *
     * <li>`null -> PREPARED` : Occurs when the initial sync is completed
     * first time. This involves setting up filters and obtaining push rules.
     *
     * <li>`null -> ERROR` : Occurs when the initial sync failed first time.
     *
     * <li>`ERROR -> PREPARED` : Occurs when the initial sync succeeds
     * after previously failing.
     *
     * <li>`PREPARED -> SYNCING` : Occurs immediately after transitioning
     * to PREPARED. Starts listening for live updates rather than catching up.
     *
     * <li>`SYNCING -> RECONNECTING` : Occurs when the live update fails.
     *
     * <li>`RECONNECTING -> RECONNECTING` : Can occur if the update calls
     * continue to fail, but the keepalive calls (to /versions) succeed.
     *
     * <li>`RECONNECTING -> ERROR` : Occurs when the keepalive call also fails
     *
     * <li>`ERROR -> SYNCING` : Occurs when the client has performed a
     * live update after having previously failed.
     *
     * <li>`ERROR -> ERROR` : Occurs when the client has failed to keepalive
     * for a second time or more.</li>
     *
     * <li>`SYNCING -> SYNCING` : Occurs when the client has performed a live
     * update. This is called <i>after</i> processing.</li>
     *
     * <li>`* -> STOPPED` : Occurs once the client has stopped syncing or
     * trying to sync after stopClient has been called.</li>
     * </ul>
     *
     * The payloads consits of the following 3 parameters:
     *
     * - state - An enum representing the syncing state. One of "PREPARED",
     * "SYNCING", "ERROR", "STOPPED".
     *
     * - prevState - An enum representing the previous syncing state.
     * One of "PREPARED", "SYNCING", "ERROR", "STOPPED" <b>or null</b>.
     *
     * - data - Data about this transition.
     *
     * @example
     * ```
     * matrixClient.on("sync", function(state, prevState, data) {
     *   switch (state) {
     *     case "ERROR":
     *       // update UI to say "Connection Lost"
     *       break;
     *     case "SYNCING":
     *       // update UI to remove any "Connection Lost" message
     *       break;
     *     case "PREPARED":
     *       // the client instance is ready to be queried.
     *       var rooms = matrixClient.getRooms();
     *       break;
     *   }
     * });
     * ```
     */
    Sync = "sync",
    /**
     * Fires whenever the SDK receives a new event.
     * <p>
     * This is only fired for live events received via /sync - it is not fired for
     * events received over context, search, or pagination APIs.
     *
     * The payload is the matrix event which caused this event to fire.
     * @example
     * ```
     * matrixClient.on("event", function(event){
     *   var sender = event.getSender();
     * });
     * ```
     */
    Event = "event",
    /** @deprecated Use {@link ReceivedToDeviceMessage}.
     * Fires whenever the SDK receives a new to-device event.
     * The payload is the matrix event ({@link MatrixEvent}) which caused this event to fire.
     * @example
     * ```
     * matrixClient.on("toDeviceEvent", function(event){
     *   var sender = event.getSender();
     * });
     * ```
     */
    ToDeviceEvent = "toDeviceEvent",
    /**
     * Fires whenever the SDK receives a new (potentially decrypted) to-device message.
     * The payload is the to-device message and the encryption info for that message ({@link ReceivedToDeviceMessage}).
     * @example
     * ```
     * matrixClient.on("receivedToDeviceMessage", function(payload){
     *   const { message, encryptionInfo } = payload;
     *   var claimed_sender = encryptionInfo ? encryptionInfo.sender : message.sender;
     *   var isVerified = encryptionInfo ? encryptionInfo.verified : false;
     *   var type = message.type;
     * });
     */
    ReceivedToDeviceMessage = "receivedToDeviceMessage",
    /**
     * Fires whenever new user-scoped account_data is added.
     * The payload is a pair of event ({@link MatrixEvent}) describing the account_data just added, and the previous event, if known:
     *  - event: The event describing the account_data just added
     *  - oldEvent: The previous account data, if known.
     * @example
     * ```
     * matrixClient.on("accountData", function(event, oldEvent){
     *   myAccountData[event.type] = event.content;
     * });
     * ```
     */
    AccountData = "accountData",
    /**
     * Fires whenever a new Room is added. This will fire when you are invited to a
     * room, as well as when you join a room. <strong>This event is experimental and
     * may change.</strong>
     *
     * The payload is the newly created room, fully populated.
     * @example
     * ```
     * matrixClient.on("Room", function(room){
     *   var roomId = room.roomId;
     * });
     * ```
     */
    Room = "Room",
    /**
     * Fires whenever a Room is removed. This will fire when you forget a room.
     * <strong>This event is experimental and may change.</strong>
     * The payload is the roomId of the deleted room.
     * @example
     * ```
     * matrixClient.on("deleteRoom", function(roomId){
     *   // update UI from getRooms()
     * });
     * ```
     */
    DeleteRoom = "deleteRoom",
    SyncUnexpectedError = "sync.unexpectedError",
    /**
     * Fires when the client .well-known info is fetched.
     * The payload is the JSON object (see {@link IClientWellKnown}) returned by the server
     */
    ClientWellKnown = "WellKnown.client",
    ReceivedVoipEvent = "received_voip_event",
    TurnServers = "turnServers",
    TurnServersError = "turnServers.error"
}
type RoomEvents = RoomEvent.Name | RoomEvent.Redaction | RoomEvent.RedactionCancelled | RoomEvent.Receipt | RoomEvent.Tags | RoomEvent.LocalEchoUpdated | RoomEvent.HistoryImportedWithinTimeline | RoomEvent.AccountData | RoomEvent.MyMembership | RoomEvent.Timeline | RoomEvent.TimelineReset;
type RoomStateEvents = RoomStateEvent.Events | RoomStateEvent.Members | RoomStateEvent.NewMember | RoomStateEvent.Update | RoomStateEvent.Marker;
type CryptoEvents = (typeof CryptoEvent)[keyof typeof CryptoEvent];
type MatrixEventEvents = MatrixEventEvent.Decrypted | MatrixEventEvent.Replaced | MatrixEventEvent.VisibilityChange;
type RoomMemberEvents = RoomMemberEvent.Name | RoomMemberEvent.Typing | RoomMemberEvent.PowerLevel | RoomMemberEvent.Membership;
type UserEvents = UserEvent.AvatarUrl | UserEvent.DisplayName | UserEvent.Presence | UserEvent.CurrentlyActive | UserEvent.LastPresenceTs;
export type EmittedEvents = ClientEvent | RoomEvents | RoomStateEvents | CryptoEvents | MatrixEventEvents | RoomMemberEvents | UserEvents | CallEvent | CallEventHandlerEvent.Incoming | GroupCallEventHandlerEvent.Incoming | GroupCallEventHandlerEvent.Outgoing | GroupCallEventHandlerEvent.Ended | GroupCallEventHandlerEvent.Participants | HttpApiEvent.SessionLoggedOut | HttpApiEvent.NoConsent | BeaconEvent;
export type ClientEventHandlerMap = {
    [ClientEvent.Sync]: (state: SyncState, prevState: SyncState | null, data?: ISyncStateData) => void;
    [ClientEvent.Event]: (event: MatrixEvent) => void;
    [ClientEvent.ToDeviceEvent]: (event: MatrixEvent) => void;
    [ClientEvent.ReceivedToDeviceMessage]: (payload: ReceivedToDeviceMessage) => void;
    [ClientEvent.AccountData]: (event: MatrixEvent, lastEvent?: MatrixEvent) => void;
    [ClientEvent.Room]: (room: Room) => void;
    [ClientEvent.DeleteRoom]: (roomId: string) => void;
    [ClientEvent.SyncUnexpectedError]: (error: Error) => void;
    [ClientEvent.ClientWellKnown]: (data: IClientWellKnown) => void;
    [ClientEvent.ReceivedVoipEvent]: (event: MatrixEvent) => void;
    [ClientEvent.TurnServers]: (servers: ITurnServer[]) => void;
    [ClientEvent.TurnServersError]: (error: Error, fatal: boolean) => void;
} & RoomEventHandlerMap & RoomStateEventHandlerMap & CryptoEventHandlerMap & MatrixEventHandlerMap & RoomMemberEventHandlerMap & UserEventHandlerMap & CallEventHandlerEventHandlerMap & GroupCallEventHandlerEventHandlerMap & CallEventHandlerMap & HttpApiEventHandlerMap & BeaconEventHandlerMap;
/**
 * Represents a Matrix Client. Only directly construct this if you want to use
 * custom modules. Normally, {@link createClient} should be used
 * as it specifies 'sensible' defaults for these modules.
 */
export declare class MatrixClient extends TypedEventEmitter<EmittedEvents, ClientEventHandlerMap> {
    static readonly RESTORE_BACKUP_ERROR_BAD_KEY = "RESTORE_BACKUP_ERROR_BAD_KEY";
    private readonly logger;
    reEmitter: TypedReEmitter<EmittedEvents, ClientEventHandlerMap>;
    olmVersion: [number, number, number] | null;
    usingExternalCrypto: boolean;
    private _store;
    deviceId: string | null;
    credentials: {
        userId: string | null;
    };
    /**
     * Encryption key used for encrypting sensitive data (such as e2ee keys) in storage.
     *
     * As supplied in the constructor via {@link IMatrixClientCreateOpts#pickleKey}.
     * Used for migration from the legacy crypto to the rust crypto
     */
    private readonly legacyPickleKey?;
    scheduler?: MatrixScheduler;
    clientRunning: boolean;
    timelineSupport: boolean;
    urlPreviewCache: {
        [key: string]: Promise<IPreviewUrlResponse>;
    };
    identityServer?: IIdentityServerProvider;
    http: MatrixHttpApi<IHttpOpts & {
        onlyData: true;
    }>;
    private cryptoBackend?;
    private readonly enableEncryptedStateEvents;
    cryptoCallbacks: CryptoCallbacks;
    callEventHandler?: CallEventHandler;
    groupCallEventHandler?: GroupCallEventHandler;
    supportsCallTransfer: boolean;
    forceTURN: boolean;
    iceCandidatePoolSize: number;
    idBaseUrl?: string;
    baseUrl: string;
    readonly isVoipWithNoMediaAllowed: boolean;
    disableVoip: boolean;
    useLivekitForGroupCalls: boolean;
    protected canSupportVoip: boolean;
    protected peekSync: SyncApi | null;
    protected isGuestAccount: boolean;
    protected ongoingScrollbacks: {
        [roomId: string]: {
            promise?: Promise<Room>;
            errorTs?: number;
        };
    };
    protected notifTimelineSet: EventTimelineSet | null;
    /**
     * Legacy crypto store used for migration from the legacy crypto to the rust crypto
     * @private
     */
    private readonly legacyCryptoStore?;
    protected verificationMethods?: string[];
    protected fallbackICEServerAllowed: boolean;
    protected syncApi?: SlidingSyncSdk | SyncApi;
    roomNameGenerator?: ICreateClientOpts["roomNameGenerator"];
    pushRules?: IPushRules;
    protected syncLeftRoomsPromise?: Promise<Room[]>;
    protected syncedLeftRooms: boolean;
    protected clientOpts?: IStoredClientOpts;
    protected clientWellKnownIntervalID?: ReturnType<typeof setInterval>;
    protected canResetTimelineCallback?: ResetTimelineCallback;
    canSupport: Map<Feature, ServerSupport>;
    readonly pushProcessor: PushProcessor;
    protected serverVersionsPromise?: Promise<IServerVersions>;
    protected clientWellKnown?: IClientWellKnown;
    protected clientWellKnownPromise?: Promise<IClientWellKnown>;
    protected turnServers: ITurnServer[];
    protected turnServersExpiry: number;
    protected checkTurnServersIntervalID?: ReturnType<typeof setInterval>;
    protected txnCtr: number;
    protected mediaHandler: MediaHandler;
    protected sessionId: string;
    /** IDs of events which are currently being encrypted.
     *
     * This is part of the cancellation mechanism: if the event is no longer listed here when encryption completes,
     * that tells us that it has been cancelled, and we should not send it.
     */
    private eventsBeingEncrypted;
    private useE2eForGroupCall;
    private toDeviceMessageQueue;
    livekitServiceURL?: string;
    private _secretStorage;
    readonly ignoredInvites: IgnoredInvites;
    readonly matrixRTC: MatrixRTCSessionManager;
    private serverCapabilitiesService;
    constructor(opts: IMatrixClientCreateOpts);
    set store(newStore: Store);
    get store(): Store;
    /**
     * High level helper method to begin syncing and poll for new events. To listen for these
     * events, add a listener for {@link ClientEvent.Event}
     * via {@link MatrixClient#on}. Alternatively, listen for specific
     * state change events.
     * @param opts - Options to apply when syncing.
     */
    startClient(opts?: IStartClientOpts): Promise<void>;
    /**
     * Construct a SyncApiOptions for this client, suitable for passing into the SyncApi constructor
     */
    protected buildSyncApiOptions(): SyncApiOptions;
    /**
     * High level helper method to stop the client from polling and allow a
     * clean shutdown.
     */
    stopClient(): void;
    /**
     * Clear any data out of the persistent stores used by the client.
     *
     * @param args.cryptoDatabasePrefix - The database name to use for indexeddb, defaults to 'matrix-js-sdk'.
     * @returns Promise which resolves when the stores have been cleared.
     */
    clearStores(args?: {
        cryptoDatabasePrefix?: string;
    }): Promise<void>;
    /**
     * Get the user-id of the logged-in user
     *
     * @returns MXID for the logged-in user, or null if not logged in
     */
    getUserId(): string | null;
    /**
     * Get the user-id of the logged-in user
     *
     * @returns MXID for the logged-in user
     * @throws Error if not logged in
     */
    getSafeUserId(): string;
    /**
     * Get the domain for this client's MXID
     * @returns Domain of this MXID
     */
    getDomain(): string | null;
    /**
     * Get the local part of the current user ID e.g. "foo" in "\@foo:bar".
     * @returns The user ID localpart or null.
     */
    getUserIdLocalpart(): string | null;
    /**
     * Get the device ID of this client
     * @returns device ID
     */
    getDeviceId(): string | null;
    /**
     * Get the session ID of this client
     * @returns session ID
     */
    getSessionId(): string;
    /**
     * Check if the runtime environment supports VoIP calling.
     * @returns True if VoIP is supported.
     */
    supportsVoip(): boolean;
    /**
     * @returns
     */
    getMediaHandler(): MediaHandler;
    /**
     * Set whether VoIP calls are forced to use only TURN
     * candidates. This is the same as the forceTURN option
     * when creating the client.
     * @param force - True to force use of TURN servers
     */
    setForceTURN(force: boolean): void;
    /**
     * Set whether to advertise transfer support to other parties on Matrix calls.
     * @param support - True to advertise the 'm.call.transferee' capability
     */
    setSupportsCallTransfer(support: boolean): void;
    /**
     * Returns true if to-device signalling for group calls will be encrypted with Olm.
     * If false, it will be sent unencrypted.
     * @returns boolean Whether group call signalling will be encrypted
     */
    getUseE2eForGroupCall(): boolean;
    /**
     * Creates a new call.
     * The place*Call methods on the returned call can be used to actually place a call
     *
     * @param roomId - The room the call is to be placed in.
     * @returns the call or null if the browser doesn't support calling.
     */
    createCall(roomId: string): MatrixCall | null;
    /**
     * Creates a new group call and sends the associated state event
     * to alert other members that the room now has a group call.
     *
     * @param roomId - The room the call is to be placed in.
     */
    createGroupCall(roomId: string, type: GroupCallType, isPtt: boolean, intent: GroupCallIntent, dataChannelsEnabled?: boolean, dataChannelOptions?: IGroupCallDataChannelOptions): Promise<GroupCall>;
    getLivekitServiceURL(): string | undefined;
    setLivekitServiceURL(newURL: string): void;
    /**
     * Wait until an initial state for the given room has been processed by the
     * client and the client is aware of any ongoing group calls. Awaiting on
     * the promise returned by this method before calling getGroupCallForRoom()
     * avoids races where getGroupCallForRoom is called before the state for that
     * room has been processed. It does not, however, fix other races, eg. two
     * clients both creating a group call at the same time.
     * @param roomId - The room ID to wait for
     * @returns A promise that resolves once existing group calls in the room
     *          have been processed.
     */
    waitUntilRoomReadyForGroupCalls(roomId: string): Promise<void>;
    /**
     * Get an existing group call for the provided room.
     * @returns The group call or null if it doesn't already exist.
     */
    getGroupCallForRoom(roomId: string): GroupCall | null;
    /**
     * Get the current sync state.
     * @returns the sync state, which may be null.
     * @see MatrixClient#event:"sync"
     */
    getSyncState(): SyncState | null;
    /**
     * Returns the additional data object associated with
     * the current sync state, or null if there is no
     * such data.
     * Sync errors, if available, are put in the 'error' key of
     * this object.
     */
    getSyncStateData(): ISyncStateData | null;
    /**
     * Whether the initial sync has completed.
     * @returns True if at least one sync has happened.
     */
    isInitialSyncComplete(): boolean;
    /**
     * Return whether the client is configured for a guest account.
     * @returns True if this is a guest access_token (or no token is supplied).
     */
    isGuest(): boolean;
    /**
     * Set whether this client is a guest account. <b>This method is experimental
     * and may change without warning.</b>
     * @param guest - True if this is a guest account.
     * @experimental if the token is a macaroon, it should be encoded in it that it is a 'guest'
     * access token, which means that the SDK can determine this entirely without
     * the dev manually flipping this flag.
     */
    setGuest(guest: boolean): void;
    /**
     * Return the provided scheduler, if any.
     * @returns The scheduler or undefined
     */
    getScheduler(): MatrixScheduler | undefined;
    /**
     * Retry a backed off syncing request immediately. This should only be used when
     * the user <b>explicitly</b> attempts to retry their lost connection.
     * Will also retry any outbound to-device messages currently in the queue to be sent
     * (retries of regular outgoing events are handled separately, per-event).
     * @returns True if this resulted in a request being retried.
     */
    retryImmediately(): boolean;
    /**
     * Return the global notification EventTimelineSet, if any
     *
     * @returns the globl notification EventTimelineSet
     */
    getNotifTimelineSet(): EventTimelineSet | null;
    /**
     * Set the global notification EventTimelineSet
     *
     */
    setNotifTimelineSet(set: EventTimelineSet): void;
    /**
     * Gets the cached capabilities of the homeserver, returning cached ones if available.
     * If there are no cached capabilities and none can be fetched, throw an exception.
     *
     * @returns Promise resolving with The capabilities of the homeserver
     */
    getCapabilities(): Promise<Capabilities>;
    /**
     * Gets the cached capabilities of the homeserver. If none have been fetched yet,
     * return undefined.
     *
     * @returns The capabilities of the homeserver
     */
    getCachedCapabilities(): Capabilities | undefined;
    /**
     * Fetches the latest capabilities from the homeserver, ignoring any cached
     * versions. The newly returned version is cached.
     *
     * @returns A promise which resolves to the capabilities of the homeserver
     */
    fetchCapabilities(): Promise<Capabilities>;
    /**
     * Initialise support for end-to-end encryption in this client, using the rust matrix-sdk-crypto.
     *
     * **WARNING**: the cryptography stack is not thread-safe. Having multiple `MatrixClient` instances connected to
     * the same Indexed DB will cause data corruption and decryption failures. The application layer is responsible for
     * ensuring that only one `MatrixClient` issue is instantiated at a time.
     *
     * @param args.useIndexedDB - True to use an indexeddb store, false to use an in-memory store. Defaults to 'true'.
     * @param args.cryptoDatabasePrefix - The database name to use for indexeddb, defaults to 'matrix-js-sdk'.
     *    Unused if useIndexedDB is 'false'.
     * @param args.storageKey - A key with which to encrypt the indexeddb store. If provided, it must be exactly
     *    32 bytes of data, and must be the same each time the client is initialised for a given device.
     *    If both this and `storagePassword` are unspecified, the store will be unencrypted.
     * @param args.storagePassword - An alternative to `storageKey`. A password which will be used to derive a key to
     *    encrypt the store with. Deriving a key from a password is (deliberately) a slow operation, so prefer
     *    to pass a `storageKey` directly where possible.
     *
     * @returns a Promise which will resolve when the crypto layer has been
     *    successfully initialised.
     */
    initRustCrypto(args?: {
        useIndexedDB?: boolean;
        cryptoDatabasePrefix?: string;
        storageKey?: Uint8Array;
        storagePassword?: string;
    }): Promise<void>;
    /**
     * Access the server-side secret storage API for this client.
     */
    get secretStorage(): ServerSideSecretStorage;
    /**
     * Access the crypto API for this client.
     *
     * If end-to-end encryption has been enabled for this client (via {@link initRustCrypto}),
     * returns an object giving access to the crypto API. Otherwise, returns `undefined`.
     */
    getCrypto(): CryptoApi | undefined;
    /**
     * Whether encryption is enabled for a room.
     * @param roomId - the room id to query.
     * @returns whether encryption is enabled.
     *
     * @deprecated Not correctly supported for Rust Cryptography. Use {@link CryptoApi.isEncryptionEnabledInRoom} and/or
     *    {@link Room.hasEncryptionStateEvent}.
     */
    isRoomEncrypted(roomId: string): boolean;
    /**
     * Check whether the key backup private key is stored in secret storage.
     * @returns map of key name to key info the secret is
     *     encrypted with, or null if it is not present or not encrypted with a
     *     trusted key
     */
    isKeyBackupKeyStored(): Promise<Record<string, SecretStorageKeyDescription> | null>;
    private makeKeyBackupPath;
    deleteKeysFromBackup(roomId: undefined, sessionId: undefined, version?: string): Promise<void>;
    deleteKeysFromBackup(roomId: string, sessionId: undefined, version?: string): Promise<void>;
    deleteKeysFromBackup(roomId: string, sessionId: string, version?: string): Promise<void>;
    /**
     * Get the config for the media repository.
     *
     * @param useAuthenticatedMedia - If true, the caller supports authenticated
     * media and wants an authentication-required URL. Note that server support
     * for authenticated media will *not* be checked - it is the caller's responsibility
     * to do so before calling this function.
     *
     * @returns Promise which resolves with an object containing the config.
     */
    getMediaConfig(useAuthenticatedMedia?: boolean): Promise<IMediaConfig>;
    /**
     * Get the room for the given room ID.
     * This function will return a valid room for any room for which a Room event
     * has been emitted. Note in particular that other events, eg. RoomState.members
     * will be emitted for a room before this function will return the given room.
     * @param roomId - The room ID
     * @returns The Room or null if it doesn't exist or there is no data store.
     */
    getRoom(roomId: string | undefined): Room | null;
    /**
     * Retrieve all known rooms.
     * @returns A list of rooms, or an empty list if there is no data store.
     */
    getRooms(): Room[];
    /**
     * Retrieve all rooms that should be displayed to the user
     * This is essentially getRooms() with some rooms filtered out, eg. old versions
     * of rooms that have been replaced or (in future) other rooms that have been
     * marked at the protocol level as not to be displayed to the user.
     *
     * @param msc3946ProcessDynamicPredecessor - if true, look for an
     *                                           m.room.predecessor state event and
     *                                           use it if found (MSC3946).
     * @returns A list of rooms, or an empty list if there is no data store.
     */
    getVisibleRooms(msc3946ProcessDynamicPredecessor?: boolean): Room[];
    /**
     * Retrieve a user.
     * @param userId - The user ID to retrieve.
     * @returns A user or null if there is no data store or the user does
     * not exist.
     */
    getUser(userId: string): User | null;
    /**
     * Retrieve all known users.
     * @returns A list of users, or an empty list if there is no data store.
     */
    getUsers(): User[];
    /**
     * Set account data event for the current user, and wait for the result to be echoed over `/sync`.
     *
     * Waiting for the remote echo ensures that a subsequent call to {@link getAccountData} will return the updated
     * value.
     *
     * If called before the client is started with {@link startClient}, logs a warning and falls back to
     * {@link setAccountDataRaw}.
     *
     * Retries the request up to 5 times in the case of an {@link ConnectionError}.
     *
     * @param eventType - The event type
     * @param content - the contents object for the event
     */
    setAccountData<K extends keyof AccountDataEvents>(eventType: K, content: AccountDataEvents[K] | Record<string, never>): Promise<EmptyObject>;
    /**
     * Set account data event for the current user, without waiting for the remote echo.
     *
     * @param eventType - The event type
     * @param content - the contents object for the event
     */
    setAccountDataRaw<K extends keyof AccountDataEvents>(eventType: K, content: AccountDataEvents[K] | Record<string, never>): Promise<EmptyObject>;
    /**
     * Get account data event of given type for the current user.
     * @param eventType - The event type
     * @returns The contents of the given account data event
     */
    getAccountData<K extends keyof AccountDataEvents>(eventType: K): MatrixEvent | undefined;
    /**
     * Get account data event of given type for the current user. This variant
     * gets account data directly from the homeserver if the local store is not
     * ready, which can be useful very early in startup before the initial sync.
     * @param eventType - The event type
     * @returns Promise which resolves: The contents of the given account data event.
     * @returns Rejects: with an error response.
     */
    getAccountDataFromServer<K extends keyof AccountDataEvents>(eventType: K): Promise<AccountDataEvents[K] | null>;
    deleteAccountData(eventType: keyof AccountDataEvents): Promise<void>;
    /**
     * Gets the users that are ignored by this client
     * @returns The array of users that are ignored (empty if none)
     */
    getIgnoredUsers(): string[];
    /**
     * Sets the users that the current user should ignore.
     * @param userIds - the user IDs to ignore
     * @returns Promise which resolves: an empty object
     * @returns Rejects: with an error response.
     */
    setIgnoredUsers(userIds: string[]): Promise<EmptyObject>;
    /**
     * Gets whether or not a specific user is being ignored by this client.
     * @param userId - the user ID to check
     * @returns true if the user is ignored, false otherwise
     */
    isUserIgnored(userId: string): boolean;
    /**
     * Join a room. If you have already joined the room, this will no-op.
     * @param roomIdOrAlias - The room ID or room alias to join.
     * @param opts - Options when joining the room.
     * @returns Promise which resolves: Room object.
     * @returns Rejects: with an error response.
     */
    joinRoom(roomIdOrAlias: string, opts?: IJoinRoomOpts): Promise<Room>;
    /**
     * Knock a room. If you have already knocked the room, this will no-op.
     * @param roomIdOrAlias - The room ID or room alias to knock.
     * @param opts - Options when knocking the room.
     * @returns Promise which resolves: `{room_id: {string}}`
     * @returns Rejects: with an error response.
     */
    knockRoom(roomIdOrAlias: string, opts?: KnockRoomOpts): Promise<{
        room_id: string;
    }>;
    /**
     * Resend an event. Will also retry any to-device messages waiting to be sent.
     * @param event - The event to resend.
     * @param room - Optional. The room the event is in. Will update the
     * timeline entry if provided.
     * @returns Promise which resolves: to an ISendEventResponse object
     * @returns Rejects: with an error response.
     */
    resendEvent(event: MatrixEvent, room: Room): Promise<ISendEventResponse>;
    /**
     * Cancel a queued or unsent event.
     *
     * @param event -   Event to cancel
     * @throws Error if the event is not in QUEUED, NOT_SENT or ENCRYPTING state
     */
    cancelPendingEvent(event: MatrixEvent): void;
    /**
     * @returns Promise which resolves: TODO
     * @returns Rejects: with an error response.
     */
    setRoomName(roomId: string, name: string): Promise<ISendEventResponse>;
    /**
     * @param roomId - The room to update the topic in.
     * @param topic - The plaintext topic. May be empty to remove the topic.
     * @param htmlTopic - Optional.
     * @returns Promise which resolves: TODO
     * @returns Rejects: with an error response.
     */
    setRoomTopic(roomId: string, topic?: string, htmlTopic?: string): Promise<ISendEventResponse>;
    /**
     * @returns Promise which resolves: to an object keyed by tagId with objects containing a numeric order field.
     * @returns Rejects: with an error response.
     */
    getRoomTags(roomId: string): Promise<ITagsResponse>;
    /**
     * @param tagName - name of room tag to be set
     * @param metadata - associated with that tag to be stored
     * @returns Promise which resolves: to an empty object
     * @returns Rejects: with an error response.
     */
    setRoomTag(roomId: string, tagName: string, metadata?: ITagMetadata): Promise<EmptyObject>;
    /**
     * @param tagName - name of room tag to be removed
     * @returns Promise which resolves: to an empty object
     * @returns Rejects: with an error response.
     */
    deleteRoomTag(roomId: string, tagName: string): Promise<EmptyObject>;
    /**
     * @param eventType - event type to be set
     * @param content - event content
     * @returns Promise which resolves: to an empty object `{}`
     * @returns Rejects: with an error response.
     */
    setRoomAccountData(roomId: string, eventType: string, content: Record<string, any>): Promise<EmptyObject>;
    /**
     * Set a power level to one or multiple users.
     * Will apply changes atop of current power level event from local state if running & synced, falling back
     * to fetching latest from the `/state/` API.
     * @param roomId - the room to update power levels in
     * @param userId - the ID of the user or users to update power levels of
     * @param powerLevel - the numeric power level to update given users to
     * @returns Promise which resolves: to an ISendEventResponse object
     * @returns Rejects: with an error response.
     */
    setPowerLevel(roomId: string, userId: string | string[], powerLevel: number | undefined): Promise<ISendEventResponse>;
    /**
     * Create an m.beacon_info event
     * @returns
     */
    unstable_createLiveBeacon(roomId: Room["roomId"], beaconInfoContent: MBeaconInfoEventContent): Promise<ISendEventResponse>;
    /**
     * Upsert a live beacon event
     * using a specific m.beacon_info.* event variable type
     * @param roomId - string
     * @returns
     */
    unstable_setLiveBeacon(roomId: string, beaconInfoContent: MBeaconInfoEventContent): Promise<ISendEventResponse>;
    sendEvent<K extends keyof TimelineEvents>(roomId: string, eventType: K, content: TimelineEvents[K], txnId?: string): Promise<ISendEventResponse>;
    sendEvent<K extends keyof TimelineEvents>(roomId: string, threadId: string | null, eventType: K, content: TimelineEvents[K], txnId?: string): Promise<ISendEventResponse>;
    /**
     * If we expect that an event is part of a thread but is missing the relation
     * we need to add it manually, as well as the reply fallback
     */
    private addThreadRelationIfNeeded;
    /**
     * @param eventObject - An object with the partial structure of an event, to which event_id, user_id, room_id and origin_server_ts will be added.
     * @param txnId - Optional.
     * @returns Promise which resolves: to an empty object `{}`
     * @returns Rejects: with an error response.
     */
    private sendCompleteEvent;
    /**
     * encrypts the event if necessary; adds the event to the queue, or sends it; marks the event as sent/unsent
     * @returns returns a promise which resolves with the result of the send request
     */
    protected encryptAndSendEvent(room: Room | null, event: MatrixEvent, queryDict?: QueryDict): Promise<ISendEventResponse>;
    /**
     * Simply sends a delayed event without encrypting it.
     * TODO: Allow encrypted delayed events, and encrypt them properly
     * @param delayOpts - Properties of the delay for this event.
     * @returns returns a promise which resolves with the result of the delayed send request
     */
    protected encryptAndSendEvent(room: Room | null, event: MatrixEvent, delayOpts: SendDelayedEventRequestOpts, queryDict?: QueryDict): Promise<ISendEventResponse>;
    private encryptEventIfNeeded;
    /**
     * Determine whether a given event should be encrypted when we send it to the given room.
     *
     * This takes into account event type and room configuration.
     */
    private shouldEncryptEventForRoom;
    /**
     * Returns the eventType that should be used taking encryption into account
     * for a given eventType.
     * @param roomId - the room for the events `eventType` relates to
     * @param eventType - the event type
     * @returns the event type taking encryption into account
     */
    private getEncryptedIfNeededEventType;
    protected updatePendingEventStatus(room: Room | null, event: MatrixEvent, newStatus: EventStatus): void;
    private sendEventHttpRequest;
    /**
     * @param txnId -  transaction id. One will be made up if not supplied.
     * @param opts - Redact options
     * @returns Promise which resolves: TODO
     * @returns Rejects: with an error response.
     * @throws Error if called with `with_rel_types` (MSC3912) but the server does not support it.
     *         Callers should check whether the server supports MSC3912 via `MatrixClient.canSupport`.
     */
    redactEvent(roomId: string, eventId: string, txnId?: string | undefined, opts?: IRedactOpts): Promise<ISendEventResponse>;
    redactEvent(roomId: string, threadId: string | null, eventId: string, txnId?: string | undefined, opts?: IRedactOpts): Promise<ISendEventResponse>;
    /**
     * @param txnId - Optional.
     * @returns Promise which resolves: to an ISendEventResponse object
     * @returns Rejects: with an error response.
     */
    sendMessage(roomId: string, content: RoomMessageEventContent, txnId?: string): Promise<ISendEventResponse>;
    sendMessage(roomId: string, threadId: string | null, content: RoomMessageEventContent, txnId?: string): Promise<ISendEventResponse>;
    /**
     * @param txnId - Optional.
     * @returns
     * @returns Rejects: with an error response.
     */
    sendTextMessage(roomId: string, body: string, txnId?: string): Promise<ISendEventResponse>;
    sendTextMessage(roomId: string, threadId: string | null, body: string, txnId?: string): Promise<ISendEventResponse>;
    /**
     * @param txnId - Optional.
     * @returns Promise which resolves: to a ISendEventResponse object
     * @returns Rejects: with an error response.
     */
    sendNotice(roomId: string, body: string, txnId?: string): Promise<ISendEventResponse>;
    sendNotice(roomId: string, threadId: string | null, body: string, txnId?: string): Promise<ISendEventResponse>;
    /**
     * @param txnId - Optional.
     * @returns Promise which resolves: to a ISendEventResponse object
     * @returns Rejects: with an error response.
     */
    sendEmoteMessage(roomId: string, body: string, txnId?: string): Promise<ISendEventResponse>;
    sendEmoteMessage(roomId: string, threadId: string | null, body: string, txnId?: string): Promise<ISendEventResponse>;
    /**
     * @returns Promise which resolves: to a ISendEventResponse object
     * @returns Rejects: with an error response.
     */
    sendImageMessage(roomId: string, url: string, info?: ImageInfo, text?: string): Promise<ISendEventResponse>;
    sendImageMessage(roomId: string, threadId: string | null, url: string, info?: ImageInfo, text?: string): Promise<ISendEventResponse>;
    /**
     * @returns Promise which resolves: to a ISendEventResponse object
     * @returns Rejects: with an error response.
     */
    sendStickerMessage(roomId: string, url: string, info?: ImageInfo, text?: string): Promise<ISendEventResponse>;
    sendStickerMessage(roomId: string, threadId: string | null, url: string, info?: ImageInfo, text?: string): Promise<ISendEventResponse>;
    /**
     * @returns Promise which resolves: to a ISendEventResponse object
     * @returns Rejects: with an error response.
     */
    sendHtmlMessage(roomId: string, body: string, htmlBody: string): Promise<ISendEventResponse>;
    sendHtmlMessage(roomId: string, threadId: string | null, body: string, htmlBody: string): Promise<ISendEventResponse>;
    /**
     * @returns Promise which resolves: to a ISendEventResponse object
     * @returns Rejects: with an error response.
     */
    sendHtmlNotice(roomId: string, body: string, htmlBody: string): Promise<ISendEventResponse>;
    sendHtmlNotice(roomId: string, threadId: string | null, body: string, htmlBody: string): Promise<ISendEventResponse>;
    /**
     * @returns Promise which resolves: to a ISendEventResponse object
     * @returns Rejects: with an error response.
     */
    sendHtmlEmote(roomId: string, body: string, htmlBody: string): Promise<ISendEventResponse>;
    sendHtmlEmote(roomId: string, threadId: string | null, body: string, htmlBody: string): Promise<ISendEventResponse>;
    /**
     * Send a delayed timeline event.
     *
     * Note: This endpoint is unstable, and can throw an `Error`.
     *   Check progress on [MSC4140](https://github.com/matrix-org/matrix-spec-proposals/pull/4140) for more details.
     */
    _unstable_sendDelayedEvent<K extends keyof TimelineEvents>(roomId: string, delayOpts: SendDelayedEventRequestOpts, threadId: string | null, eventType: K, content: TimelineEvents[K], txnId?: string): Promise<SendDelayedEventResponse>;
    /**
     * Send a delayed sticky timeline event.
     *
     * Note: This endpoint is unstable, and can throw an `Error`.
     *   Check progress on [MSC4140](https://github.com/matrix-org/matrix-spec-proposals/pull/4140) and
     *   [MSC4354](https://github.com/matrix-org/matrix-spec-proposals/pull/4354) for more details.
     */
    _unstable_sendStickyDelayedEvent<K extends keyof TimelineEvents>(roomId: string, stickDuration: number, delayOpts: SendDelayedEventRequestOpts, threadId: string | null, eventType: K, content: TimelineEvents[K] & {
        msc4354_sticky_key: string;
    }, txnId?: string): Promise<SendDelayedEventResponse>;
    /**
     * Send a delayed state event.
     *
     * Note: This endpoint is unstable, and can throw an `Error`.
     *   Check progress on [MSC4140](https://github.com/matrix-org/matrix-spec-proposals/pull/4140) for more details.
     */
    _unstable_sendDelayedStateEvent<K extends keyof StateEvents>(roomId: string, delayOpts: SendDelayedEventRequestOpts, eventType: K, content: StateEvents[K], stateKey?: string, opts?: IRequestOpts): Promise<SendDelayedEventResponse>;
    /**
     * Send a sticky timeline event.
     *
     * Note: This endpoint is unstable, and can throw an `Error`.
     *   Check progress on [MSC4354](https://github.com/matrix-org/matrix-spec-proposals/pull/4354) for more details.
     */
    _unstable_sendStickyEvent<K extends keyof TimelineEvents>(roomId: string, stickDuration: number, threadId: string | null, eventType: K, content: TimelineEvents[K] & {
        msc4354_sticky_key: string;
    }, txnId?: string): Promise<ISendEventResponse>;
    /**
     * Get all pending delayed events for the calling user.
     *
     * Note: This endpoint is unstable, and can throw an `Error`.
     *   Check progress on [MSC4140](https://github.com/matrix-org/matrix-spec-proposals/pull/4140) for more details.
     */
    _unstable_getDelayedEvents(fromToken?: string): Promise<DelayedEventInfo>;
    /**
     * Manage a delayed event associated with the given delay_id.
     *
     * Note: This endpoint is unstable, and can throw an `Error`.
     *   Check progress on [MSC4140](https://github.com/matrix-org/matrix-spec-proposals/pull/4140) for more details.
     */
    _unstable_updateDelayedEvent(delayId: string, action: UpdateDelayedEventAction, requestOptions?: IRequestOpts): Promise<EmptyObject>;
    /**
     * Send a receipt.
     * @param event - The event being acknowledged
     * @param receiptType - The kind of receipt e.g. "m.read". Other than
     * ReceiptType.Read are experimental!
     * @param body - Additional content to send alongside the receipt.
     * @param unthreaded - An unthreaded receipt will clear room+thread notifications
     * @returns Promise which resolves: to an empty object `{}`
     * @returns Rejects: with an error response.
     */
    sendReceipt(event: MatrixEvent, receiptType: ReceiptType, body?: Record<string, any>, unthreaded?: boolean): Promise<EmptyObject>;
    /**
     * Send a read receipt.
     * @param event - The event that has been read.
     * @param receiptType - other than ReceiptType.Read are experimental! Optional.
     * @returns Promise which resolves: to an empty object `{}`
     * @returns Rejects: with an error response.
     */
    sendReadReceipt(event: MatrixEvent | null, receiptType?: ReceiptType, unthreaded?: boolean): Promise<EmptyObject | undefined>;
    /**
     * Set a marker to indicate the point in a room before which the user has read every
     * event. This can be retrieved from room account data (the event type is `m.fully_read`)
     * and displayed as a horizontal line in the timeline that is visually distinct to the
     * position of the user's own read receipt.
     * @param roomId - ID of the room that has been read
     * @param rmEventId - ID of the event that has been read
     * @param rrEvent - the event tracked by the read receipt. This is here for
     * convenience because the RR and the RM are commonly updated at the same time as each
     * other. The local echo of this receipt will be done if set. Optional.
     * @param rpEvent - the m.read.private read receipt event for when we don't
     * want other users to see the read receipts. This is experimental. Optional.
     * @returns Promise which resolves: the empty object, `{}`.
     */
    setRoomReadMarkers(roomId: string, rmEventId: string, rrEvent?: MatrixEvent, rpEvent?: MatrixEvent): Promise<EmptyObject>;
    sendRtcDecline(roomId: string, notificationEventId: string): Promise<ISendEventResponse>;
    /**
     * Get a preview of the given URL as of (roughly) the given point in time,
     * described as an object with OpenGraph keys and associated values.
     * Attributes may be synthesized where actual OG metadata is lacking.
     * Caches results to prevent hammering the server.
     * @param url - The URL to get preview data for
     * @param ts - The preferred point in time that the preview should
     * describe (ms since epoch).  The preview returned will either be the most
     * recent one preceding this timestamp if available, or failing that the next
     * most recent available preview.
     * @returns Promise which resolves: Object of OG metadata.
     * @returns Rejects: with an error response.
     * May return synthesized attributes if the URL lacked OG meta.
     */
    getUrlPreview(url: string, ts: number): Promise<IPreviewUrlResponse>;
    /**
     * @returns Promise which resolves: to an empty object `{}`
     * @returns Rejects: with an error response.
     */
    sendTyping(roomId: string, isTyping: boolean, timeoutMs: number): Promise<EmptyObject>;
    /**
     * Determines the history of room upgrades for a given room, as far as the
     * client can see. Returns an array of Rooms where the first entry is the
     * oldest and the last entry is the newest (likely current) room. If the
     * provided room is not found, this returns an empty list. This works in
     * both directions, looking for older and newer rooms of the given room.
     * @param roomId - The room ID to search from
     * @param verifyLinks - If true, the function will only return rooms
     * which can be proven to be linked. For example, rooms which have a create
     * event pointing to an old room which the client is not aware of or doesn't
     * have a matching tombstone would not be returned.
     * @param msc3946ProcessDynamicPredecessor - if true, look for
     * m.room.predecessor state events as well as create events, and prefer
     * predecessor events where they exist (MSC3946).
     * @returns An array of rooms representing the upgrade
     * history.
     */
    getRoomUpgradeHistory(roomId: string, verifyLinks?: boolean, msc3946ProcessDynamicPredecessor?: boolean): Room[];
    private findPredecessorRooms;
    private findSuccessorRooms;
    /**
     * Send an invite to the given user to join the given room.
     *
     * @param roomId - The ID of the room to which the user should be invited.
     * @param userId - The ID of the user that should be invited.
     * @param opts - Optional reason object. For backwards compatibility, a string is also accepted, and will be interpreted as a reason.
     *
     * @returns An empty object.
     */
    invite(roomId: string, userId: string, opts?: InviteOpts | string): Promise<EmptyObject>;
    /**
     * Invite a user to a room based on their email address.
     * @param roomId - The room to invite the user to.
     * @param email - The email address to invite.
     * @returns Promise which resolves: `{}` an empty object.
     * @returns Rejects: with an error response.
     */
    inviteByEmail(roomId: string, email: string): Promise<EmptyObject>;
    /**
     * Invite a user to a room based on a third-party identifier.
     * @param roomId - The room to invite the user to.
     * @param medium - The medium to invite the user e.g. "email".
     * @param address - The address for the specified medium.
     * @returns Promise which resolves: `{}` an empty object.
     * @returns Rejects: with an error response.
     */
    inviteByThreePid(roomId: string, medium: string, address: string): Promise<EmptyObject>;
    /**
     * @returns Promise which resolves: `{}` an empty object.
     * @returns Rejects: with an error response.
     */
    leave(roomId: string): Promise<EmptyObject>;
    /**
     * Leaves all rooms in the chain of room upgrades based on the given room. By
     * default, this will leave all the previous and upgraded rooms, including the
     * given room. To only leave the given room and any previous rooms, keeping the
     * upgraded (modern) rooms untouched supply `false` to `includeFuture`.
     * @param roomId - The room ID to start leaving at
     * @param includeFuture - If true, the whole chain (past and future) of
     * upgraded rooms will be left.
     * @returns Promise which resolves when completed with an object keyed
     * by room ID and value of the error encountered when leaving or null.
     */
    leaveRoomChain(roomId: string, includeFuture?: boolean): Promise<{
        [roomId: string]: Error | MatrixError | null;
    }>;
    /**
     * @param reason - Optional.
     * @returns Promise which resolves: TODO
     * @returns Rejects: with an error response.
     */
    ban(roomId: string, userId: string, reason?: string): Promise<EmptyObject>;
    /**
     * @param deleteRoom - True to delete the room from the store on success.
     * Default: true.
     * @returns Promise which resolves: `{}` an empty object.
     * @returns Rejects: with an error response.
     */
    forget(roomId: string, deleteRoom?: boolean): Promise<EmptyObject>;
    /**
     * @returns Promise which resolves: Object (currently empty)
     * @returns Rejects: with an error response.
     */
    unban(roomId: string, userId: string): Promise<EmptyObject>;
    /**
     * @param reason - Optional.
     * @returns Promise which resolves: `{}` an empty object.
     * @returns Rejects: with an error response.
     */
    kick(roomId: string, userId: string, reason?: string): Promise<EmptyObject>;
    private membershipChange;
    /**
     * Obtain a dict of actions which should be performed for this event according
     * to the push rules for this user.  Caches the dict on the event.
     * @param event - The event to get push actions for.
     * @param forceRecalculate - forces to recalculate actions for an event
     * Useful when an event just got decrypted
     * @returns A dict of actions to perform.
     */
    getPushActionsForEvent(event: MatrixEvent, forceRecalculate?: boolean): IActionsObject | null;
    /**
     * Obtain a dict of actions which should be performed for this event according
     * to the push rules for this user.  Caches the dict on the event.
     * @param event - The event to get push actions for.
     * @param forceRecalculate - forces to recalculate actions for an event
     * Useful when an event just got decrypted
     * @returns A dict of actions to perform.
     */
    getPushDetailsForEvent(event: MatrixEvent, forceRecalculate?: boolean): PushDetails | null;
    /**
     * @param info - The kind of info to set (e.g. 'avatar_url')
     * @param data - The JSON object to set.
     * @returns
     * @returns Rejects: with an error response.
     */
    setProfileInfo(info: "avatar_url", data: {
        avatar_url: string;
    }): Promise<EmptyObject>;
    setProfileInfo(info: "displayname", data: {
        displayname: string;
    }): Promise<EmptyObject>;
    /**
     * @returns Promise which resolves: `{}` an empty object.
     * @returns Rejects: with an error response.
     */
    setDisplayName(name: string): Promise<EmptyObject>;
    /**
     * @returns Promise which resolves: `{}` an empty object.
     * @returns Rejects: with an error response.
     */
    setAvatarUrl(url: string): Promise<EmptyObject>;
    /**
     * Turn an MXC URL into an HTTP one. <strong>This method is experimental and
     * may change.</strong>
     * @param mxcUrl - The MXC URL
     * @param width - The desired width of the thumbnail.
     * @param height - The desired height of the thumbnail.
     * @param resizeMethod - The thumbnail resize method to use, either
     * "crop" or "scale".
     * @param allowDirectLinks - If true, return any non-mxc URLs
     * directly. Fetching such URLs will leak information about the user to
     * anyone they share a room with. If false, will return null for such URLs.
     * @param allowRedirects - If true, the caller supports the URL being 307 or
     * 308 redirected to another resource upon request. If false, redirects
     * are not expected. Implied `true` when `useAuthentication` is `true`.
     * @param useAuthentication - If true, the caller supports authenticated
     * media and wants an authentication-required URL. Note that server support
     * for authenticated media will *not* be checked - it is the caller's responsibility
     * to do so before calling this function. Note also that `useAuthentication`
     * implies `allowRedirects`. Defaults to false (unauthenticated endpoints).
     * @returns the avatar URL or null.
     */
    mxcUrlToHttp(mxcUrl: string, width?: number, height?: number, resizeMethod?: string, allowDirectLinks?: boolean, allowRedirects?: boolean, useAuthentication?: boolean): string | null;
    /**
     * Specify the set_presence value to be used for subsequent calls to the Sync API.
     * This has an advantage over calls to the PUT /presence API in that it
     * doesn't clobber status_msg set by other devices.
     * @param presence - the presence to specify to set_presence of sync calls
     */
    setSyncPresence(presence?: SetPresence): Promise<void>;
    /**
     * @param opts - Options to apply
     * @returns Promise which resolves
     * @returns Rejects: with an error response.
     * @throws If 'presence' isn't a valid presence enum value.
     */
    setPresence(opts: IPresenceOpts): Promise<void>;
    /**
     * @param userId - The user to get presence for
     * @returns Promise which resolves: The presence state for this user.
     * @returns Rejects: with an error response.
     */
    getPresence(userId: string): Promise<IStatusResponse>;
    /**
     * Retrieve older messages from the given room and put them in the timeline.
     *
     * If this is called multiple times whilst a request is ongoing, the <i>same</i>
     * Promise will be returned. If there was a problem requesting scrollback, there
     * will be a small delay before another request can be made (to prevent tight-looping
     * when there is no connection).
     *
     * @param room - The room to get older messages in.
     * @param limit - Optional. The maximum number of previous events to
     * pull in. Default: 30.
     * @returns Promise which resolves: Room. If you are at the beginning
     * of the timeline, `Room.oldState.paginationToken` will be
     * `null`.
     * @returns Rejects: with an error response.
     */
    scrollback(room: Room, limit?: number): Promise<Room>;
    getEventMapper(options?: MapperOpts): EventMapper;
    /**
     * Get an EventTimeline for the given event
     *
     * <p>If the EventTimelineSet object already has the given event in its store, the
     * corresponding timeline will be returned. Otherwise, a /context request is
     * made, and used to construct an EventTimeline.
     * If the event does not belong to this EventTimelineSet then undefined will be returned.
     *
     * @param timelineSet -  The timelineSet to look for the event in, must be bound to a room
     * @param eventId -  The ID of the event to look for
     *
     * @returns Promise which resolves:
     *    {@link EventTimeline} including the given event
     */
    getEventTimeline(timelineSet: EventTimelineSet, eventId: string): Promise<Optional<EventTimeline>>;
    getThreadTimeline(timelineSet: EventTimelineSet, eventId: string): Promise<EventTimeline | undefined>;
    /**
     * Get an EventTimeline for the latest events in the room. This will just
     * call `/messages` to get the latest message in the room, then use
     * `client.getEventTimeline(...)` to construct a new timeline from it.
     *
     * @param timelineSet -  The timelineSet to find or add the timeline to
     *
     * @returns Promise which resolves:
     *    {@link EventTimeline} timeline with the latest events in the room
     */
    getLatestTimeline(timelineSet: EventTimelineSet): Promise<Optional<EventTimeline>>;
    /**
     * Makes a request to /messages with the appropriate lazy loading filter set.
     * XXX: if we do get rid of scrollback (as it's not used at the moment),
     * we could inline this method again in paginateEventTimeline as that would
     * then be the only call-site
     * @param limit - the maximum amount of events the retrieve
     * @param dir - 'f' or 'b'
     * @param timelineFilter - the timeline filter to pass
     */
    createMessagesRequest(roomId: string, fromToken: string | null, limit: number | undefined, dir: Direction, timelineFilter?: Filter): Promise<IMessagesResponse>;
    /**
     * Makes a request to /messages with the appropriate lazy loading filter set.
     * XXX: if we do get rid of scrollback (as it's not used at the moment),
     * we could inline this method again in paginateEventTimeline as that would
     * then be the only call-site
     * @param limit - the maximum amount of events the retrieve
     * @param dir - 'f' or 'b'
     * @param timelineFilter - the timeline filter to pass
     */
    createThreadListMessagesRequest(roomId: string, fromToken: string | null, limit?: number, dir?: Direction, threadListType?: ThreadFilterType | null, timelineFilter?: Filter): Promise<IMessagesResponse>;
    /**
     * Take an EventTimeline, and back/forward-fill results.
     *
     * @param eventTimeline - timeline object to be updated
     *
     * @returns Promise which resolves to a boolean: false if there are no
     *    events and we reached either end of the timeline; else true.
     */
    paginateEventTimeline(eventTimeline: EventTimeline, opts: IPaginateOpts): Promise<boolean>;
    /**
     * Reset the notifTimelineSet entirely, paginating in some historical notifs as
     * a starting point for subsequent pagination.
     */
    resetNotifTimelineSet(): void;
    /**
     * Peek into a room and receive updates about the room. This only works if the
     * history visibility for the room is world_readable.
     * @param roomId - The room to attempt to peek into.
     * @param limit - The number of timeline events to initially retrieve.
     * @returns Promise which resolves: Room object
     * @returns Rejects: with an error response.
     */
    peekInRoom(roomId: string, limit?: number): Promise<Room>;
    /**
     * Stop any ongoing room peeking.
     */
    stopPeeking(): void;
    /**
     * Set r/w flags for guest access in a room.
     * @param roomId - The room to configure guest access in.
     * @param opts - Options
     * @returns Promise which resolves
     * @returns Rejects: with an error response.
     */
    setGuestAccess(roomId: string, opts: IGuestAccessOpts): Promise<void>;
    /**
     * Requests an email verification token for the purposes of registration.
     * This API requests a token from the homeserver.
     * The doesServerRequireIdServerParam() method can be used to determine if
     * the server requires the id_server parameter to be provided.
     *
     * Parameters and return value are as for requestEmailToken

     * @param email - As requestEmailToken
     * @param clientSecret - As requestEmailToken
     * @param sendAttempt - As requestEmailToken
     * @param nextLink - As requestEmailToken
     * @returns Promise which resolves: As requestEmailToken
     */
    requestRegisterEmailToken(email: string, clientSecret: string, sendAttempt: number, nextLink?: string): Promise<IRequestTokenResponse>;
    /**
     * Requests a text message verification token for the purposes of registration.
     * This API requests a token from the homeserver.
     * The doesServerRequireIdServerParam() method can be used to determine if
     * the server requires the id_server parameter to be provided.
     *
     * @param phoneCountry - The ISO 3166-1 alpha-2 code for the country in which
     *    phoneNumber should be parsed relative to.
     * @param phoneNumber - The phone number, in national or international format
     * @param clientSecret - As requestEmailToken
     * @param sendAttempt - As requestEmailToken
     * @param nextLink - As requestEmailToken
     * @returns Promise which resolves: As requestEmailToken
     */
    requestRegisterMsisdnToken(phoneCountry: string, phoneNumber: string, clientSecret: string, sendAttempt: number, nextLink?: string): Promise<IRequestMsisdnTokenResponse>;
    /**
     * Requests an email verification token for the purposes of adding a
     * third party identifier to an account.
     * This API requests a token from the homeserver.
     * The doesServerRequireIdServerParam() method can be used to determine if
     * the server requires the id_server parameter to be provided.
     * If an account with the given email address already exists and is
     * associated with an account other than the one the user is authed as,
     * it will either send an email to the address informing them of this
     * or return M_THREEPID_IN_USE (which one is up to the homeserver).
     *
     * @param email - As requestEmailToken
     * @param clientSecret - As requestEmailToken
     * @param sendAttempt - As requestEmailToken
     * @param nextLink - As requestEmailToken
     * @returns Promise which resolves: As requestEmailToken
     */
    requestAdd3pidEmailToken(email: string, clientSecret: string, sendAttempt: number, nextLink?: string): Promise<IRequestTokenResponse>;
    /**
     * Requests a text message verification token for the purposes of adding a
     * third party identifier to an account.
     * This API proxies the identity server /validate/email/requestToken API,
     * adding specific behaviour for the addition of phone numbers to an
     * account, as requestAdd3pidEmailToken.
     *
     * @param phoneCountry - As requestRegisterMsisdnToken
     * @param phoneNumber - As requestRegisterMsisdnToken
     * @param clientSecret - As requestEmailToken
     * @param sendAttempt - As requestEmailToken
     * @param nextLink - As requestEmailToken
     * @returns Promise which resolves: As requestEmailToken
     */
    requestAdd3pidMsisdnToken(phoneCountry: string, phoneNumber: string, clientSecret: string, sendAttempt: number, nextLink?: string): Promise<IRequestMsisdnTokenResponse>;
    /**
     * Requests an email verification token for the purposes of resetting
     * the password on an account.
     * This API proxies the identity server /validate/email/requestToken API,
     * adding specific behaviour for the password resetting. Specifically,
     * if no account with the given email address exists, it may either
     * return M_THREEPID_NOT_FOUND or send an email
     * to the address informing them of this (which one is up to the homeserver).
     *
     * requestEmailToken calls the equivalent API directly on the identity server,
     * therefore bypassing the password reset specific logic.
     *
     * @param email - As requestEmailToken
     * @param clientSecret - As requestEmailToken
     * @param sendAttempt - As requestEmailToken
     * @param nextLink - As requestEmailToken
     * @returns Promise which resolves: As requestEmailToken
     */
    requestPasswordEmailToken(email: string, clientSecret: string, sendAttempt: number, nextLink?: string): Promise<IRequestTokenResponse>;
    /**
     * Requests a text message verification token for the purposes of resetting
     * the password on an account.
     * This API proxies the identity server /validate/email/requestToken API,
     * adding specific behaviour for the password resetting, as requestPasswordEmailToken.
     *
     * @param phoneCountry - As requestRegisterMsisdnToken
     * @param phoneNumber - As requestRegisterMsisdnToken
     * @param clientSecret - As requestEmailToken
     * @param sendAttempt - As requestEmailToken
     * @param nextLink - As requestEmailToken
     * @returns Promise which resolves: As requestEmailToken
     */
    requestPasswordMsisdnToken(phoneCountry: string, phoneNumber: string, clientSecret: string, sendAttempt: number, nextLink: string): Promise<IRequestMsisdnTokenResponse>;
    /**
     * Internal utility function for requesting validation tokens from usage-specific
     * requestToken endpoints.
     *
     * @param endpoint - The endpoint to send the request to
     * @param params - Parameters for the POST request
     * @returns Promise which resolves: As requestEmailToken
     */
    private requestTokenFromEndpoint;
    /**
     * Get the room-kind push rule associated with a room.
     * @param scope - "global" or device-specific.
     * @param roomId - the id of the room.
     * @returns the rule or undefined.
     */
    getRoomPushRule(scope: "global" | "device", roomId: string): IPushRule | undefined;
    /**
     * Set a room-kind muting push rule in a room.
     * The operation also updates MatrixClient.pushRules at the end.
     * @param scope - "global" or device-specific.
     * @param roomId - the id of the room.
     * @param mute - the mute state.
     * @returns Promise which resolves: result object
     * @returns Rejects: with an error response.
     */
    setRoomMutePushRule(scope: "global" | "device", roomId: string, mute: boolean): Promise<void> | undefined;
    searchMessageText(opts: ISearchOpts): Promise<ISearchResponse>;
    /**
     * Perform a server-side search for room events.
     *
     * The returned promise resolves to an object containing the fields:
     *
     *  * count:       estimate of the number of results
     *  * next_batch:  token for back-pagination; if undefined, there are no more results
     *  * highlights:  a list of words to highlight from the stemming algorithm
     *  * results:     a list of results
     *
     * Each entry in the results list is a SearchResult.
     *
     * @returns Promise which resolves: result object
     * @returns Rejects: with an error response.
     */
    searchRoomEvents(opts: IEventSearchOpts): Promise<ISearchResults>;
    /**
     * Take a result from an earlier searchRoomEvents call, and backfill results.
     *
     * @param searchResults -  the results object to be updated
     * @returns Promise which resolves: updated result object
     * @returns Rejects: with an error response.
     */
    backPaginateRoomEventsSearch<T extends ISearchResults>(searchResults: T): Promise<T>;
    /**
     * helper for searchRoomEvents and backPaginateRoomEventsSearch. Processes the
     * response from the API call and updates the searchResults
     *
     * @returns searchResults
     * @internal
     */
    processRoomEventsSearch<T extends ISearchResults>(searchResults: T, response: ISearchResponse): T;
    /**
     * Populate the store with rooms the user has left.
     * @returns Promise which resolves: TODO - Resolved when the rooms have
     * been added to the data store.
     * @returns Rejects: with an error response.
     */
    syncLeftRooms(): Promise<Room[]>;
    /**
     * Create a new filter.
     * @param content - The HTTP body for the request
     * @returns Promise which resolves to a Filter object.
     * @returns Rejects: with an error response.
     */
    createFilter(content: IFilterDefinition): Promise<Filter>;
    /**
     * Retrieve a filter.
     * @param userId - The user ID of the filter owner
     * @param filterId - The filter ID to retrieve
     * @param allowCached - True to allow cached filters to be returned.
     * Default: True.
     * @returns Promise which resolves: a Filter object
     * @returns Rejects: with an error response.
     */
    getFilter(userId: string, filterId: string, allowCached: boolean): Promise<Filter>;
    /**
     * @returns Filter ID
     */
    getOrCreateFilter(filterName: string, filter: Filter): Promise<string>;
    /**
     * Gets a bearer token from the homeserver that the user can
     * present to a third party in order to prove their ownership
     * of the Matrix account they are logged into.
     * @returns Promise which resolves: Token object
     * @returns Rejects: with an error response.
     */
    getOpenIdToken(): Promise<IOpenIDToken>;
    private startCallEventHandler;
    private startMatrixRTC;
    /**
     * Once the client has been initialised, we want to clear notifications we
     * know for a fact should be here.
     * This issue should also be addressed on synapse's side and is tracked as part
     * of https://github.com/matrix-org/synapse/issues/14837
     *
     * We consider a room or a thread as fully read if the current user has sent
     * the last event in the live timeline of that context and if the read receipt
     * we have on record matches.
     */
    private fixupRoomNotifications;
    /**
     * @returns Promise which resolves: ITurnServerResponse object
     * @returns Rejects: with an error response.
     */
    turnServer(): Promise<ITurnServerResponse>;
    /**
     * Get the TURN servers for this homeserver.
     * @returns The servers or an empty list.
     */
    getTurnServers(): ITurnServer[];
    /**
     * Get the unix timestamp (in milliseconds) at which the current
     * TURN credentials (from getTurnServers) expire
     * @returns The expiry timestamp in milliseconds
     */
    getTurnServersExpiry(): number;
    get pollingTurnServers(): boolean;
    checkTurnServers(): Promise<boolean | undefined>;
    /**
     * Set whether to allow a fallback ICE server should be used for negotiating a
     * WebRTC connection if the homeserver doesn't provide any servers. Defaults to
     * false.
     *
     */
    setFallbackICEServerAllowed(allow: boolean): void;
    /**
     * Get whether to allow a fallback ICE server should be used for negotiating a
     * WebRTC connection if the homeserver doesn't provide any servers. Defaults to
     * false.
     *
     * @returns
     */
    isFallbackICEServerAllowed(): boolean;
    /**
     * Determines if the current user is an administrator of the Synapse homeserver.
     * Returns false if untrue or the homeserver does not appear to be a Synapse
     * homeserver. <strong>This function is implementation specific and may change
     * as a result.</strong>
     * @returns true if the user appears to be a Synapse administrator.
     */
    isSynapseAdministrator(): Promise<boolean>;
    /**
     * Performs a whois lookup on a user using Synapse's administrator API.
     * <strong>This function is implementation specific and may change as a
     * result.</strong>
     * @param userId - the User ID to look up.
     * @returns the whois response - see Synapse docs for information.
     */
    whoisSynapseUser(userId: string): Promise<ISynapseAdminWhoisResponse>;
    /**
     * Deactivates a user using Synapse's administrator API. <strong>This
     * function is implementation specific and may change as a result.</strong>
     * @param userId - the User ID to deactivate.
     * @returns the deactivate response - see Synapse docs for information.
     */
    deactivateSynapseUser(userId: string): Promise<ISynapseAdminDeactivateResponse>;
    protected fetchClientWellKnown(): Promise<void>;
    getClientWellKnown(): IClientWellKnown | undefined;
    waitForClientWellKnown(): Promise<IClientWellKnown>;
    /**
     * store client options with boolean/string/numeric values
     * to know in the next session what flags the sync data was
     * created with (e.g. lazy loading)
     * @returns for store operation
     */
    storeClientOptions(): Promise<void>;
    /**
     * Gets a set of room IDs in common with another user.
     *
     * Note: This endpoint is unstable, and can throw an `Error`.
     *   Check progress on [MSC2666](https://github.com/matrix-org/matrix-spec-proposals/pull/2666) for more details.
     *
     * @param userId - The userId to check.
     * @returns Promise which resolves to an array of rooms
     * @returns Rejects: with an error response.
     */
    _unstable_getSharedRooms(userId: string): Promise<string[]>;
    /**
     * Get the API versions supported by the server, along with any
     * unstable APIs it supports
     * @returns The server /versions response
     */
    getVersions(): Promise<IServerVersions>;
    /**
     * Check if a particular spec version is supported by the server.
     * @param version - The spec version (such as "r0.5.0") to check for.
     * @returns Whether it is supported
     */
    isVersionSupported(version: string): Promise<boolean>;
    /**
     * Query the server to see if it lists support for an unstable feature
     * in the /versions response
     * @param feature - the feature name
     * @returns true if the feature is supported
     */
    doesServerSupportUnstableFeature(feature: string): Promise<boolean>;
    /**
     * Query the server to see if it is forcing encryption to be enabled for
     * a given room preset, based on the /versions response.
     * @param presetName - The name of the preset to check.
     * @returns true if the server is forcing encryption
     * for the preset.
     */
    doesServerForceEncryptionForPreset(presetName: Preset): Promise<boolean>;
    doesServerSupportThread(): Promise<{
        threads: FeatureSupport;
        list: FeatureSupport;
        fwdPagination: FeatureSupport;
    }>;
    /**
     * Get if lazy loading members is being used.
     * @returns Whether or not members are lazy loaded by this client
     */
    hasLazyLoadMembersEnabled(): boolean;
    /**
     * Set a function which is called when /sync returns a 'limited' response.
     * It is called with a room ID and returns a boolean. It should return 'true' if the SDK
     * can SAFELY remove events from this room. It may not be safe to remove events if there
     * are other references to the timelines for this room, e.g because the client is
     * actively viewing events in this room.
     * Default: returns false.
     * @param cb - The callback which will be invoked.
     */
    setCanResetTimelineCallback(cb: ResetTimelineCallback): void;
    /**
     * Get the callback set via `setCanResetTimelineCallback`.
     * @returns The callback or null
     */
    getCanResetTimelineCallback(): ResetTimelineCallback | undefined;
    /**
     * Returns relations for a given event. Handles encryption transparently,
     * with the caveat that the amount of events returned might be 0, even though you get a nextBatch.
     * When the returned promise resolves, all messages should have finished trying to decrypt.
     * @param roomId - the room of the event
     * @param eventId - the id of the event
     * @param relationType - the rel_type of the relations requested
     * @param eventType - the event type of the relations requested
     * @param opts - options with optional values for the request.
     * @returns an object with `events` as `MatrixEvent[]` and optionally `nextBatch` if more relations are available.
     */
    relations(roomId: string, eventId: string, relationType: RelationType | string | null, eventType?: EventType | string | null, opts?: IRelationsRequestOpts): Promise<{
        originalEvent?: MatrixEvent | null;
        events: MatrixEvent[];
        nextBatch?: string | null;
        prevBatch?: string | null;
    }>;
    /**
     * Generates a random string suitable for use as a client secret. <strong>This
     * method is experimental and may change.</strong>
     * @returns A new client secret
     */
    generateClientSecret(): string;
    /**
     * Attempts to decrypt an event
     * @param event - The event to decrypt
     * @returns A decryption promise
     */
    decryptEventIfNeeded(event: MatrixEvent, options?: IDecryptOptions): Promise<void>;
    private termsUrlForService;
    /**
     * Get the Homeserver URL of this client
     * @returns Homeserver URL of this client
     */
    getHomeserverUrl(): string;
    /**
     * Get the identity server URL of this client
     * @param stripProto - whether or not to strip the protocol from the URL
     * @returns Identity server URL of this client
     */
    getIdentityServerUrl(stripProto?: boolean): string | undefined;
    /**
     * Set the identity server URL of this client
     * @param url - New identity server URL
     */
    setIdentityServerUrl(url?: string): void;
    /**
     * Get the access token associated with this account.
     * @returns The access_token or null
     */
    getAccessToken(): string | null;
    /**
     * Get the refresh token associated with this account.
     * @returns The refresh_token or null
     */
    getRefreshToken(): string | null;
    /**
     * Set the access token associated with this account.
     * @param token - The new access token.
     */
    setAccessToken(token: string): void;
    /**
     * @returns true if there is a valid access_token for this client.
     */
    isLoggedIn(): boolean;
    /**
     * Make up a new transaction id
     *
     * @returns a new, unique, transaction id
     */
    makeTxnId(): string;
    /**
     * Check whether a username is available prior to registration. An error response
     * indicates an invalid/unavailable username.
     * @param username - The username to check the availability of.
     * @returns Promise which resolves: to boolean of whether the username is available.
     */
    isUsernameAvailable(username: string): Promise<boolean>;
    /**
     * @param bindThreepids - Set key 'email' to true to bind any email
     *     threepid uses during registration in the identity server. Set 'msisdn' to
     *     true to bind msisdn.
     * @returns Promise which resolves to a RegisterResponse object
     * @returns Rejects: with an error response.
     */
    register(username: string, password: string, sessionId: string | null, auth: {
        session?: string;
        type: string;
    }, bindThreepids?: {
        email?: boolean;
        msisdn?: boolean;
    }, guestAccessToken?: string, inhibitLogin?: boolean): Promise<RegisterResponse>;
    /**
     * Register a guest account.
     * This method returns the auth info needed to create a new authenticated client,
     * Remember to call `setGuest(true)` on the (guest-)authenticated client, e.g:
     * ```javascript
     * const tmpClient = await sdk.createClient(MATRIX_INSTANCE);
     * const { user_id, device_id, access_token } = tmpClient.registerGuest();
     * const client = createClient({
     *   baseUrl: MATRIX_INSTANCE,
     *   accessToken: access_token,
     *   userId: user_id,
     *   deviceId: device_id,
     * })
     * client.setGuest(true);
     * ```
     *
     * @param body - JSON HTTP body to provide.
     * @returns Promise which resolves: JSON object that contains:
     *                   `{ user_id, device_id, access_token, home_server }`
     * @returns Rejects: with an error response.
     */
    registerGuest({ body }?: {
        body?: RegisterRequest;
    }): Promise<RegisterResponse>;
    /**
     * @param data - parameters for registration request
     * @param kind - type of user to register. may be "guest"
     * @returns Promise which resolves: to the /register response
     * @returns Rejects: with an error response.
     */
    registerRequest(data: RegisterRequest, kind?: string): Promise<RegisterResponse>;
    /**
     * Refreshes an access token using a provided refresh token. The refresh token
     * must be valid for the current access token known to the client instance.
     *
     * Note that this function will not cause a logout if the token is deemed
     * unknown by the server - the caller is responsible for managing logout
     * actions on error.
     * @param refreshToken - The refresh token.
     * @returns Promise which resolves to the new token.
     * @returns Rejects with an error response.
     */
    refreshToken(refreshToken: string): Promise<IRefreshTokenResponse>;
    /**
     * @returns Promise which resolves to the available login flows
     * @returns Rejects: with an error response.
     */
    loginFlows(): Promise<ILoginFlowsResponse>;
    /**
     * @returns Promise which resolves to a LoginResponse object
     * @returns Rejects: with an error response.
     *
     * @deprecated This method has unintuitive behaviour: it updates the `MatrixClient` instance with *some* of the
     *    returned credentials. Instead, call {@link loginRequest} and create a new `MatrixClient` instance using the
     *    results. See https://github.com/matrix-org/matrix-js-sdk/issues/4502.
     */
    login(loginType: LoginRequest["type"], data: Omit<LoginRequest, "type">): Promise<LoginResponse>;
    /**
     * @returns Promise which resolves to a LoginResponse object
     * @returns Rejects: with an error response.
     *
     * @deprecated This method has unintuitive behaviour: it updates the `MatrixClient` instance with *some* of the
     *   returned credentials. Instead, call {@link loginRequest} with `data.type: "m.login.password"`, and create a new
     *   `MatrixClient` instance using the results. See https://github.com/matrix-org/matrix-js-sdk/issues/4502.
     */
    loginWithPassword(user: string, password: string): Promise<LoginResponse>;
    /**
     * @param redirectUrl - The URL to redirect to after the HS
     * authenticates with CAS.
     * @returns The HS URL to hit to begin the CAS login process.
     */
    getCasLoginUrl(redirectUrl: string): string;
    /**
     * @param redirectUrl - The URL to redirect to after the HS
     *     authenticates with the SSO.
     * @param loginType - The type of SSO login we are doing (sso or cas).
     *     Defaults to 'sso'.
     * @param idpId - The ID of the Identity Provider being targeted, optional.
     * @param action - the SSO flow to indicate to the IdP, optional.
     * @returns The HS URL to hit to begin the SSO login process.
     */
    getSsoLoginUrl(redirectUrl: string, loginType?: string, idpId?: string, action?: SSOAction): string;
    /**
     * @param token - Login token previously received from homeserver
     * @returns Promise which resolves to a LoginResponse object
     * @returns Rejects: with an error response.
     *
     * @deprecated This method has unintuitive behaviour: it updates the `MatrixClient` instance with *some* of the
     *   returned credentials. Instead, call {@link loginRequest} with `data.type: "m.login.token"`, and create a new
     *   `MatrixClient` instance using the results. See https://github.com/matrix-org/matrix-js-sdk/issues/4502.
     */
    loginWithToken(token: string): Promise<LoginResponse>;
    /**
     * Sends a `POST /login` request to the server.
     *
     * If successful, this will create a new device and access token for the user.
     *
     * @see {@link MatrixClient.loginFlows} which makes a `GET /login` request.
     * @see https://spec.matrix.org/v1.13/client-server-api/#post_matrixclientv3login
     *
     * @param data - Credentials and other details for the login request.
     */
    loginRequest(data: LoginRequest): Promise<LoginResponse>;
    /**
     * Logs out the current session.
     * Obviously, further calls that require authorisation should fail after this
     * method is called. The state of the MatrixClient object is not affected:
     * it is up to the caller to either reset or destroy the MatrixClient after
     * this method succeeds.
     * @param stopClient - whether to stop the client before calling /logout to prevent invalid token errors.
     * @returns Promise which resolves: On success, the empty object `{}`
     */
    logout(stopClient?: boolean): Promise<EmptyObject>;
    /**
     * Deactivates the logged-in account.
     * Obviously, further calls that require authorisation should fail after this
     * method is called. The state of the MatrixClient object is not affected:
     * it is up to the caller to either reset or destroy the MatrixClient after
     * this method succeeds.
     * @param auth - Optional. Auth data to supply for User-Interactive auth.
     * @param erase - Optional. If set, send as `erase` attribute in the
     * JSON request body, indicating whether the account should be erased. Defaults
     * to false.
     * @returns Promise which resolves: On success, the empty object
     */
    deactivateAccount(auth?: AuthDict, erase?: boolean): Promise<{
        id_server_unbind_result: IdServerUnbindResult;
    }>;
    /**
     * Make a request for an `m.login.token` to be issued as per
     * https://spec.matrix.org/v1.7/client-server-api/#post_matrixclientv1loginget_token
     *
     * The server may require User-Interactive auth.
     *
     * @param auth - Optional. Auth data to supply for User-Interactive auth.
     * @returns Promise which resolves: On success, the token response
     * or UIA auth data.
     */
    requestLoginToken(auth?: AuthDict): Promise<LoginTokenPostResponse>;
    /**
     * Get the fallback URL to use for unknown interactive-auth stages.
     *
     * @param loginType -     the type of stage being attempted
     * @param authSessionId - the auth session ID provided by the homeserver
     *
     * @returns HS URL to hit to for the fallback interface
     */
    getFallbackAuthUrl(loginType: string, authSessionId: string): string;
    /**
     * Create a new room.
     * @param options - a list of options to pass to the /createRoom API.
     * @returns Promise which resolves: `{room_id: {string}}`
     * @returns Rejects: with an error response.
     */
    createRoom(options: ICreateRoomOpts): Promise<{
        room_id: string;
    }>;
    /**
     * Fetches relations for a given event
     * @param roomId - the room of the event
     * @param eventId - the id of the event
     * @param relationType - the rel_type of the relations requested
     * @param eventType - the event type of the relations requested
     * @param opts - options with optional values for the request.
     * @returns the response, with chunk, prev_batch and, next_batch.
     */
    fetchRelations(roomId: string, eventId: string, relationType: RelationType | string | null, eventType?: EventType | string | null, opts?: IRelationsRequestOpts): Promise<IRelationsResponse>;
    /**
     * @returns Promise which resolves: TODO
     * @returns Rejects: with an error response.
     */
    roomState(roomId: string): Promise<IStateEventWithRoomId[]>;
    /**
     * Get an event in a room by its event id.
     *
     * @returns Promise which resolves to an object containing the event.
     * @returns Rejects: with an error response.
     */
    fetchRoomEvent(roomId: string, eventId: string): Promise<Partial<IEvent>>;
    /**
     * @param includeMembership - the membership type to include in the response
     * @param excludeMembership - the membership type to exclude from the response
     * @param atEventId - the id of the event for which moment in the timeline the members should be returned for
     * @returns Promise which resolves: dictionary of userid to profile information
     * @returns Rejects: with an error response.
     */
    members(roomId: string, includeMembership?: string, excludeMembership?: string, atEventId?: string): Promise<{
        [userId: string]: IStateEventWithRoomId[];
    }>;
    /**
     * Upgrades a room to a new protocol version
     * @param newVersion - The target version to upgrade to
     * @returns Promise which resolves: Object with key 'replacement_room'
     * @returns Rejects: with an error response.
     */
    upgradeRoom(roomId: string, newVersion: string): Promise<{
        replacement_room: string;
    }>;
    /**
     * Retrieve a state event.
     * @returns Promise which resolves: TODO
     * @returns Rejects: with an error response.
     */
    getStateEvent(roomId: string, eventType: string, stateKey: string): Promise<Record<string, any>>;
    /**
     * Send a state event into a room
     * @param roomId - ID of the room to send the event into
     * @param eventType - type of the state event to send
     * @param content - content of the event to send
     * @param stateKey - the stateKey to send into the room
     * @param opts - Options for the request function.
     * @returns Promise which resolves: TODO
     * @returns Rejects: with an error response.
     */
    sendStateEvent<K extends keyof StateEvents>(roomId: string, eventType: K, content: StateEvents[K], stateKey?: string, opts?: IRequestOpts): Promise<ISendEventResponse>;
    private encryptStateEventIfNeeded;
    /**
     * @returns Promise which resolves: TODO
     * @returns Rejects: with an error response.
     */
    roomInitialSync(roomId: string, limit: number): Promise<IRoomInitialSyncResponse>;
    /**
     * Set a marker to indicate the point in a room before which the user has read every
     * event. This can be retrieved from room account data (the event type is `m.fully_read`)
     * and displayed as a horizontal line in the timeline that is visually distinct to the
     * position of the user's own read receipt.
     * @param roomId - ID of the room that has been read
     * @param rmEventId - ID of the event that has been read
     * @param rrEventId - ID of the event tracked by the read receipt. This is here
     * for convenience because the RR and the RM are commonly updated at the same time as
     * each other. Optional.
     * @param rpEventId - rpEvent the m.read.private read receipt event for when we
     * don't want other users to see the read receipts. This is experimental. Optional.
     * @returns Promise which resolves: the empty object, `{}`.
     */
    setRoomReadMarkersHttpRequest(roomId: string, rmEventId: string, rrEventId?: string, rpEventId?: string): Promise<EmptyObject>;
    /**
     * @returns Promise which resolves: A list of the user's current rooms
     * @returns Rejects: with an error response.
     */
    getJoinedRooms(): Promise<IJoinedRoomsResponse>;
    /**
     * Retrieve membership info. for a room.
     * @param roomId - ID of the room to get membership for
     * @returns Promise which resolves: A list of currently joined users
     *                                 and their profile data.
     * @returns Rejects: with an error response.
     */
    getJoinedRoomMembers(roomId: string): Promise<IJoinedMembersResponse>;
    /**
     * @param params - Options for this request
     * @returns Promise which resolves: IPublicRoomsResponse
     * @returns Rejects: with an error response.
     */
    publicRooms({ server, limit, since, ...options }?: IRoomDirectoryOptions): Promise<IPublicRoomsResponse>;
    /**
     * Create an alias to room ID mapping.
     * @param alias - The room alias to create.
     * @param roomId - The room ID to link the alias to.
     * @returns Promise which resolves: an empty object `{}`
     * @returns Rejects: with an error response.
     */
    createAlias(alias: string, roomId: string): Promise<EmptyObject>;
    /**
     * Delete an alias to room ID mapping. This alias must be on your local server,
     * and you must have sufficient access to do this operation.
     * @param alias - The room alias to delete.
     * @returns Promise which resolves: an empty object `{}`.
     * @returns Rejects: with an error response.
     */
    deleteAlias(alias: string): Promise<EmptyObject>;
    /**
     * Gets the local aliases for the room. Note: this includes all local aliases, unlike the
     * curated list from the m.room.canonical_alias state event.
     * @param roomId - The room ID to get local aliases for.
     * @returns Promise which resolves: an object with an `aliases` property, containing an array of local aliases
     * @returns Rejects: with an error response.
     */
    getLocalAliases(roomId: string): Promise<{
        aliases: string[];
    }>;
    /**
     * Get room info for the given alias.
     * @param alias - The room alias to resolve.
     * @returns Promise which resolves: Object with room_id and servers.
     * @returns Rejects: with an error response.
     */
    getRoomIdForAlias(alias: string): Promise<{
        room_id: string;
        servers: string[];
    }>;
    /**
     * Get the visibility of a room in the current HS's room directory
     * @returns Promise which resolves: TODO
     * @returns Rejects: with an error response.
     */
    getRoomDirectoryVisibility(roomId: string): Promise<{
        visibility: Visibility;
    }>;
    /**
     * Set the visibility of a room in the current HS's room directory
     * @param visibility - "public" to make the room visible
     *                 in the public directory, or "private" to make
     *                 it invisible.
     * @returns Promise which resolves: to an empty object `{}`
     * @returns Rejects: with an error response.
     */
    setRoomDirectoryVisibility(roomId: string, visibility: Visibility): Promise<EmptyObject>;
    /**
     * Query the user directory with a term matching user IDs, display names and domains.
     * @param options
     * @param options.term - the term with which to search.
     * @param options.limit - the maximum number of results to return. The server will apply a limit if unspecified.
     * @returns Promise which resolves: an array of results.
     */
    searchUserDirectory({ term, limit }: {
        term: string;
        limit?: number;
    }): Promise<IUserDirectoryResponse>;
    /**
     * Upload a file to the media repository on the homeserver.
     *
     * @param file - The object to upload. On a browser, something that
     *   can be sent to XMLHttpRequest.send (typically a File).  Under node.js,
     *   a a Buffer, String or ReadStream.
     *
     * @param opts -  options object
     *
     * @returns Promise which resolves to response object, or rejects with an error (usually a MatrixError).
     */
    uploadContent(file: FileType, opts?: UploadOpts): Promise<UploadResponse>;
    /**
     * Cancel a file upload in progress
     * @param upload - The object returned from uploadContent
     * @returns true if canceled, otherwise false
     */
    cancelUpload(upload: Promise<UploadResponse>): boolean;
    /**
     * Get a list of all file uploads in progress
     * @returns Array of objects representing current uploads.
     * Currently in progress is element 0. Keys:
     *  - promise: The promise associated with the upload
     *  - loaded: Number of bytes uploaded
     *  - total: Total number of bytes to upload
     */
    getCurrentUploads(): Upload[];
    /**
     * @param info - The kind of info to retrieve (e.g. 'displayname',
     * 'avatar_url').
     * @returns Promise which resolves: TODO
     * @returns Rejects: with an error response.
     */
    getProfileInfo(userId: string, info?: string): Promise<{
        avatar_url?: string;
        displayname?: string;
    }>;
    /**
     * Determine if the server supports extended profiles, as described by MSC4133.
     *
     * @returns `true` if supported, otherwise `false`
     */
    doesServerSupportExtendedProfiles(): Promise<boolean>;
    /**
     * Get the prefix used for extended profile requests.
     *
     * @returns The prefix for use with `authedRequest`
     */
    private getExtendedProfileRequestPrefix;
    /**
     * Fetch a user's *extended* profile, which may include additional keys.
     *
     * @see https://github.com/tcpipuk/matrix-spec-proposals/blob/main/proposals/4133-extended-profiles.md
     * @param userId The user ID to fetch the profile of.
     * @returns A set of keys to property values.
     *
     * @throws An error if the server does not support MSC4133.
     * @throws A M_NOT_FOUND error if the profile could not be found.
     */
    getExtendedProfile(userId: string): Promise<Record<string, unknown>>;
    /**
     * Fetch a specific key from the user's *extended* profile.
     *
     * @see https://github.com/tcpipuk/matrix-spec-proposals/blob/main/proposals/4133-extended-profiles.md
     * @param userId The user ID to fetch the profile of.
     * @param key The key of the property to fetch.
     * @returns The property value.
     *
     * @throws An error if the server does not support MSC4133.
     * @throws A M_NOT_FOUND error if the key was not set OR the profile could not be found.
     */
    getExtendedProfileProperty(userId: string, key: string): Promise<unknown>;
    /**
     * Set a property on your *extended* profile.
     *
     * @see https://github.com/tcpipuk/matrix-spec-proposals/blob/main/proposals/4133-extended-profiles.md
     * @param key The key of the property to set.
     * @param value The value to set on the property.
     *
     * @throws An error if the server does not support MSC4133 OR the server disallows editing the user profile.
     */
    setExtendedProfileProperty(key: string, value: unknown): Promise<void>;
    /**
     * Delete a property on your *extended* profile.
     *
     * @see https://github.com/tcpipuk/matrix-spec-proposals/blob/main/proposals/4133-extended-profiles.md
     * @param key The key of the property to delete.
     *
     * @throws An error if the server does not support MSC4133 OR the server disallows editing the user profile.
     */
    deleteExtendedProfileProperty(key: string): Promise<void>;
    /**
     * Update multiple properties on your *extended* profile. This will
     * merge with any existing keys.
     *
     * @see https://github.com/tcpipuk/matrix-spec-proposals/blob/main/proposals/4133-extended-profiles.md
     * @param profile The profile object to merge with the existing profile.
     * @returns The newly merged profile.
     *
     * @throws An error if the server does not support MSC4133 OR the server disallows editing the user profile.
     */
    patchExtendedProfile(profile: Record<string, unknown>): Promise<Record<string, unknown>>;
    /**
     * Set multiple properties on your *extended* profile. This will completely
     * replace the existing profile, removing any unspecified keys.
     *
     * @see https://github.com/tcpipuk/matrix-spec-proposals/blob/main/proposals/4133-extended-profiles.md
     * @param profile The profile object to set.
     *
     * @throws An error if the server does not support MSC4133 OR the server disallows editing the user profile.
     */
    setExtendedProfile(profile: Record<string, unknown>): Promise<void>;
    /**
     * @returns Promise which resolves to a list of the user's threepids.
     * @returns Rejects: with an error response.
     */
    getThreePids(): Promise<{
        threepids: IThreepid[];
    }>;
    /**
     * Add a 3PID to your homeserver account. This API does not use an identity
     * server, as the homeserver is expected to handle 3PID ownership validation.
     *
     * @param data - A object with 3PID validation data from having called
     * `account/3pid/<medium>/requestToken` on the homeserver.
     * @returns Promise which resolves: to an empty object `{}`
     * @returns Rejects: with an error response.
     */
    addThreePidOnly(data: IAddThreePidOnlyBody): Promise<EmptyObject>;
    /**
     * Bind a 3PID for discovery onto an identity server via the homeserver. The
     * identity server handles 3PID ownership validation and the homeserver records
     * the new binding to track where all 3PIDs for the account are bound.
     *
     * @param data - A object with 3PID validation data from having called
     * `validate/<medium>/requestToken` on the identity server. It should also
     * contain `id_server` and `id_access_token` fields as well.
     * @returns Promise which resolves: to an empty object `{}`
     * @returns Rejects: with an error response.
     */
    bindThreePid(data: IBindThreePidBody): Promise<EmptyObject>;
    /**
     * Unbind a 3PID for discovery on an identity server via the homeserver. The
     * homeserver removes its record of the binding to keep an updated record of
     * where all 3PIDs for the account are bound.
     *
     * @param medium - The threepid medium (eg. 'email')
     * @param address - The threepid address (eg. 'bob\@example.com')
     *        this must be as returned by getThreePids.
     * @returns Promise which resolves: on success
     * @returns Rejects: with an error response.
     */
    unbindThreePid(medium: string, address: string): Promise<{
        id_server_unbind_result: IdServerUnbindResult;
    }>;
    /**
     * @param medium - The threepid medium (eg. 'email')
     * @param address - The threepid address (eg. 'bob\@example.com')
     *        this must be as returned by getThreePids.
     * @returns Promise which resolves: The server response on success
     *     (generally the empty JSON object)
     * @returns Rejects: with an error response.
     */
    deleteThreePid(medium: string, address: string): Promise<{
        id_server_unbind_result: IdServerUnbindResult;
    }>;
    /**
     * Make a request to change your password.
     * @param newPassword - The new desired password.
     * @param logoutDevices - Should all sessions be logged out after the password change. Defaults to true.
     * @returns Promise which resolves: to an empty object `{}`
     * @returns Rejects: with an error response.
     */
    setPassword(authDict: AuthDict, newPassword: string, logoutDevices?: boolean): Promise<EmptyObject>;
    /**
     * Gets all devices recorded for the logged-in user
     * @returns Promise which resolves: result object
     * @returns Rejects: with an error response.
     */
    getDevices(): Promise<{
        devices: IMyDevice[];
    }>;
    /**
     * Gets specific device details for the logged-in user
     * @param deviceId -  device to query
     * @returns Promise which resolves: result object
     * @returns Rejects: with an error response.
     */
    getDevice(deviceId: string): Promise<IMyDevice>;
    /**
     * Update the given device
     *
     * @param deviceId -  device to update
     * @param body -       body of request
     * @returns Promise which resolves: to an empty object `{}`
     * @returns Rejects: with an error response.
     */
    setDeviceDetails(deviceId: string, body: {
        display_name: string;
    }): Promise<EmptyObject>;
    /**
     * Delete the given device
     *
     * @param deviceId -  device to delete
     * @param auth - Optional. Auth data to supply for User-Interactive auth.
     * @returns Promise which resolves: result object
     * @returns Rejects: with an error response.
     */
    deleteDevice(deviceId: string, auth?: AuthDict): Promise<EmptyObject>;
    /**
     * Delete multiple device
     *
     * @param devices - IDs of the devices to delete
     * @param auth - Optional. Auth data to supply for User-Interactive auth.
     * @returns Promise which resolves: result object
     * @returns Rejects: with an error response.
     */
    deleteMultipleDevices(devices: string[], auth?: AuthDict): Promise<EmptyObject>;
    /**
     * Gets all pushers registered for the logged-in user
     *
     * @returns Promise which resolves: Array of objects representing pushers
     * @returns Rejects: with an error response.
     */
    getPushers(): Promise<{
        pushers: IPusher[];
    }>;
    /**
     * Adds a new pusher or updates an existing pusher
     *
     * @param pusher - Object representing a pusher
     * @returns Promise which resolves: Empty json object on success
     * @returns Rejects: with an error response.
     */
    setPusher(pusher: IPusherRequest): Promise<EmptyObject>;
    /**
     * Removes an existing pusher
     * @param pushKey - pushkey of pusher to remove
     * @param appId - app_id of pusher to remove
     * @returns Promise which resolves: Empty json object on success
     * @returns Rejects: with an error response.
     */
    removePusher(pushKey: string, appId: string): Promise<EmptyObject>;
    /**
     * Persists local notification settings
     * @returns Promise which resolves: an empty object
     * @returns Rejects: with an error response.
     */
    setLocalNotificationSettings(deviceId: string, notificationSettings: LocalNotificationSettings): Promise<EmptyObject>;
    /**
     * Get the push rules for the account from the server.
     * @returns Promise which resolves to the push rules.
     * @returns Rejects: with an error response.
     */
    getPushRules(): Promise<IPushRules>;
    /**
     * Update the push rules for the account. This should be called whenever
     * updated push rules are available.
     */
    setPushRules(rules: IPushRules): void;
    /**
     * @returns Promise which resolves: an empty object `{}`
     * @returns Rejects: with an error response.
     */
    addPushRule(scope: string, kind: PushRuleKind, ruleId: Exclude<string, RuleId>, body: Pick<IPushRule, "actions" | "conditions" | "pattern">): Promise<EmptyObject>;
    /**
     * @returns Promise which resolves: an empty object `{}`
     * @returns Rejects: with an error response.
     */
    deletePushRule(scope: string, kind: PushRuleKind, ruleId: Exclude<string, RuleId>): Promise<EmptyObject>;
    /**
     * Enable or disable a push notification rule.
     * @returns Promise which resolves: to an empty object `{}`
     * @returns Rejects: with an error response.
     */
    setPushRuleEnabled(scope: string, kind: PushRuleKind, ruleId: RuleId | string, enabled: boolean): Promise<EmptyObject>;
    /**
     * Set the actions for a push notification rule.
     * @returns Promise which resolves: to an empty object `{}`
     * @returns Rejects: with an error response.
     */
    setPushRuleActions(scope: string, kind: PushRuleKind, ruleId: RuleId | string, actions: PushRuleAction[]): Promise<EmptyObject>;
    /**
     * Perform a server-side search.
     * @param params
     * @param params.next_batch - the batch token to pass in the query string
     * @param params.body - the JSON object to pass to the request body.
     * @param abortSignal - optional signal used to cancel the http request.
     * @returns Promise which resolves to the search response object.
     * @returns Rejects: with an error response.
     */
    search({ body, next_batch: nextBatch }: {
        body: ISearchRequestBody;
        next_batch?: string;
    }, abortSignal?: AbortSignal): Promise<ISearchResponse>;
    /**
     * Upload keys
     *
     * @param content -  body of upload request
     *
     * @param opts - this method no longer takes any opts,
     *  used to take opts.device_id but this was not removed from the spec as a redundant parameter
     *
     * @returns Promise which resolves: result object. Rejects: with
     *     an error response ({@link MatrixError}).
     */
    uploadKeysRequest(content: IUploadKeysRequest, opts?: void): Promise<IKeysUploadResponse>;
    uploadKeySignatures(content: KeySignatures): Promise<IUploadKeySignaturesResponse>;
    /**
     * Download device keys
     *
     * @param userIds -  list of users to get keys for
     *
     * @param token - sync token to pass in the query request, to help
     *   the HS give the most recent results
     *
     * @returns Promise which resolves: result object. Rejects: with
     *     an error response ({@link MatrixError}).
     */
    downloadKeysForUsers(userIds: string[], { token }?: {
        token?: string;
    }): Promise<IDownloadKeyResult>;
    /**
     * Claim one-time keys
     *
     * @param devices -  a list of [userId, deviceId] pairs
     *
     * @param keyAlgorithm -  desired key type
     *
     * @param timeout - the time (in milliseconds) to wait for keys from remote
     *     servers
     *
     * @returns Promise which resolves: result object. Rejects: with
     *     an error response ({@link MatrixError}).
     */
    claimOneTimeKeys(devices: [string, string][], keyAlgorithm?: string, timeout?: number): Promise<IClaimOTKsResult>;
    /**
     * Ask the server for a list of users who have changed their device lists
     * between a pair of sync tokens
     *
     *
     * @returns Promise which resolves: result object. Rejects: with
     *     an error response ({@link MatrixError}).
     */
    getKeyChanges(oldToken: string, newToken: string): Promise<{
        changed: string[];
        left: string[];
    }>;
    uploadDeviceSigningKeys(auth?: AuthDict, keys?: CrossSigningKeys): Promise<EmptyObject>;
    /**
     * Register with an identity server using the OpenID token from the user's
     * Homeserver, which can be retrieved via
     * {@link MatrixClient#getOpenIdToken}.
     *
     * Note that the `/account/register` endpoint (as well as IS authentication in
     * general) was added as part of the v2 API version.
     *
     * @returns Promise which resolves: with object containing an Identity
     * Server access token.
     * @returns Rejects: with an error response.
     */
    registerWithIdentityServer(hsOpenIdToken: IOpenIDToken): Promise<{
        access_token: string;
        token: string;
    }>;
    /**
     * Requests an email verification token directly from an identity server.
     *
     * This API is used as part of binding an email for discovery on an identity
     * server. The validation data that results should be passed to the
     * `bindThreePid` method to complete the binding process.
     *
     * @param email - The email address to request a token for
     * @param clientSecret - A secret binary string generated by the client.
     *                 It is recommended this be around 16 ASCII characters.
     * @param sendAttempt - If an identity server sees a duplicate request
     *                 with the same sendAttempt, it will not send another email.
     *                 To request another email to be sent, use a larger value for
     *                 the sendAttempt param as was used in the previous request.
     * @param nextLink - Optional If specified, the client will be redirected
     *                 to this link after validation.
     * @param identityAccessToken - The `access_token` field of the identity
     * server `/account/register` response (see {@link registerWithIdentityServer}).
     *
     * @returns Promise which resolves: TODO
     * @returns Rejects: with an error response.
     * @throws Error if no identity server is set
     */
    requestEmailToken(email: string, clientSecret: string, sendAttempt: number, nextLink?: string, identityAccessToken?: string): Promise<IRequestTokenResponse>;
    /**
     * Requests a MSISDN verification token directly from an identity server.
     *
     * This API is used as part of binding a MSISDN for discovery on an identity
     * server. The validation data that results should be passed to the
     * `bindThreePid` method to complete the binding process.
     *
     * @param phoneCountry - The ISO 3166-1 alpha-2 code for the country in
     *                 which phoneNumber should be parsed relative to.
     * @param phoneNumber - The phone number, in national or international
     *                 format
     * @param clientSecret - A secret binary string generated by the client.
     *                 It is recommended this be around 16 ASCII characters.
     * @param sendAttempt - If an identity server sees a duplicate request
     *                 with the same sendAttempt, it will not send another SMS.
     *                 To request another SMS to be sent, use a larger value for
     *                 the sendAttempt param as was used in the previous request.
     * @param nextLink - Optional If specified, the client will be redirected
     *                 to this link after validation.
     * @param identityAccessToken - The `access_token` field of the Identity
     * Server `/account/register` response (see {@link registerWithIdentityServer}).
     *
     * @returns Promise which resolves to an object with a sid string
     * @returns Rejects: with an error response.
     * @throws Error if no identity server is set
     */
    requestMsisdnToken(phoneCountry: string, phoneNumber: string, clientSecret: string, sendAttempt: number, nextLink?: string, identityAccessToken?: string): Promise<IRequestMsisdnTokenResponse>;
    /**
     * Submits a MSISDN token to the identity server
     *
     * This is used when submitting the code sent by SMS to a phone number.
     * The identity server has an equivalent API for email but the js-sdk does
     * not expose this, since email is normally validated by the user clicking
     * a link rather than entering a code.
     *
     * @param sid - The sid given in the response to requestToken
     * @param clientSecret - A secret binary string generated by the client.
     *                 This must be the same value submitted in the requestToken call.
     * @param msisdnToken - The MSISDN token, as entered by the user.
     * @param identityAccessToken - The `access_token` field of the Identity
     * Server `/account/register` response (see {@link registerWithIdentityServer}).
     * Some legacy identity servers had no authentication here.
     *
     * @returns Promise which resolves: Object, containing success boolean.
     * @returns Rejects: with an error response.
     * @throws Error if No identity server is set
     */
    submitMsisdnToken(sid: string, clientSecret: string, msisdnToken: string, identityAccessToken: string | null): Promise<{
        success: boolean;
    }>;
    /**
     * Submits a MSISDN token to an arbitrary URL.
     *
     * This is used when submitting the code sent by SMS to a phone number in the
     * newer 3PID flow where the homeserver validates 3PID ownership (as part of
     * `requestAdd3pidMsisdnToken`). The homeserver response may include a
     * `submit_url` to specify where the token should be sent, and this helper can
     * be used to pass the token to this URL.
     *
     * @param url - The URL to submit the token to
     * @param sid - The sid given in the response to requestToken
     * @param clientSecret - A secret binary string generated by the client.
     *                 This must be the same value submitted in the requestToken call.
     * @param msisdnToken - The MSISDN token, as entered by the user.
     *
     * @returns Promise which resolves: Object, containing success boolean.
     * @returns Rejects: with an error response.
     */
    submitMsisdnTokenOtherUrl(url: string, sid: string, clientSecret: string, msisdnToken: string): Promise<{
        success: boolean;
    }>;
    /**
     * Gets the V2 hashing information from the identity server. Primarily useful for
     * lookups.
     * @param identityAccessToken - The access token for the identity server.
     * @returns The hashing information for the identity server.
     */
    getIdentityHashDetails(identityAccessToken: string): Promise<{
        /**
         * The algorithms the server supports. Must contain at least sha256.
         */
        algorithms: string[];
        /**
         * The pepper the client MUST use in hashing identifiers,
         * and MUST supply to the /lookup endpoint when performing lookups.
         */
        lookup_pepper: string;
    }>;
    /**
     * Performs a hashed lookup of addresses against the identity server. This is
     * only supported on identity servers which have at least the version 2 API.
     * @param addressPairs - An array of 2 element arrays.
     * The first element of each pair is the address, the second is the 3PID medium.
     * Eg: `["email@example.org", "email"]`
     * @param identityAccessToken - The access token for the identity server.
     * @returns A collection of address mappings to
     * found MXIDs. Results where no user could be found will not be listed.
     */
    identityHashedLookup(addressPairs: [string, string][], identityAccessToken: string): Promise<{
        address: string;
        mxid: string;
    }[]>;
    /**
     * Looks up the public Matrix ID mapping for a given 3rd party
     * identifier from the identity server
     *
     * @param medium - The medium of the threepid, eg. 'email'
     * @param address - The textual address of the threepid
     * @param identityAccessToken - The `access_token` field of the Identity
     * Server `/account/register` response (see {@link registerWithIdentityServer}).
     *
     * @returns Promise which resolves: A threepid mapping
     *                                 object or the empty object if no mapping
     *                                 exists
     * @returns Rejects: with an error response.
     */
    lookupThreePid(medium: string, address: string, identityAccessToken: string): Promise<{
        address: string;
        medium: string;
        mxid: string;
    } | EmptyObject>;
    /**
     * Looks up the public Matrix ID mappings for multiple 3PIDs.
     *
     * @param query - Array of arrays containing
     * [medium, address]
     * @param identityAccessToken - The `access_token` field of the Identity
     * Server `/account/register` response (see {@link registerWithIdentityServer}).
     *
     * @returns Promise which resolves: Lookup results from IS.
     * @returns Rejects: with an error response.
     */
    bulkLookupThreePids(query: [string, string][], identityAccessToken: string): Promise<{
        threepids: [medium: string, address: string, mxid: string][];
    }>;
    /**
     * Get account info from the identity server. This is useful as a neutral check
     * to verify that other APIs are likely to approve access by testing that the
     * token is valid, terms have been agreed, etc.
     *
     * @param identityAccessToken - The `access_token` field of the Identity
     * Server `/account/register` response (see {@link registerWithIdentityServer}).
     *
     * @returns Promise which resolves: an object with account info.
     * @returns Rejects: with an error response.
     */
    getIdentityAccount(identityAccessToken: string): Promise<{
        user_id: string;
    }>;
    /**
     * Send an event to a specific list of devices.
     * This is a low-level API that simply wraps the HTTP API
     * call to send to-device messages. We recommend using
     * queueToDevice() which is a higher level API.
     *
     * @param eventType -  type of event to send
     *    content to send. Map from user_id to device_id to content object.
     * @param txnId -     transaction id. One will be made up if not
     *    supplied.
     * @returns Promise which resolves: to an empty object `{}`
     */
    sendToDevice(eventType: string, contentMap: SendToDeviceContentMap, txnId?: string): Promise<EmptyObject>;
    /**
     * This will encrypt the payload for all devices in the list and will queue it.
     * The type of the sent to-device message will be `m.room.encrypted`.
     * @param eventType - The type of event to send
     * @param devices - The list of devices to send the event to.
     * @param payload - The payload to send. This will be encrypted.
     * @returns Promise which resolves once queued there is no error feedback when sending fails.
     */
    encryptAndSendToDevice(eventType: string, devices: {
        userId: string;
        deviceId: string;
    }[], payload: ToDevicePayload): Promise<void>;
    /**
     * Sends events directly to specific devices using Matrix's to-device
     * messaging system. The batch will be split up into appropriately sized
     * batches for sending and stored in the store so they can be retried
     * later if they fail to send. Retries will happen automatically.
     * @param batch - The to-device messages to send
     */
    queueToDevice(batch: ToDeviceBatch): Promise<void>;
    /**
     * Get the third party protocols that can be reached using
     * this HS
     * @returns Promise which resolves to the result object
     */
    getThirdpartyProtocols(): Promise<{
        [protocol: string]: IProtocol;
    }>;
    /**
     * Get information on how a specific place on a third party protocol
     * may be reached.
     * @param protocol - The protocol given in getThirdpartyProtocols()
     * @param params - Protocol-specific parameters, as given in the
     *                        response to getThirdpartyProtocols()
     * @returns Promise which resolves to the result object
     */
    getThirdpartyLocation(protocol: string, params: {
        searchFields?: string[];
    }): Promise<IThirdPartyLocation[]>;
    /**
     * Get information on how a specific user on a third party protocol
     * may be reached.
     * @param protocol - The protocol given in getThirdpartyProtocols()
     * @param params - Protocol-specific parameters, as given in the
     *                        response to getThirdpartyProtocols()
     * @returns Promise which resolves to the result object
     */
    getThirdpartyUser(protocol: string, params?: QueryDict): Promise<IThirdPartyUser[]>;
    getTerms(serviceType: SERVICE_TYPES, baseUrl: string): Promise<Terms>;
    agreeToTerms(serviceType: SERVICE_TYPES, baseUrl: string, accessToken: string, termsUrls: string[]): Promise<EmptyObject>;
    /**
     * Reports an event as inappropriate to the server, which may then notify the appropriate people.
     * @param roomId - The room in which the event being reported is located.
     * @param eventId - The event to report.
     * @param score - The score to rate this content as where -100 is most offensive and 0 is inoffensive.
     * @param reason - The reason the content is being reported. May be blank.
     * @returns Promise which resolves to an empty object if successful
     */
    reportEvent(roomId: string, eventId: string, score: number, reason: string): Promise<EmptyObject>;
    /**
     * Reports a room as inappropriate to the server, which may then notify the appropriate people.
     *
     * This API was introduced in Matrix v1.13.
     *
     * @param roomId - The room being reported.
     * @param reason - The reason the room is being reported. May be blank.
     * @returns Promise which resolves to an empty object if successful
     */
    reportRoom(roomId: string, reason: string): Promise<EmptyObject>;
    /**
     * Fetches or paginates a room hierarchy asmatrix-js-sdk/spec/unit/matrix-client.spec.ts defined by MSC2946.
     * Falls back gracefully to sourcing its data from `getSpaceSummary` if this API is not yet supported by the server.
     * @param roomId - The ID of the space-room to use as the root of the summary.
     * @param limit - The maximum number of rooms to return per page.
     * @param maxDepth - The maximum depth in the tree from the root room to return.
     * @param suggestedOnly - Whether to only return rooms with suggested=true.
     * @param fromToken - The opaque token to paginate a previous request.
     * @returns the response, with next_batch & rooms fields.
     */
    getRoomHierarchy(roomId: string, limit?: number, maxDepth?: number, suggestedOnly?: boolean, fromToken?: string): Promise<IRoomHierarchy>;
    /**
     * Creates a new file tree space with the given name. The client will pick
     * defaults for how it expects to be able to support the remaining API offered
     * by the returned class.
     *
     * Note that this is UNSTABLE and may have breaking changes without notice.
     * @param name - The name of the tree space.
     * @returns Promise which resolves to the created space.
     */
    unstableCreateFileTree(name: string): Promise<MSC3089TreeSpace>;
    /**
     * Gets a reference to a tree space, if the room ID given is a tree space. If the room
     * does not appear to be a tree space then null is returned.
     *
     * Note that this is UNSTABLE and may have breaking changes without notice.
     * @param roomId - The room ID to get a tree space reference for.
     * @returns The tree space, or null if not a tree space.
     */
    unstableGetFileTreeSpace(roomId: string): MSC3089TreeSpace | null;
    /**
     * Perform a single MSC3575 sliding sync request.
     * @param req - The request to make.
     * @param proxyBaseUrl - The base URL for the sliding sync proxy.
     * @param abortSignal - Optional signal to abort request mid-flight.
     * @returns The sliding sync response, or a standard error.
     * @throws on non 2xx status codes with an object with a field "httpStatus":number.
     */
    slidingSync(req: MSC3575SlidingSyncRequest, proxyBaseUrl?: string, abortSignal?: AbortSignal): Promise<MSC3575SlidingSyncResponse>;
    /**
     * A helper to determine thread support
     * @returns a boolean to determine if threads are enabled
     */
    supportsThreads(): boolean;
    /**
     * A helper to determine intentional mentions support
     * @returns a boolean to determine if intentional mentions are enabled on the server
     * @experimental
     */
    supportsIntentionalMentions(): boolean;
    /**
     * Fetches the summary of a room as defined by an initial version of MSC3266 and implemented in Synapse
     * Proposed at https://github.com/matrix-org/matrix-doc/pull/3266
     * @param roomIdOrAlias - The ID or alias of the room to get the summary of.
     * @param via - The list of servers which know about the room if only an ID was provided.
     */
    getRoomSummary(roomIdOrAlias: string, via?: string[]): Promise<RoomSummary>;
    /**
     * Processes a list of threaded events and adds them to their respective timelines
     * @param room - the room the adds the threaded events
     * @param threadedEvents - an array of the threaded events
     * @param toStartOfTimeline - the direction in which we want to add the events
     */
    processThreadEvents(room: Room, threadedEvents: MatrixEvent[], toStartOfTimeline: boolean): void;
    /**
     * Processes a list of thread roots and creates a thread model
     * @param room - the room to create the threads in
     * @param threadedEvents - an array of thread roots
     * @param toStartOfTimeline - the direction
     */
    processThreadRoots(room: Room, threadedEvents: MatrixEvent[], toStartOfTimeline: boolean): void;
    processBeaconEvents(room?: Room, events?: MatrixEvent[]): void;
    /**
     * Calls aggregation functions for event types that are aggregated
     * Polls and location beacons
     * @param room - room the events belong to
     * @param events - timeline events to be processed
     * @returns
     */
    processAggregatedTimelineEvents(room?: Room, events?: MatrixEvent[]): void;
    /**
     * Fetches information about the user for the configured access token.
     */
    whoami(): Promise<IWhoamiResponse>;
    /**
     * Find the event_id closest to the given timestamp in the given direction.
     * @returns Resolves: A promise of an object containing the event_id and
     *    origin_server_ts of the closest event to the timestamp in the given direction
     * @returns Rejects: when the request fails (module:http-api.MatrixError)
     */
    timestampToEvent(roomId: string, timestamp: number, dir: Direction): Promise<TimestampToEventResponse>;
    /**
     * Discover and validate delegated auth configuration
     * - delegated auth issuer openid-configuration is reachable
     * - delegated auth issuer openid-configuration is configured correctly for us
     * Fetches /auth_metadata falling back to legacy implementation using /auth_issuer followed by
     * https://oidc-issuer.example.com/.well-known/openid-configuration and other files linked therein.
     * When successful, validated metadata is returned
     * @returns validated authentication metadata and optionally signing keys
     * @throws when delegated auth config is invalid or unreachable
     * @experimental - part of MSC2965
     */
    getAuthMetadata(): Promise<OidcClientConfig>;
}
/**
 * recalculates an accurate notifications count on event decryption.
 * Servers do not have enough knowledge about encrypted events to calculate an
 * accurate notification_count
 */
export declare function fixNotificationCountOnDecryption(cli: MatrixClient, event: MatrixEvent): void;
/**
 * Given an event, figure out the thread ID we should use for it in a receipt.
 *
 * This will either be "main", or event.threadRootId. For the thread root, or
 * e.g. reactions to the thread root, this will be main. For events inside the
 * thread, or e.g. reactions to them, this will be event.threadRootId.
 *
 * (Exported for test.)
 */
export declare function threadIdForReceipt(event: MatrixEvent): string;
/**
 * a) True for non-threaded messages, thread roots and non-thread relations to thread roots.
 * b) False for messages with thread relations to the thread root.
 * c) False for messages with any kind of relation to a message from case b.
 *
 * Note: true for redactions of messages that are in threads. Redacted messages
 * are not really in threads (because their relations are gone), so if they look
 * like they are in threads, that is a sign of a bug elsewhere. (At time of
 * writing, this bug definitely exists - messages are not moved to another
 * thread when they are redacted.)
 *
 * @returns true if this event is considered to be in the main timeline as far
 *               as receipts are concerned.
 */
export declare function inMainTimelineForReceipt(event: MatrixEvent): boolean;
export {};
//# sourceMappingURL=client.d.ts.map