"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CredentialProvider = void 0;
const child_process_1 = require("child_process");
const common_1 = require("../../common");
class CredentialProvider {
    async getAuthConfig(registry, dockerConfig) {
        const credentialProviderName = this.getCredentialProviderName(registry, dockerConfig);
        if (!credentialProviderName) {
            return undefined;
        }
        const programName = `docker-credential-${credentialProviderName}`;
        common_1.log.debug(`Executing Docker credential provider "${programName}"`);
        const response = await this.runCredentialProvider(registry, programName);
        return {
            username: response.Username,
            password: response.Secret,
            registryAddress: response.ServerURL ?? registry,
        };
    }
    runCredentialProvider(registry, providerName) {
        return new Promise((resolve, reject) => {
            const sink = (0, child_process_1.spawn)(providerName, ["get"]);
            const chunks = [];
            sink.stdout.on("data", (chunk) => chunks.push(chunk));
            sink.on("close", (code) => {
                if (code !== 0) {
                    common_1.log.error(`An error occurred getting a credential: ${code}`);
                    return reject(new Error("An error occurred getting a credential"));
                }
                const response = chunks.join("");
                try {
                    const parsedResponse = JSON.parse(response);
                    return resolve(parsedResponse);
                }
                catch (e) {
                    common_1.log.error(`Unexpected response from Docker credential provider GET command: "${response}"`);
                    return reject(new Error("Unexpected response from Docker credential provider GET command"));
                }
            });
            sink.stdin.write(`${registry}\n`);
            sink.stdin.end();
        });
    }
}
exports.CredentialProvider = CredentialProvider;
//# sourceMappingURL=credential-provider.js.map