'use strict';

var ImportChecker = require('../ImportChecker');

module.exports = {
    meta: {
        docs: {
            description: 'Forbids importing from given files.'
        }
    },
    create: function create(context) {
        var imports = context.options.map(function (opt) {
            return new ImportChecker(opt);
        });
        return {
            ImportDeclaration: function ImportDeclaration(node) {
                var importPath = node.source.value;
                var importedModules = node.specifiers.length ? node.specifiers.filter(function (childNode) {
                    return !!childNode.imported;
                }).map(function (childNode) {
                    return childNode.imported.name;
                }) : [];
                var importedModulesSet = new Set(importedModules);
                imports.filter(function (imp) {
                    return imp.check(importPath, importedModulesSet);
                }).forEach(function (imp) {
                    context.report({ node: node, message: imp.getErrMessage() });
                });
            },
            CallExpression: function CallExpression(node) {
                if (node.callee.name !== 'require' || !node.arguments.length) {
                    return;
                }
                var requireArg = node.arguments[0];
                if (requireArg.type !== 'Literal') {
                    return;
                }
                imports.filter(function (imp) {
                    return imp.check(requireArg.value);
                }).forEach(function (imp) {
                    context.report({ node: node, message: imp.getErrMessage() });
                });
            },
            MemberExpression: function MemberExpression(node) {
                if (!node.object.callee) {
                    return;
                }
                if ((node.object.callee.name !== 'require' || !node.object.arguments.length) && !node.property) {
                    return;
                }
                var requireArg = node.object.arguments[0];
                if (!requireArg || requireArg.type !== 'Literal') {
                    return;
                }
                var propertyNameSet = new Set([node.property.name]);
                imports.filter(function (imp) {
                    return imp.check(requireArg.value, propertyNameSet);
                }).forEach(function (imp) {
                    context.report({ node: node, message: imp.getErrMessage() });
                });
            },
            VariableDeclaration: function VariableDeclaration(node) {
                if (!(node.declarations && node.declarations.length)) return;

                var childNode = node.declarations[0];
                if (!(childNode.id && childNode.id.type === 'ObjectPattern' && childNode.init)) return;

                var requireArg = childNode.init.arguments && childNode.init.arguments[0];
                if (!requireArg || requireArg.type !== 'Literal' || childNode.init.callee.name !== 'require') {
                    return;
                }

                var namedImports = childNode.id.properties.filter(function (module) {
                    return !!module.value.name;
                }).map(function (module) {
                    return module.value.name;
                });

                var namedImportsSet = new Set(namedImports);
                imports.filter(function (imp) {
                    return imp.check(requireArg.value, namedImportsSet);
                }).forEach(function (imp) {
                    context.report({ node: node, message: imp.getErrMessage() });
                });
            }
        };
    }
};