import { useId, useFloating, offset, flip, shift, arrow, autoUpdate, useDelayGroup, useHover, useFocus, useDismiss, useRole, useInteractions } from "@floating-ui/react";
import { useRef, useState, useEffect, useMemo } from "react";
import { hoverDelay } from "./TooltipProvider.js";
function useTooltip({
  open: controlledOpen,
  disabled = false,
  onOpenChange,
  placement = "bottom",
  isTriggerInteractive,
  caption,
  "aria-atomic": ariaAtomic,
  "aria-live": ariaLive,
  ...props
}) {
  const labelId = useId();
  const captionId = useId();
  const arrowRef = useRef(null);
  const [uncontrolledOpen, setUncontrolledOpen] = useState(false);
  const open = disabled ? false : controlledOpen ?? uncontrolledOpen;
  const setOpen = (open2, event, reason) => {
    onOpenChange?.(open2, event, reason);
    if (controlledOpen === void 0) setUncontrolledOpen(open2);
  };
  const data = useFloating({
    placement,
    open,
    onOpenChange: setOpen,
    whileElementsMounted: autoUpdate,
    middleware: [
      // arrow height 6px
      offset(6),
      flip({
        crossAxis: placement.includes("-"),
        fallbackAxisSideDirection: "start",
        padding: 5
      }),
      shift({ padding: 5 }),
      // add the little arrow along with the floating content
      arrow({
        element: arrowRef
      })
    ]
  });
  const context = data.context;
  const { delay, initialDelay } = useDelayGroup(context);
  if (initialDelay !== hoverDelay)
    throw new Error("Tooltips must be wrapped in a global <TooltipProvider>");
  const hover = useHover(context, {
    move: false,
    enabled: controlledOpen === void 0,
    // Show tooltip after a delay when trigger is interactive
    delay: isTriggerInteractive ? delay : {},
    mouseOnly: true
  });
  const focus = useFocus(context, {
    enabled: controlledOpen === void 0
  });
  const pressTimer = useRef(void 0);
  useEffect(() => () => window.clearTimeout(pressTimer.current), []);
  const press = useMemo(() => {
    const onTouchEnd = () => {
      if (pressTimer.current === void 0)
        pressTimer.current = window.setTimeout(() => {
          setOpen(false);
          pressTimer.current = void 0;
        }, 1500);
      else window.clearTimeout(pressTimer.current);
    };
    return {
      // Set these props on the anchor element
      reference: {
        onTouchStart: () => {
          if (pressTimer.current !== void 0)
            window.clearTimeout(pressTimer.current);
          pressTimer.current = window.setTimeout(() => {
            setOpen(true);
            pressTimer.current = void 0;
          }, 500);
        },
        onTouchEnd,
        onTouchCancel: onTouchEnd
      }
    };
  }, []);
  const dismiss = useDismiss(context);
  const purpose = "label" in props ? "label" : "description";
  const role = useRole(context, {
    enabled: purpose === "description",
    role: "tooltip"
  });
  const label = useMemo(
    () => purpose === "label" ? {
      // Set these props on the anchor element
      reference: {
        "aria-labelledby": labelId,
        "aria-describedby": caption ? captionId : void 0
      }
    } : {},
    [purpose, labelId, captionId]
  );
  const interactions = useInteractions([
    hover,
    focus,
    press,
    dismiss,
    role,
    label
  ]);
  return useMemo(
    () => ({
      labelId,
      captionId: caption ? captionId : void 0,
      caption,
      purpose,
      open,
      setOpen,
      tooltipProps: {
        "aria-atomic": ariaAtomic,
        "aria-live": ariaLive
      },
      ...interactions,
      ...data,
      arrowRef
    }),
    [
      labelId,
      captionId,
      caption,
      role,
      open,
      setOpen,
      interactions,
      data,
      arrowRef
    ]
  );
}
export {
  useTooltip
};
//# sourceMappingURL=useTooltip.js.map
