"use strict";
Object.defineProperty(exports, Symbol.toStringTag, { value: "Module" });
const react = require("@floating-ui/react");
const React = require("react");
const TooltipProvider = require("./TooltipProvider.cjs");
function useTooltip({
  open: controlledOpen,
  disabled = false,
  onOpenChange,
  placement = "bottom",
  isTriggerInteractive,
  caption,
  "aria-atomic": ariaAtomic,
  "aria-live": ariaLive,
  ...props
}) {
  const labelId = react.useId();
  const captionId = react.useId();
  const arrowRef = React.useRef(null);
  const [uncontrolledOpen, setUncontrolledOpen] = React.useState(false);
  const open = disabled ? false : controlledOpen ?? uncontrolledOpen;
  const setOpen = (open2, event, reason) => {
    onOpenChange?.(open2, event, reason);
    if (controlledOpen === void 0) setUncontrolledOpen(open2);
  };
  const data = react.useFloating({
    placement,
    open,
    onOpenChange: setOpen,
    whileElementsMounted: react.autoUpdate,
    middleware: [
      // arrow height 6px
      react.offset(6),
      react.flip({
        crossAxis: placement.includes("-"),
        fallbackAxisSideDirection: "start",
        padding: 5
      }),
      react.shift({ padding: 5 }),
      // add the little arrow along with the floating content
      react.arrow({
        element: arrowRef
      })
    ]
  });
  const context = data.context;
  const { delay, initialDelay } = react.useDelayGroup(context);
  if (initialDelay !== TooltipProvider.hoverDelay)
    throw new Error("Tooltips must be wrapped in a global <TooltipProvider>");
  const hover = react.useHover(context, {
    move: false,
    enabled: controlledOpen === void 0,
    // Show tooltip after a delay when trigger is interactive
    delay: isTriggerInteractive ? delay : {},
    mouseOnly: true
  });
  const focus = react.useFocus(context, {
    enabled: controlledOpen === void 0
  });
  const pressTimer = React.useRef(void 0);
  React.useEffect(() => () => window.clearTimeout(pressTimer.current), []);
  const press = React.useMemo(() => {
    const onTouchEnd = () => {
      if (pressTimer.current === void 0)
        pressTimer.current = window.setTimeout(() => {
          setOpen(false);
          pressTimer.current = void 0;
        }, 1500);
      else window.clearTimeout(pressTimer.current);
    };
    return {
      // Set these props on the anchor element
      reference: {
        onTouchStart: () => {
          if (pressTimer.current !== void 0)
            window.clearTimeout(pressTimer.current);
          pressTimer.current = window.setTimeout(() => {
            setOpen(true);
            pressTimer.current = void 0;
          }, 500);
        },
        onTouchEnd,
        onTouchCancel: onTouchEnd
      }
    };
  }, []);
  const dismiss = react.useDismiss(context);
  const purpose = "label" in props ? "label" : "description";
  const role = react.useRole(context, {
    enabled: purpose === "description",
    role: "tooltip"
  });
  const label = React.useMemo(
    () => purpose === "label" ? {
      // Set these props on the anchor element
      reference: {
        "aria-labelledby": labelId,
        "aria-describedby": caption ? captionId : void 0
      }
    } : {},
    [purpose, labelId, captionId]
  );
  const interactions = react.useInteractions([
    hover,
    focus,
    press,
    dismiss,
    role,
    label
  ]);
  return React.useMemo(
    () => ({
      labelId,
      captionId: caption ? captionId : void 0,
      caption,
      purpose,
      open,
      setOpen,
      tooltipProps: {
        "aria-atomic": ariaAtomic,
        "aria-live": ariaLive
      },
      ...interactions,
      ...data,
      arrowRef
    }),
    [
      labelId,
      captionId,
      caption,
      role,
      open,
      setOpen,
      interactions,
      data,
      arrowRef
    ]
  );
}
exports.useTooltip = useTooltip;
//# sourceMappingURL=useTooltip.cjs.map
