import { jsxs, jsx } from "react/jsx-runtime";
import { TooltipContext, useTooltipContext } from "./TooltipContext.js";
import { useMergeRefs, FloatingPortal, FloatingArrow } from "@floating-ui/react";
import { useMemo, isValidElement, cloneElement } from "react";
import classNames from "classnames";
import styles from "./Tooltip.module.css.js";
import { useTooltip } from "./useTooltip.js";
const hasLabel = (props) => "label" in props && !!props.label;
function Tooltip({
  children,
  isTriggerInteractive = true,
  nonInteractiveTriggerTabIndex = 0,
  ...props
}) {
  const context = useTooltip({ isTriggerInteractive, ...props });
  return /* @__PURE__ */ jsxs(TooltipContext.Provider, { value: context, children: [
    /* @__PURE__ */ jsx(
      TooltipAnchor,
      {
        isTriggerInteractive,
        nonInteractiveTriggerTabIndex,
        children
      }
    ),
    /* @__PURE__ */ jsxs(TooltipContent, { children: [
      /* @__PURE__ */ jsx("span", { id: context.labelId, children: hasLabel(props) ? props.label : props.description }),
      /* @__PURE__ */ jsx(Caption, {})
    ] })
  ] });
}
function Caption() {
  const { caption, captionId } = useTooltipContext();
  if (!caption) return null;
  const isCaptionString = typeof caption === "string";
  const Container = isCaptionString ? "span" : "div";
  return /* @__PURE__ */ jsx(
    Container,
    {
      id: captionId,
      className: classNames(styles.caption, "cpd-theme-dark"),
      children: caption
    }
  );
}
function TooltipContent({
  children
}) {
  const {
    context: floatingContext,
    open,
    arrowRef,
    purpose,
    ...rest
  } = useTooltipContext();
  if (!open && purpose !== "label") return null;
  return /* @__PURE__ */ jsx(FloatingPortal, { children: /* @__PURE__ */ jsxs(
    "div",
    {
      ref: rest.refs.setFloating,
      style: rest.floatingStyles,
      ...rest.tooltipProps,
      ...rest.getFloatingProps(),
      className: classNames(styles.tooltip, {
        [styles.invisible]: purpose === "label" && !open
      }),
      children: [
        /* @__PURE__ */ jsx(
          FloatingArrow,
          {
            ref: arrowRef,
            context: floatingContext,
            width: 10,
            height: 6,
            className: styles.arrow
          }
        ),
        children
      ]
    }
  ) });
}
const TooltipAnchor = ({
  children,
  isTriggerInteractive,
  nonInteractiveTriggerTabIndex
}) => {
  const context = useTooltipContext();
  const childrenRef = children?.ref;
  const ref = useMergeRefs([context.refs.setReference, childrenRef]);
  const element = useMemo(() => {
    if (!isValidElement(children)) return;
    if (isTriggerInteractive) {
      const props = context.getReferenceProps({
        // To support React 18, we need to explicitly pass the children's props. See  https://github.com/element-hq/compound/issues/333
        // In React 19, this is not necessary. `getReferenceProps` is able to get the props directly from the ref.
        ...typeof children.props === "object" ? children.props : {},
        ref
      });
      return cloneElement(children, props);
    } else {
      const props = context.getReferenceProps({
        ref,
        tabIndex: nonInteractiveTriggerTabIndex
      });
      const {
        "aria-labelledby": labelId,
        "aria-describedby": descriptionId,
        ...spanProps
      } = props;
      return /* @__PURE__ */ jsx("span", { tabIndex: nonInteractiveTriggerTabIndex, ...spanProps, children: cloneElement(children, {
        "aria-labelledby": labelId,
        "aria-describedby": descriptionId
      }) });
    }
  }, [context, ref, children]);
  if (!element) {
    throw new Error("Tooltip anchor must be a single valid React element");
  }
  return element;
};
export {
  Tooltip
};
//# sourceMappingURL=Tooltip.js.map
