import { jsxs, jsx } from "react/jsx-runtime";
import ChevronDown from "@vector-im/compound-design-tokens/assets/web/icons/chevron-down";
import CheckIcon from "@vector-im/compound-design-tokens/assets/web/icons/check";
import ErrorIcon from "@vector-im/compound-design-tokens/assets/web/icons/error-solid";
import { forwardRef, useState, useMemo, useCallback, useRef, useEffect, memo } from "react";
import classNames from "classnames";
import styles from "./Dropdown.module.css.js";
import { useId } from "@floating-ui/react";
const Dropdown = forwardRef(
  function Dropdown2({
    className,
    label,
    placeholder,
    helpLabel,
    onValueChange,
    error,
    value: controlledValue,
    defaultValue,
    values,
    ...props
  }, ref) {
    const [uncontrolledValue, setUncontrolledValue] = useState(defaultValue);
    const value = controlledValue ?? uncontrolledValue;
    const text = useMemo(
      () => value === void 0 ? placeholder : values.find(([v]) => v === value)?.[1] ?? placeholder,
      [value, values, placeholder]
    );
    const setValue = useCallback(
      (value2) => {
        setUncontrolledValue(value2);
        onValueChange?.(value2);
      },
      [setUncontrolledValue, onValueChange]
    );
    const [open, setOpen, dropdownRef] = useOpen();
    const { listRef, onComboboxKeyDown, onOptionKeyDown } = useKeyboardShortcut(
      open,
      setOpen,
      setValue
    );
    const buttonRef = useRef(null);
    useEffect(() => {
      if (value !== void 0) buttonRef.current?.focus();
    }, [value]);
    const hasPlaceholder = text === placeholder;
    const buttonClasses = classNames({
      [styles.placeholder]: hasPlaceholder
    });
    const borderClasses = classNames(styles.border, {
      [styles.open]: open
    });
    const contentClasses = classNames(styles.content, {
      [styles.open]: open
    });
    const labelId = useId();
    const contentId = useId();
    return /* @__PURE__ */ jsxs(
      "div",
      {
        ref: dropdownRef,
        className: classNames(className, styles.container),
        "aria-invalid": Boolean(error),
        children: [
          /* @__PURE__ */ jsx("label", { id: labelId, children: label }),
          /* @__PURE__ */ jsxs(
            "button",
            {
              className: buttonClasses,
              role: "combobox",
              "aria-haspopup": "listbox",
              "aria-labelledby": labelId,
              "aria-controls": contentId,
              "aria-expanded": open,
              ref: (element) => {
                buttonRef.current = element;
                if (typeof ref === "function") {
                  ref(element);
                } else if (ref) {
                  ref.current = element;
                }
              },
              onClick: () => setOpen((_open) => !_open),
              onKeyDown: onComboboxKeyDown,
              ...props,
              children: [
                text,
                /* @__PURE__ */ jsx(ChevronDown, { width: "24", height: "24" })
              ]
            }
          ),
          /* @__PURE__ */ jsx("div", { className: borderClasses }),
          /* @__PURE__ */ jsx("div", { className: contentClasses, children: /* @__PURE__ */ jsx(
            "ul",
            {
              ref: listRef,
              id: contentId,
              role: "listbox",
              className: styles.content,
              children: values.map(([v, text2]) => /* @__PURE__ */ jsx(
                DropdownItem,
                {
                  isDisplayed: open,
                  isSelected: value === v,
                  onClick: () => {
                    setOpen(false);
                    setValue(v);
                  },
                  onKeyDown: (e) => onOptionKeyDown(e, v),
                  children: text2
                },
                v
              ))
            }
          ) }),
          !error && helpLabel && /* @__PURE__ */ jsx("span", { className: styles.help, children: helpLabel }),
          error && /* @__PURE__ */ jsxs("span", { className: styles.error, children: [
            /* @__PURE__ */ jsx(ErrorIcon, { width: "20", height: "20" }),
            error
          ] })
        ]
      }
    );
  }
);
const DropdownItem = memo(function DropdownItem2({
  children,
  isSelected,
  isDisplayed,
  ...props
}) {
  const ref = useRef(null);
  useEffect(() => {
    if (isSelected && isDisplayed) {
      ref.current?.focus();
    }
  }, [isSelected, isDisplayed]);
  return /* @__PURE__ */ jsxs(
    "li",
    {
      tabIndex: 0,
      role: "option",
      ref,
      "aria-selected": isSelected,
      ...props,
      children: [
        children,
        " ",
        isSelected && /* @__PURE__ */ jsx(CheckIcon, { width: "20", height: "20" })
      ]
    }
  );
});
function useOpen() {
  const [open, setOpen] = useState(false);
  const ref = useRef(null);
  useEffect(() => {
    const closeIfOutside = (e) => {
      if (ref.current && !ref.current.contains(e.target)) {
        setOpen(false);
      }
    };
    document.addEventListener("click", closeIfOutside);
    return () => document.removeEventListener("click", closeIfOutside);
  }, [setOpen]);
  return [open, setOpen, ref];
}
function useKeyboardShortcut(open, setOpen, setValue) {
  const listRef = useRef(null);
  const onComboboxKeyDown = useCallback(
    ({ key }) => {
      switch (key) {
        // Enter and Space already managed because it's a button
        case "Escape":
          setOpen(false);
          break;
        case "ArrowDown":
          setOpen(true);
          if (open) {
            listRef.current?.firstElementChild?.focus();
          }
          break;
        case "ArrowUp":
          setOpen(true);
          break;
        case "Home": {
          setOpen(true);
          Promise.resolve().then(() => {
            listRef.current?.firstElementChild?.focus();
          });
          break;
        }
        case "End": {
          setOpen(true);
          Promise.resolve().then(() => {
            listRef.current?.lastElementChild?.focus();
          });
          break;
        }
      }
    },
    [listRef, open, setOpen]
  );
  const onOptionKeyDown = useCallback(
    (evt, value) => {
      const { key, altKey } = evt;
      evt.stopPropagation();
      evt.preventDefault();
      switch (key) {
        case "Enter":
        case " ": {
          setValue(value);
          setOpen(false);
          break;
        }
        case "Tab":
        case "Escape":
          setOpen(false);
          break;
        case "ArrowDown": {
          const currentFocus = document.activeElement;
          if (listRef.current?.contains(currentFocus) && currentFocus) {
            currentFocus.nextElementSibling?.focus();
          }
          break;
        }
        case "ArrowUp": {
          if (altKey) {
            setValue(value);
            setOpen(false);
          } else {
            const currentFocus = document.activeElement;
            if (listRef.current?.contains(currentFocus) && currentFocus) {
              currentFocus.previousElementSibling?.focus();
            }
          }
          break;
        }
        case "Home": {
          listRef.current?.firstElementChild?.focus();
          break;
        }
        case "End": {
          listRef.current?.lastElementChild?.focus();
          break;
        }
      }
    },
    [listRef, setValue, setOpen]
  );
  return { listRef, onComboboxKeyDown, onOptionKeyDown };
}
export {
  Dropdown
};
//# sourceMappingURL=Dropdown.js.map
