"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PM = void 0;
exports.getPackageManagerCommand = getPackageManagerCommand;
exports.detectPackageManagerByEnv = detectPackageManagerByEnv;
exports.detectPackageManagerByLockfile = detectPackageManagerByLockfile;
const path = require("path");
const fs = require("fs");
const which = require("which");
var PM;
(function (PM) {
    PM["NPM"] = "npm";
    PM["YARN"] = "yarn";
    PM["PNPM"] = "pnpm";
    PM["YARN_BERRY"] = "yarn-berry";
})(PM || (exports.PM = PM = {}));
// Cache for resolved paths
const pmPathCache = {
    [PM.NPM]: undefined,
    [PM.YARN]: undefined,
    [PM.PNPM]: undefined,
    [PM.YARN_BERRY]: undefined,
};
function resolveCommand(pm) {
    const fallback = pm === PM.YARN_BERRY ? "yarn" : pm;
    if (process.platform !== "win32") {
        return fallback;
    }
    try {
        return which.sync(fallback);
    }
    catch {
        // If `which` fails (not found), still return the fallback string
        return fallback;
    }
}
function getPackageManagerCommand(pm) {
    if (pmPathCache[pm] !== undefined) {
        return pmPathCache[pm];
    }
    const resolved = resolveCommand(pm);
    pmPathCache[pm] = resolved;
    return resolved;
}
function detectPackageManagerByEnv(pm) {
    var _a, _b, _c;
    const ua = (_a = process.env.npm_config_user_agent) !== null && _a !== void 0 ? _a : "";
    const execPath = (_c = (_b = process.env.npm_execpath) === null || _b === void 0 ? void 0 : _b.toLowerCase()) !== null && _c !== void 0 ? _c : "";
    const yarnVersion = process.env.YARN_VERSION;
    const isBerry = (yarnVersion === null || yarnVersion === void 0 ? void 0 : yarnVersion.startsWith("2.")) || (yarnVersion === null || yarnVersion === void 0 ? void 0 : yarnVersion.startsWith("3."));
    switch (pm) {
        case "pnpm":
            return ua.includes("pnpm") || execPath.includes("pnpm") || process.env.PNPM_HOME ? PM.PNPM : null;
        case "yarn":
            if (ua.includes("yarn") || execPath.includes("yarn") || process.env.YARN_REGISTRY) {
                return isBerry || ua.includes("yarn/2") || ua.includes("yarn/3") ? PM.YARN_BERRY : PM.YARN;
            }
            return null;
        case "npm":
            return ua.includes("npm") || execPath.includes("npm") || process.env.npm_package_json ? PM.NPM : null;
        default:
            return null;
    }
}
function detectPackageManagerByLockfile(cwd) {
    const has = (file) => fs.existsSync(path.join(cwd, file));
    const yarn = has("yarn.lock");
    const pnpm = has("pnpm-lock.yaml");
    const npm = has("package-lock.json");
    const detected = [];
    if (yarn)
        detected.push(PM.YARN);
    if (pnpm)
        detected.push(PM.PNPM);
    if (npm)
        detected.push(PM.NPM);
    if (detected.length === 1) {
        if (detected[0] === PM.YARN) {
            return detectPackageManagerByEnv("yarn") === PM.YARN_BERRY ? PM.YARN_BERRY : PM.YARN;
        }
        return detected[0];
    }
    return null;
}
//# sourceMappingURL=packageManager.js.map