"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodeModulesCollector = void 0;
const hoist_1 = require("./hoist");
const path = require("path");
const fs = require("fs-extra");
const builder_util_1 = require("builder-util");
const packageManager_1 = require("./packageManager");
const child_process_1 = require("child_process");
const util_1 = require("util");
const fs_1 = require("fs");
const execAsync = (0, util_1.promisify)(child_process_1.exec);
class NodeModulesCollector {
    constructor(rootDir, tempDirManager) {
        this.rootDir = rootDir;
        this.tempDirManager = tempDirManager;
        this.nodeModules = [];
        this.allDependencies = new Map();
        this.productionGraph = {};
    }
    async getNodeModules() {
        const tree = await this.getDependenciesTree();
        this.collectAllDependencies(tree); // Parse from the root, as npm list can host and deduplicate across projects in the workspace
        const realTree = this.getTreeFromWorkspaces(tree);
        this.extractProductionDependencyGraph(realTree, "." /*root project name*/);
        const hoisterResult = (0, hoist_1.hoist)(this.transToHoisterTree(this.productionGraph), { check: true });
        this._getNodeModules(hoisterResult.dependencies, this.nodeModules);
        return this.nodeModules;
    }
    async getDependenciesTree() {
        const command = (0, packageManager_1.getPackageManagerCommand)(this.installOptions.manager);
        const args = this.getArgs();
        const tempOutputFile = await this.tempDirManager.getTempFile({
            prefix: path.basename(command, path.extname(command)),
            suffix: "output.json",
        });
        return (0, builder_util_1.retry)(async () => {
            await this.streamCollectorCommandToJsonFile(command, args, this.rootDir, tempOutputFile);
            const dependencies = await fs.readFile(tempOutputFile, { encoding: "utf8" });
            try {
                return this.parseDependenciesTree(dependencies);
            }
            catch (error) {
                builder_util_1.log.debug({ message: error.message || error.stack, shellOutput: dependencies }, "error parsing dependencies tree");
                throw new Error(`Failed to parse dependencies tree: ${error.message || error.stack}. Use DEBUG=electron-builder env var to see the dependency query output.`);
            }
        }, {
            retries: 2,
            interval: 2000,
            backoff: 2000,
            shouldRetry: async (error) => {
                var _a;
                if (!(await (0, builder_util_1.exists)(tempOutputFile))) {
                    builder_util_1.log.error({ error: error.message || error.stack, tempOutputFile }, "error getting dependencies tree, unable to find output; retrying");
                    return true;
                }
                const dependencies = await fs.readFile(tempOutputFile, { encoding: "utf8" });
                if (dependencies.trim().length === 0 || ((_a = error.message) === null || _a === void 0 ? void 0 : _a.includes("Unexpected end of JSON input"))) {
                    // If the output file is empty or contains invalid JSON, we retry
                    // This can happen if the command fails or if the output is not as expected
                    builder_util_1.log.error({ error: error.message || error.stack, tempOutputFile }, "dependency tree output file is empty, retrying");
                    return true;
                }
                return false;
            },
        });
    }
    resolvePath(filePath) {
        try {
            const stats = fs.lstatSync(filePath);
            if (stats.isSymbolicLink()) {
                return fs.realpathSync(filePath);
            }
            else {
                return filePath;
            }
        }
        catch (error) {
            builder_util_1.log.debug({ message: error.message || error.stack }, "error resolving path");
            return filePath;
        }
    }
    getTreeFromWorkspaces(tree) {
        if (tree.workspaces && tree.dependencies) {
            const packageJson = require(path.join(this.rootDir, "package.json"));
            const dependencyName = packageJson.name;
            for (const [key, value] of Object.entries(tree.dependencies)) {
                if (key === dependencyName) {
                    return value;
                }
            }
        }
        return tree;
    }
    transToHoisterTree(obj, key = `.`, nodes = new Map()) {
        let node = nodes.get(key);
        const name = key.match(/@?[^@]+/)[0];
        if (!node) {
            node = {
                name,
                identName: name,
                reference: key.match(/@?[^@]+@?(.+)?/)[1] || ``,
                dependencies: new Set(),
                peerNames: new Set([]),
            };
            nodes.set(key, node);
            for (const dep of (obj[key] || {}).dependencies || []) {
                node.dependencies.add(this.transToHoisterTree(obj, dep, nodes));
            }
        }
        return node;
    }
    _getNodeModules(dependencies, result) {
        var _a;
        if (dependencies.size === 0) {
            return;
        }
        for (const d of dependencies.values()) {
            const reference = [...d.references][0];
            const p = (_a = this.allDependencies.get(`${d.name}@${reference}`)) === null || _a === void 0 ? void 0 : _a.path;
            if (p === undefined) {
                builder_util_1.log.debug({ name: d.name, reference }, "cannot find path for dependency");
                continue;
            }
            const node = {
                name: d.name,
                version: reference,
                dir: this.resolvePath(p),
            };
            result.push(node);
            if (d.dependencies.size > 0) {
                node.dependencies = [];
                this._getNodeModules(d.dependencies, node.dependencies);
            }
        }
        result.sort((a, b) => a.name.localeCompare(b.name));
    }
    static async safeExec(command, args, cwd) {
        const payload = await execAsync([`"${command}"`, ...args].join(" "), { cwd, maxBuffer: 100 * 1024 * 1024 }); // 100MB buffer LOL, some projects can have extremely large dependency trees
        return payload.stdout.trim();
    }
    async streamCollectorCommandToJsonFile(command, args, cwd, tempOutputFile) {
        const execName = path.basename(command, path.extname(command));
        const isWindowsScriptFile = process.platform === "win32" && path.extname(command).toLowerCase() === ".cmd";
        if (isWindowsScriptFile) {
            // If the command is a Windows script file (.cmd), we need to wrap it in a .bat file to ensure it runs correctly with cmd.exe
            // This is necessary because .cmd files are not directly executable in the same way as .bat files.
            // We create a temporary .bat file that calls the .cmd file with the provided arguments. The .bat file will be executed by cmd.exe.
            const tempBatFile = await this.tempDirManager.getTempFile({
                prefix: execName,
                suffix: ".bat",
            });
            const batScript = `@echo off\r\n"${command}" %*\r\n`; // <-- CRLF required for .bat
            await fs.writeFile(tempBatFile, batScript, { encoding: "utf8" });
            command = "cmd.exe";
            args = ["/c", tempBatFile, ...args];
        }
        await new Promise((resolve, reject) => {
            const outStream = (0, fs_1.createWriteStream)(tempOutputFile);
            const child = (0, child_process_1.spawn)(command, args, {
                cwd,
                shell: false, // required to prevent console logs polution from shell profile loading when `true`
            });
            let stderr = "";
            child.stdout.pipe(outStream);
            child.stderr.on("data", chunk => {
                stderr += chunk.toString();
            });
            child.on("error", err => {
                reject(new Error(`Spawn failed: ${err.message}`));
            });
            child.on("close", code => {
                outStream.close();
                // https://github.com/npm/npm/issues/17624
                if (code === 1 && execName.toLowerCase() === "npm" && args.includes("list")) {
                    builder_util_1.log.debug({ code, stderr }, "`npm list` returned non-zero exit code, but it MIGHT be expected (https://github.com/npm/npm/issues/17624). Check stderr for details.");
                    // This is a known issue with npm list command, it can return code 1 even when the command is "technically" successful
                    resolve();
                    return;
                }
                if (code !== 0) {
                    return reject(new Error(`Process exited with code ${code}:\n${stderr}`));
                }
                resolve();
            });
        });
    }
}
exports.NodeModulesCollector = NodeModulesCollector;
//# sourceMappingURL=nodeModulesCollector.js.map