/// <reference types="node" />
import * as http from 'node:http';
import { Span } from '@opentelemetry/api';
import { InstrumentationConfig } from '@opentelemetry/instrumentation';
import { InstrumentationBase, InstrumentationNodeModuleDefinition } from '@opentelemetry/instrumentation';
export type SentryHttpInstrumentationOptions = InstrumentationConfig & {
    /**
     * Whether breadcrumbs should be recorded for requests.
     *
     * @default `true`
     */
    breadcrumbs?: boolean;
    /**
     * Whether to create spans for requests or not.
     * As of now, creates spans for incoming requests, but not outgoing requests.
     *
     * @default `true`
     */
    spans?: boolean;
    /**
     * Whether to extract the trace ID from the `sentry-trace` header for incoming requests.
     * By default this is done by the HttpInstrumentation, but if that is not added (e.g. because tracing is disabled, ...)
     * then this instrumentation can take over.
     *
     * @deprecated This is always true and the option will be removed in the future.
     *
     * @default `true`
     */
    extractIncomingTraceFromHeader?: boolean;
    /**
     * Whether to propagate Sentry trace headers in outgoing requests.
     * By default this is done by the HttpInstrumentation, but if that is not added (e.g. because tracing is disabled)
     * then this instrumentation can take over.
     *
     * @default `false`
     */
    propagateTraceInOutgoingRequests?: boolean;
    /**
     * Whether to automatically ignore common static asset requests like favicon.ico, robots.txt, etc.
     * This helps reduce noise in your transactions.
     *
     * @default `true`
     */
    ignoreStaticAssets?: boolean;
    /**
     * If true, do not generate spans for incoming requests at all.
     * This is used by Remix to avoid generating spans for incoming requests, as it generates its own spans.
     */
    disableIncomingRequestSpans?: boolean;
    /**
     * Do not capture breadcrumbs for outgoing HTTP requests to URLs where the given callback returns `true`.
     * For the scope of this instrumentation, this callback only controls breadcrumb creation.
     * The same option can be passed to the top-level httpIntegration where it controls both, breadcrumb and
     * span creation.
     *
     * @param url Contains the entire URL, including query string (if any), protocol, host, etc. of the outgoing request.
     * @param request Contains the {@type RequestOptions} object used to make the outgoing request.
     */
    ignoreOutgoingRequests?: (url: string, request: http.RequestOptions) => boolean;
    /**
     * Do not capture spans for incoming HTTP requests to URLs where the given callback returns `true`.
     *
     * @param urlPath Contains the URL path and query string (if any) of the incoming request.
     * @param request Contains the {@type IncomingMessage} object of the incoming request.
     */
    ignoreSpansForIncomingRequests?: (urlPath: string, request: http.IncomingMessage) => boolean;
    /**
     * Do not capture the request body for incoming HTTP requests to URLs where the given callback returns `true`.
     * This can be useful for long running requests where the body is not needed and we want to avoid capturing it.
     *
     * @param url Contains the entire URL, including query string (if any), protocol, host, etc. of the incoming request.
     * @param request Contains the {@type RequestOptions} object used to make the incoming request.
     */
    ignoreIncomingRequestBody?: (url: string, request: http.RequestOptions) => boolean;
    /**
     * A hook that can be used to mutate the span for incoming requests.
     * This is triggered after the span is created, but before it is recorded.
     */
    incomingRequestSpanHook?: (span: Span, request: http.IncomingMessage, response: http.ServerResponse) => void;
    /**
     * Controls the maximum size of incoming HTTP request bodies attached to events.
     *
     * Available options:
     * - 'none': No request bodies will be attached
     * - 'small': Request bodies up to 1,000 bytes will be attached
     * - 'medium': Request bodies up to 10,000 bytes will be attached (default)
     * - 'always': Request bodies will always be attached
     *
     * Note that even with 'always' setting, bodies exceeding 1MB will never be attached
     * for performance and security reasons.
     *
     * @default 'medium'
     */
    maxIncomingRequestBodySize?: 'none' | 'small' | 'medium' | 'always';
    /**
     * Whether the integration should create [Sessions](https://docs.sentry.io/product/releases/health/#sessions) for incoming requests to track the health and crash-free rate of your releases in Sentry.
     * Read more about Release Health: https://docs.sentry.io/product/releases/health/
     *
     * Defaults to `true`.
     */
    trackIncomingRequestsAsSessions?: boolean;
    /**
     * @deprecated This is deprecated in favor of `incomingRequestSpanHook`.
     */
    instrumentation?: {
        requestHook?: (span: Span, req: http.ClientRequest | http.IncomingMessage) => void;
        responseHook?: (span: Span, response: http.IncomingMessage | http.ServerResponse) => void;
        applyCustomAttributesOnSpan?: (span: Span, request: http.ClientRequest | http.IncomingMessage, response: http.IncomingMessage | http.ServerResponse) => void;
    };
    /**
     * Number of milliseconds until sessions tracked with `trackIncomingRequestsAsSessions` will be flushed as a session aggregate.
     *
     * Defaults to `60000` (60s).
     */
    sessionFlushingDelayMS?: number;
};
/**
 * This custom HTTP instrumentation is used to isolate incoming requests and annotate them with additional information.
 * It does not emit any spans.
 *
 * The reason this is isolated from the OpenTelemetry instrumentation is that users may overwrite this,
 * which would lead to Sentry not working as expected.
 *
 * Important note: Contrary to other OTEL instrumentation, this one cannot be unwrapped.
 * It only does minimal things though and does not emit any spans.
 *
 * This is heavily inspired & adapted from:
 * https://github.com/open-telemetry/opentelemetry-js/blob/f8ab5592ddea5cba0a3b33bf8d74f27872c0367f/experimental/packages/opentelemetry-instrumentation-http/src/http.ts
 */
export declare class SentryHttpInstrumentation extends InstrumentationBase<SentryHttpInstrumentationOptions> {
    private _propagationDecisionMap;
    private _ignoreOutgoingRequestsMap;
    constructor(config?: SentryHttpInstrumentationOptions);
    /** @inheritdoc */
    init(): [
        InstrumentationNodeModuleDefinition,
        InstrumentationNodeModuleDefinition
    ];
    /**
     * This is triggered when an outgoing request finishes.
     * It has access to the final request and response objects.
     */
    private _onOutgoingRequestFinish;
    /**
     * This is triggered when an outgoing request is created.
     * It has access to the request object, and can mutate it before the request is sent.
     */
    private _onOutgoingRequestCreated;
    /**
     * Check if the given outgoing request should be ignored.
     */
    private _shouldIgnoreOutgoingRequest;
}
//# sourceMappingURL=SentryHttpInstrumentation.d.ts.map
