"use strict";
/* file : BitSet.ts
MIT License

Copyright (c) 2021 David Leppik

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
*/
Object.defineProperty(exports, "__esModule", { value: true });
var base64_arraybuffer_1 = require("base64-arraybuffer");
var bitsPerWord = 8;
/**
 * A memory-efficient Boolean array. Contains just the minimal operations needed for our Bloom filter implementation.
 *
 * @author David Leppik
 */
var BitSet = /** @class */ (function () {
    /**
     * Constructor. All bits are initially set to false.
     * @param size the number of bits that can be stored. (This is NOT required to be a multiple of 8.)
     */
    function BitSet(size) {
        var diff = bitsPerWord - (size % bitsPerWord);
        this.size = size + ([0, 8].includes(diff) ? 0 : diff);
        this.array = new Uint8Array(Math.ceil(this.size / bitsPerWord));
    }
    /**
     * Returns the value of the bit at the given index
     * @param index position of the bit, zero-indexed
     */
    BitSet.prototype.has = function (index) {
        var wordIndex = Math.floor(index / bitsPerWord);
        var mask = 1 << index % bitsPerWord;
        return (this.array[wordIndex] & mask) !== 0;
    };
    /**
     * Set the bit to true
     * @param index position of the bit, zero-indexed
     */
    BitSet.prototype.add = function (index) {
        var wordIndex = Math.floor(index / bitsPerWord);
        var mask = 1 << index % bitsPerWord;
        this.array[wordIndex] = this.array[wordIndex] | mask;
    };
    /**
     * Returns the maximum true bit.
     */
    BitSet.prototype.max = function () {
        for (var i = this.array.length - 1; i >= 0; i--) {
            var bits = this.array[i];
            if (bits) {
                return BitSet.highBit(bits) + i * bitsPerWord;
            }
        }
        return 0;
    };
    /**
     * Returns the number of true bits.
     */
    BitSet.prototype.bitCount = function () {
        var result = 0;
        for (var i = 0; i < this.array.length; i++) {
            result += BitSet.countBits(this.array[i]); // Assumes we never have bits set beyond the end
        }
        return result;
    };
    /**
     * Returns true if the size and contents are identical.
     * @param other another BitSet
     */
    BitSet.prototype.equals = function (other) {
        if (other.size !== this.size) {
            return false;
        }
        for (var i = 0; i < this.array.length; i++) {
            if (this.array[i] !== other.array[i]) {
                return false;
            }
        }
        return true;
    };
    /**
     * Returns a JSON-encodable object readable by {@link import}.
     */
    BitSet.prototype.export = function () {
        return {
            size: this.size,
            content: (0, base64_arraybuffer_1.encode)(this.array),
        };
    };
    /**
     * Returns an object written by {@link export}.
     * @param data an object written by {@link export}
     */
    BitSet.import = function (data) {
        if (typeof data.size !== 'number') {
            throw Error('BitSet missing size');
        }
        if (typeof data.content !== 'string') {
            throw Error('BitSet: missing content');
        }
        var result = new BitSet(data.size);
        var buffer = (0, base64_arraybuffer_1.decode)(data.content);
        result.array = new Uint8Array(buffer);
        return result;
    };
    /**
     * Returns the index of the maximum bit in the number, or -1 for 0
     * @bits an unsigned 8-bit number
     * ```js
     * @example
     * BitSet.highBit(0) // returns -1
     * BitSet.highBit(5) // returns 2
     * ```
     */
    BitSet.highBit = function (bits) {
        var result = bitsPerWord - 1;
        var mask = 1 << result;
        while (result >= 0 && (mask & bits) !== mask) {
            mask >>>= 1;
            result--;
        }
        return result;
    };
    /**
     * Returns the number of true bits in the number
     * @bits an unsigned 8-bit number
     * @example
     * ```js
     * BitSet.countBits(0) // returns 0
     * BitSet.countBits(3) // returns 2
     * ```
     */
    BitSet.countBits = function (bits) {
        var result = bits & 1;
        while (bits !== 0) {
            bits = bits >>> 1;
            result += bits & 1;
        }
        return result;
    };
    return BitSet;
}());
exports.default = BitSet;
