import { ComponentType } from 'react';
import { JSX } from 'react';
import { ModuleApi } from '@matrix-org/react-sdk-module-api';
import { Root } from 'react-dom/client';
import { RuntimeModule } from '@matrix-org/react-sdk-module-api';

/**
 * Methods to manage authentication in the application.
 * @public
 */
export declare interface AccountAuthApiExtension {
    /**
     * Overwrite the current account's authentication state with the provided account information.
     * @param accountInfo - The account authentication information to overwrite the current state with.
     */
    overwriteAccountAuth(accountInfo: AccountAuthInfo): Promise<void>;
}

/**
 * Interface for account authentication information, used for overwriting the current account's authentication state.
 * @public
 */
export declare interface AccountAuthInfo {
    /**
     * The user ID.
     */
    userId: string;
    /**
     * The device ID.
     */
    deviceId: string;
    /**
     * The access token belonging to this device ID and user ID.
     */
    accessToken: string;
    /**
     * The refresh token belonging to this device ID and user ID.
     */
    refreshToken?: string;
    /**
     * The homeserver URL where the credentials are valid.
     */
    homeserverUrl: string;
}

/* Excluded from this release type: AliasCustomisations */

/**
 * The API for modules to interact with the application.
 * @public
 */
export declare interface Api extends LegacyModuleApiExtension, LegacyCustomisationsApiExtension, DialogApiExtension, AccountAuthApiExtension, ProfileApiExtension {
    /**
     * The API to read config.json values.
     * Keys should be scoped to the module in reverse domain name notation.
     * @public
     */
    readonly config: ConfigApi;
    /**
     * The internationalisation API.
     * @public
     */
    readonly i18n: I18nApi;
    /**
     * The root node the main application is rendered to.
     * Intended for rendering sibling React trees.
     * @public
     */
    readonly rootNode: HTMLElement;
    /* Excluded from this release type: customComponents */
    /**
     * API to navigate the application.
     * @public
     */
    readonly navigation: NavigationApi;
    /**
     * Create a ReactDOM root for rendering React components.
     * Exposed to allow modules to avoid needing to bundle their own ReactDOM.
     * @param element - the element to render use as the root.
     * @public
     */
    createRoot(element: Element): Root;
}

/* Excluded from this release type: ChatExportCustomisations */

/* Excluded from this release type: ComponentVisibilityCustomisations */

/**
 * The configuration for the application.
 * Should be extended via declaration merging.
 * @public
 */
export declare interface Config {
    brand: string;
}

/**
 * API for accessing the configuration.
 * @public
 */
export declare interface ConfigApi {
    get(): Config;
    get<K extends keyof Config>(key: K): Config[K];
    get<K extends keyof Config = never>(key?: K): Config | Config[K];
}

/* Excluded from this release type: CustomComponentsApi */

/* Excluded from this release type: CustomMessageComponentProps */

/* Excluded from this release type: CustomMessageRenderFunction */

/* Excluded from this release type: CustomMessageRenderHints */

/* Excluded from this release type: CustomRoomPreviewBarComponentProps */

/* Excluded from this release type: CustomRoomPreviewBarRenderFunction */

/**
 * Methods to manage dialogs in the application.
 * @public
 */
export declare interface DialogApiExtension {
    /**
     * Open a dialog with the given options and body component and return a handle to it.
     * @param initialOptions - The initial options for the dialog, such as title and action label.
     * @param dialog - The body component to render in the dialog. This component should accept props of type `P`.
     * @param props - Additional props to pass to the body
     */
    openDialog<M, P extends object>(initialOptions: DialogOptions, dialog: ComponentType<P & DialogProps<M>>, props: P): DialogHandle<M>;
}

/**
 * Handle returned by {@link Api#openDialog}.
 * @public
 */
export declare type DialogHandle<M> = {
    /**
     * Promise that resolves when the dialog is finished.
     */
    finished: Promise<{
        ok: boolean;
        model: M | null;
    }>;
    /**
     * Method to close the dialog.
     */
    close(): void;
};

/**
 * Options for {@link Api#openDialog}.
 * @public
 */
export declare interface DialogOptions {
    /**
     * The title of the dialog.
     */
    title: string;
}

/**
 * Props passed to the dialog body component.
 * @public
 */
export declare type DialogProps<M> = {
    /**
     * Callback to submit the dialog.
     * @param model - The model to submit with the dialog. This is typically the data collected.
     */
    onSubmit(model: M): void;
    /**
     * Cancel the dialog programmatically.
     */
    onCancel(): void;
};

/* Excluded from this release type: DirectoryCustomisations */

/**
 * The API for interacting with translations.
 * @public
 */
export declare interface I18nApi {
    /**
     * Read the current language of the user in IETF Language Tag format
     */
    get language(): string;
    /**
     * Register translations for the module, may override app's existing translations
     */
    register(translations: Partial<Translations>): void;
    /**
     * Perform a translation, with optional variables
     * @param key - The key to translate
     * @param variables - Optional variables to interpolate into the translation
     */
    translate(key: keyof Translations, variables?: Variables): string;
}

/* Excluded from this release type: LegacyCustomisations */

/* Excluded from this release type: LegacyCustomisationsApiExtension */

/* Excluded from this release type: LegacyModuleApiExtension */

/* Excluded from this release type: LifecycleCustomisations */

/* Excluded from this release type: MatrixEvent */

/* Excluded from this release type: Media */

/* Excluded from this release type: MediaContructable */

/* Excluded from this release type: MediaCustomisations */

/**
 * Module interface for modules to implement.
 * @public
 */
export declare interface Module {
    load(): Promise<void>;
}

declare interface ModuleExport {
    default: ModuleFactory;
}

/**
 * Module interface for modules to export as the default export.
 * @public
 */
export declare interface ModuleFactory {
    readonly moduleApiVersion: string;
    new (api: Api): Module;
    readonly prototype: Module;
}

/**
 * Error thrown when a module is incompatible with the engine version.
 * @public
 */
export declare class ModuleIncompatibleError extends Error {
    constructor(pluginVersion: string);
}

/**
 * A module loader for loading and starting modules.
 * @public
 */
export declare class ModuleLoader {
    private api;
    private modules;
    private started;
    constructor(api: Api);
    load(moduleExport: ModuleExport): Promise<void>;
    start(): Promise<void>;
}

/**
 * API methods to navigate the application.
 * @public
 */
export declare interface NavigationApi {
    /**
     * Navigate to a permalink, optionally causing a join if the user is not already a member of the room/space.
     * @param link - The permalink to navigate to, e.g. `https://matrix.to/#/!roomId:example.com`.
     * @param join - If true, the user will be made to attempt to join the room/space if they are not already a member.
     */
    toMatrixToLink(link: string, join?: boolean): Promise<void>;
}

/* Excluded from this release type: OriginalMessageComponentProps */

/**
 * The profile of the user currently logged in.
 * @public
 */
export declare interface Profile {
    /**
     * Indicates whether the user is a guest user.
     */
    isGuest?: boolean;
    /**
     * The user ID of the logged-in user, if undefined then no user is logged in.
     */
    userId?: string;
    /**
     * The display name of the logged-in user.
     */
    displayName?: string;
}

/**
 * API extensions for modules to access the profile of the logged-in user.
 * @public
 */
export declare interface ProfileApiExtension {
    /**
     * The profile of the user currently logged in.
     */
    readonly profile: Watchable<Profile>;
}

/* Excluded from this release type: RoomListCustomisations */

/* Excluded from this release type: RuntimeModuleConstructor */

/**
 * The translations for the module.
 * @public
 */
export declare type Translations = Record<string, {
    [ietfLanguageTag: string]: string;
}>;

/* Excluded from this release type: UserIdentifierCustomisations */

/**
 * A React hook to use an updated Watchable value.
 * @param watchable - The Watchable instance to watch.
 * @returns The live value of the Watchable.
 * @public
 */
export declare function useWatchable<T>(watchable: Watchable<T>): T;

/**
 * Variables to interpolate into a translation.
 * @public
 */
export declare type Variables = {
    /**
     * The number of items to count for pluralised translations
     */
    count?: number;
    [key: string]: number | string | undefined;
};

/**
 * Utility class to wrap a value and allow listeners to be notified when the value changes.
 * If T is an object, it will use a shallow comparison to determine if the value has changed.
 * @public
 */
export declare class Watchable<T> {
    private currentValue;
    private readonly listeners;
    constructor(currentValue: T);
    get value(): T;
    set value(value: T);
    watch(listener: (value: T) => void): void;
    unwatch(listener: (value: T) => void): void;
}

/* Excluded from this release type: WidgetPermissionsCustomisations */

/* Excluded from this release type: WidgetVariablesCustomisations */

export { }
