import { ComponentType } from 'react';
import { JSX } from 'react';
import { ModuleApi } from '@matrix-org/react-sdk-module-api';
import { Root } from 'react-dom/client';
import { RuntimeModule } from '@matrix-org/react-sdk-module-api';

/**
 * Methods to manage authentication in the application.
 * @public
 */
export declare interface AccountAuthApiExtension {
    /**
     * Overwrite the current account's authentication state with the provided account information.
     * @param accountInfo - The account authentication information to overwrite the current state with.
     */
    overwriteAccountAuth(accountInfo: AccountAuthInfo): Promise<void>;
}

/**
 * Interface for account authentication information, used for overwriting the current account's authentication state.
 * @public
 */
export declare interface AccountAuthInfo {
    /**
     * The user ID.
     */
    userId: string;
    /**
     * The device ID.
     */
    deviceId: string;
    /**
     * The access token belonging to this device ID and user ID.
     */
    accessToken: string;
    /**
     * The refresh token belonging to this device ID and user ID.
     */
    refreshToken?: string;
    /**
     * The homeserver URL where the credentials are valid.
     */
    homeserverUrl: string;
}

/**
 * The types here suck but these customisations are deprecated and will be removed soon.
 */
/**
 * @alpha
 * @deprecated in favour of the new Module API
 */
export declare interface AliasCustomisations {
    getDisplayAliasForAliasSet?(canonicalAlias: string | null, altAliases: string[]): string | null;
}

/**
 * The API for modules to interact with the application.
 * @public
 */
export declare interface Api extends LegacyModuleApiExtension, LegacyCustomisationsApiExtension, DialogApiExtension, AccountAuthApiExtension, ProfileApiExtension {
    /**
     * The API to read config.json values.
     * Keys should be scoped to the module in reverse domain name notation.
     * @public
     */
    readonly config: ConfigApi;
    /**
     * The internationalisation API.
     * @public
     */
    readonly i18n: I18nApi;
    /**
     * The root node the main application is rendered to.
     * Intended for rendering sibling React trees.
     * @public
     */
    readonly rootNode: HTMLElement;
    /**
     * The custom message component API.
     * @alpha
     */
    readonly customComponents: CustomComponentsApi;
    /**
     * API to navigate the application.
     * @public
     */
    readonly navigation: NavigationApi;
    /**
     * Create a ReactDOM root for rendering React components.
     * Exposed to allow modules to avoid needing to bundle their own ReactDOM.
     * @param element - the element to render use as the root.
     * @public
     */
    createRoot(element: Element): Root;
}

/**
 * @alpha
 * @deprecated in favour of the new Module API
 */
export declare interface ChatExportCustomisations<ExportFormat, ExportType> {
    /**
     * Force parameters in room chat export fields returned here are forced
     * and not allowed to be edited in the chat export form
     */
    getForceChatExportParameters(): {
        format?: ExportFormat;
        range?: ExportType;
        numberOfMessages?: number;
        includeAttachments?: boolean;
        sizeMb?: number;
    };
}

/**
 * @alpha
 * @deprecated in favour of the new Module API
 */
export declare interface ComponentVisibilityCustomisations {
    /**
     * Determines whether or not the active MatrixClient user should be able to use
     * the given UI component. If shown, the user might still not be able to use the
     * component depending on their contextual permissions. For example, invite options
     * might be shown to the user but they won't have permission to invite users to
     * the current room: the button will appear disabled.
     * @param component - The component to check visibility for.
     * @returns True (default) if the user is able to see the component, false otherwise.
     */
    shouldShowComponent?(component: "UIComponent.sendInvites" | "UIComponent.roomCreation" | "UIComponent.spaceCreation" | "UIComponent.exploreRooms" | "UIComponent.addIntegrations" | "UIComponent.filterContainer" | "UIComponent.roomOptionsMenu"): boolean;
}

/**
 * The configuration for the application.
 * Should be extended via declaration merging.
 * @public
 */
export declare interface Config {
    brand: string;
}

/**
 * API for accessing the configuration.
 * @public
 */
export declare interface ConfigApi {
    get(): Config;
    get<K extends keyof Config>(key: K): Config[K];
    get<K extends keyof Config = never>(key?: K): Config | Config[K];
}

/**
 * API for inserting custom components into Element.
 * @alpha Subject to change.
 */
export declare interface CustomComponentsApi {
    /**
     * Register a renderer for a message type in the timeline.
     *
     * The render function should return a rendered component.
     *
     * Multiple render function may be registered for a single event type, however the first matching
     * result will be used. If no events match or are registered then the originalComponent is rendered.
     *
     * @param eventTypeOrFilter - The event type this renderer is for. Use a function for more complex filtering.
     * @param renderer - The render function.
     * @param hints - Hints that alter the way the tile is handled.
     * @example
     * ```
     *  customComponents.registerMessageRenderer("m.room.message", (props, originalComponent) => {
     *       return <YourCustomComponent mxEvent={props.mxEvent} />;
     *  });
     *  customComponents.registerMessageRenderer(
     *      (mxEvent) => mxEvent.getType().matches(/m\.room\.(topic|name)/) && mxEvent.isState(),
     *      (props, originalComponent) => {
     *          return <YourCustomStateRenderer mxEvent={props.mxEvent} />;
     *      }
     * );
     * ```
     */
    registerMessageRenderer(eventTypeOrFilter: string | ((mxEvent: MatrixEvent) => boolean), renderer: CustomMessageRenderFunction, hints?: CustomMessageRenderHints): void;
    /**
     * Register a renderer for the room preview bar.
     *
     * The render function should return a rendered component.
     *
     * @param renderer - The render function for the room preview bar.
     * @example
     * ```
     *  customComponents.registerRoomPreviewBar((props, OriginalComponent) => {
     *      if (props.roomId === "!some_special_room_id:server") {
     *          return <YourCustomRoomPreviewBarComponent {...props} />;
     *      }
     *      return <YourCustomComponent mxEvent={props.mxEvent} />;
     *  });
     * ```
     */
    registerRoomPreviewBar(renderer: CustomRoomPreviewBarRenderFunction): void;
}

/**
 * Properties for all message components.
 * @alpha Subject to change.
 */
export declare type CustomMessageComponentProps = {
    /**
     * The Matrix event for this textual body.
     * @alpha
     */
    mxEvent: MatrixEvent;
};

/**
 * Function used to render a message component.
 * @alpha Subject to change.
 */
export declare type CustomMessageRenderFunction = (
/**
 * Properties for the message to be renderered.
 */
props: CustomMessageComponentProps, 
/**
 * Render function for the original component. This may be omitted if the message would not normally be rendered.
 */
originalComponent?: (props?: OriginalMessageComponentProps) => React.JSX.Element) => JSX.Element;

/**
 * Hints to specify to Element when rendering events.
 * @alpha Subject to change.
 */
export declare type CustomMessageRenderHints = {
    /**
     * Should the event be allowed to be edited in the client. This should
     * be set to false if you override the render function, as the module
     * API has no way to display message editing at the moment.
     * Default is true.
     */
    allowEditingEvent?: boolean;
    /**
     * If an event contains media, this function will be called to check
     * if the media can be prompted to be downloaded as a file.
     * If this function is not supplied, media downloads are allowed.
     */
    allowDownloadingMedia?: (mxEvent: MatrixEvent) => Promise<boolean>;
};

/**
 * Properties for all message components.
 * @alpha Subject to change.
 */
export declare type CustomRoomPreviewBarComponentProps = {
    roomId?: string;
    roomAlias?: string;
};

/**
 * Function used to render a room preview bar component.
 * @alpha Unlikely to change
 */
export declare type CustomRoomPreviewBarRenderFunction = (
/**
 * Properties for the room preview bar to be rendered.
 */
props: CustomRoomPreviewBarComponentProps, 
/**
 * Render function for the original component.
 */
originalComponent: (props: CustomRoomPreviewBarComponentProps) => JSX.Element) => JSX.Element;

/**
 * Methods to manage dialogs in the application.
 * @public
 */
export declare interface DialogApiExtension {
    /**
     * Open a dialog with the given options and body component and return a handle to it.
     * @param initialOptions - The initial options for the dialog, such as title and action label.
     * @param dialog - The body component to render in the dialog. This component should accept props of type `P`.
     * @param props - Additional props to pass to the body
     */
    openDialog<M, P extends object>(initialOptions: DialogOptions, dialog: ComponentType<P & DialogProps<M>>, props: P): DialogHandle<M>;
}

/**
 * Handle returned by {@link Api#openDialog}.
 * @public
 */
export declare type DialogHandle<M> = {
    /**
     * Promise that resolves when the dialog is finished.
     */
    finished: Promise<{
        ok: boolean;
        model: M | null;
    }>;
    /**
     * Method to close the dialog.
     */
    close(): void;
};

/**
 * Options for {@link Api#openDialog}.
 * @public
 */
export declare interface DialogOptions {
    /**
     * The title of the dialog.
     */
    title: string;
}

/**
 * Props passed to the dialog body component.
 * @public
 */
export declare type DialogProps<M> = {
    /**
     * Callback to submit the dialog.
     * @param model - The model to submit with the dialog. This is typically the data collected.
     */
    onSubmit(model: M): void;
    /**
     * Cancel the dialog programmatically.
     */
    onCancel(): void;
};

/**
 * @alpha
 * @deprecated in favour of the new Module API
 */
export declare interface DirectoryCustomisations {
    requireCanonicalAliasAccessToPublish?(): boolean;
}

/**
 * The API for interacting with translations.
 * @public
 */
export declare interface I18nApi {
    /**
     * Read the current language of the user in IETF Language Tag format
     */
    get language(): string;
    /**
     * Register translations for the module, may override app's existing translations
     */
    register(translations: Partial<Translations>): void;
    /**
     * Perform a translation, with optional variables
     * @param key - The key to translate
     * @param variables - Optional variables to interpolate into the translation
     */
    translate(key: keyof Translations, variables?: Variables): string;
}

/**
 * @alpha
 * @deprecated in favour of the new Module API
 */
export declare type LegacyCustomisations<T extends object> = (customisations: T) => void;

/**
 * @alpha
 * @deprecated in favour of the new Module API
 */
export declare interface LegacyCustomisationsApiExtension {
    /**
     * @deprecated in favour of the new Module API
     */
    readonly _registerLegacyAliasCustomisations: LegacyCustomisations<AliasCustomisations>;
    /**
     * @deprecated in favour of the new Module API
     */
    readonly _registerLegacyChatExportCustomisations: LegacyCustomisations<ChatExportCustomisations<never, never>>;
    /**
     * @deprecated in favour of the new Module API
     */
    readonly _registerLegacyComponentVisibilityCustomisations: LegacyCustomisations<ComponentVisibilityCustomisations>;
    /**
     * @deprecated in favour of the new Module API
     */
    readonly _registerLegacyDirectoryCustomisations: LegacyCustomisations<DirectoryCustomisations>;
    /**
     * @deprecated in favour of the new Module API
     */
    readonly _registerLegacyLifecycleCustomisations: LegacyCustomisations<LifecycleCustomisations>;
    /**
     * @deprecated in favour of the new Module API
     */
    readonly _registerLegacyMediaCustomisations: LegacyCustomisations<MediaCustomisations<never, never, never>>;
    /**
     * @deprecated in favour of the new Module API
     */
    readonly _registerLegacyRoomListCustomisations: LegacyCustomisations<RoomListCustomisations<never>>;
    /**
     * @deprecated in favour of the new Module API
     */
    readonly _registerLegacyUserIdentifierCustomisations: LegacyCustomisations<UserIdentifierCustomisations>;
    /**
     * @deprecated in favour of the new Module API
     */
    readonly _registerLegacyWidgetPermissionsCustomisations: LegacyCustomisations<WidgetPermissionsCustomisations<never, never>>;
    /**
     * @deprecated in favour of the new Module API
     */
    readonly _registerLegacyWidgetVariablesCustomisations: LegacyCustomisations<WidgetVariablesCustomisations>;
}

/**
 * @alpha
 * @deprecated in favour of the new module API
 */
export declare interface LegacyModuleApiExtension {
    /**
     * Register a legacy module based on \@matrix-org/react-sdk-module-api
     * @param LegacyModule - the module class to register
     * @deprecated provided only as a transition path for legacy modules
     */
    _registerLegacyModule(LegacyModule: RuntimeModuleConstructor): Promise<void>;
}

/**
 * @alpha
 * @deprecated in favour of the new Module API
 */
export declare interface LifecycleCustomisations {
    onLoggedOutAndStorageCleared?(): void;
}

/**
 * Representation of a Matrix event, as specified by the client server specification.
 * @alpha Subject to change.
 * @see https://spec.matrix.org/v1.14/client-server-api/#room-event-format
 */
export declare interface MatrixEvent {
    /**
     * The event ID of this event.
     */
    eventId: string;
    /**
     * The room ID which contains this event.
     */
    roomId: string;
    /**
     * The Matrix ID of the user who sent this event.
     */
    sender: string;
    /**
     * The content of the event.
     * If the event was encrypted, this is the decrypted content.
     */
    content: Record<string, unknown>;
    /**
     * Contains optional extra information about the event.
     */
    unsigned: Record<string, unknown>;
    /**
     * The type of the event.
     */
    type: string;
    /**
     * The state key of the event.
     * If this key is set, including `""` then the event is a state event.
     */
    stateKey?: string;
    /**
     * Timestamp (in milliseconds since the unix epoch) on originating homeserver when this event was sent.
     */
    originServerTs: number;
}

/**
 * @alpha
 * @deprecated in favour of the new Module API
 */
export declare interface Media {
    readonly isEncrypted: boolean;
    readonly srcMxc: string;
    readonly thumbnailMxc: string | null | undefined;
    readonly hasThumbnail: boolean;
    readonly srcHttp: string | null;
    readonly thumbnailHttp: string | null;
    getThumbnailHttp(width: number, height: number, mode?: "scale" | "crop"): string | null;
    getThumbnailOfSourceHttp(width: number, height: number, mode?: "scale" | "crop"): string | null;
    getSquareThumbnailHttp(dim: number): string | null;
    downloadSource(): Promise<Response>;
}

/**
 * @alpha
 * @deprecated in favour of the new Module API
 */
export declare interface MediaContructable<PreparedMedia> {
    new (prepared: PreparedMedia): Media;
}

/**
 * @alpha
 * @deprecated in favour of the new Module API
 */
export declare interface MediaCustomisations<Content, Client, PreparedMedia> {
    readonly Media: MediaContructable<PreparedMedia>;
    mediaFromContent(content: Content, client?: Client): Media;
    mediaFromMxc(mxc?: string, client?: Client): Media;
}

/**
 * Module interface for modules to implement.
 * @public
 */
export declare interface Module {
    load(): Promise<void>;
}

declare interface ModuleExport {
    default: ModuleFactory;
}

/**
 * Module interface for modules to export as the default export.
 * @public
 */
export declare interface ModuleFactory {
    readonly moduleApiVersion: string;
    new (api: Api): Module;
    readonly prototype: Module;
}

/**
 * Error thrown when a module is incompatible with the engine version.
 * @public
 */
export declare class ModuleIncompatibleError extends Error {
    constructor(pluginVersion: string);
}

/**
 * A module loader for loading and starting modules.
 * @public
 */
export declare class ModuleLoader {
    private api;
    private modules;
    private started;
    constructor(api: Api);
    load(moduleExport: ModuleExport): Promise<void>;
    start(): Promise<void>;
}

/**
 * API methods to navigate the application.
 * @public
 */
export declare interface NavigationApi {
    /**
     * Navigate to a permalink, optionally causing a join if the user is not already a member of the room/space.
     * @param link - The permalink to navigate to, e.g. `https://matrix.to/#/!roomId:example.com`.
     * @param join - If true, the user will be made to attempt to join the room/space if they are not already a member.
     */
    toMatrixToLink(link: string, join?: boolean): Promise<void>;
}

/**
 * Properties to alter the render function of the original component.
 * @alpha Subject to change.
 */
export declare type OriginalMessageComponentProps = {
    /**
     * Should previews be shown for this event.
     * This may be overriden by user preferences.
     */
    showUrlPreview?: boolean;
};

/**
 * The profile of the user currently logged in.
 * @public
 */
export declare interface Profile {
    /**
     * Indicates whether the user is a guest user.
     */
    isGuest?: boolean;
    /**
     * The user ID of the logged-in user, if undefined then no user is logged in.
     */
    userId?: string;
    /**
     * The display name of the logged-in user.
     */
    displayName?: string;
}

/**
 * API extensions for modules to access the profile of the logged-in user.
 * @public
 */
export declare interface ProfileApiExtension {
    /**
     * The profile of the user currently logged in.
     */
    readonly profile: Watchable<Profile>;
}

/**
 * @alpha
 * @deprecated in favour of the new Module API
 */
export declare interface RoomListCustomisations<Room> {
    /**
     * Determines if a room is visible in the room list or not. By default,
     * all rooms are visible. Where special handling is performed by Element,
     * those rooms will not be able to override their visibility in the room
     * list - Element will make the decision without calling this function.
     *
     * This function should be as fast as possible to avoid slowing down the
     * client.
     * @param room - The room to check the visibility of.
     * @returns True if the room should be visible, false otherwise.
     */
    isRoomVisible?(room: Room): boolean;
}

/**
 * @alpha
 * @deprecated in favour of the new module API
 */
export declare type RuntimeModuleConstructor = new (api: ModuleApi) => RuntimeModule;

/**
 * The translations for the module.
 * @public
 */
export declare type Translations = Record<string, {
    [ietfLanguageTag: string]: string;
}>;

/**
 * @alpha
 * @deprecated in favour of the new Module API
 */
export declare interface UserIdentifierCustomisations {
    /**
     * Customise display of the user identifier
     * hide userId for guests, display 3pid
     *
     * Set withDisplayName to true when user identifier will be displayed alongside user name
     */
    getDisplayUserIdentifier(userId: string, opts: {
        roomId?: string;
        withDisplayName?: boolean;
    }): string | null;
}

/**
 * A React hook to use an updated Watchable value.
 * @param watchable - The Watchable instance to watch.
 * @returns The live value of the Watchable.
 * @public
 */
export declare function useWatchable<T>(watchable: Watchable<T>): T;

/**
 * Variables to interpolate into a translation.
 * @public
 */
export declare type Variables = {
    /**
     * The number of items to count for pluralised translations
     */
    count?: number;
    [key: string]: number | string | undefined;
};

/**
 * Utility class to wrap a value and allow listeners to be notified when the value changes.
 * If T is an object, it will use a shallow comparison to determine if the value has changed.
 * @public
 */
export declare class Watchable<T> {
    private currentValue;
    private readonly listeners;
    constructor(currentValue: T);
    get value(): T;
    set value(value: T);
    watch(listener: (value: T) => void): void;
    unwatch(listener: (value: T) => void): void;
}

/**
 * @alpha
 * @deprecated in favour of the new Module API
 */
export declare interface WidgetPermissionsCustomisations<Widget, Capability> {
    /**
     * Approves the widget for capabilities that it requested, if any can be
     * approved. Typically this will be used to give certain widgets capabilities
     * without having to prompt the user to approve them. This cannot reject
     * capabilities that Element will be automatically granting, such as the
     * ability for Jitsi widgets to stay on screen - those will be approved
     * regardless.
     * @param widget - The widget to approve capabilities for.
     * @param requestedCapabilities - The capabilities the widget requested.
     * @returns Resolves to the capabilities that are approved for use
     * by the widget. If none are approved, this should return an empty Set.
     */
    preapproveCapabilities?(widget: Widget, requestedCapabilities: Set<Capability>): Promise<Set<Capability>>;
}

/**
 * @alpha
 * @deprecated in favour of the new Module API
 */
export declare interface WidgetVariablesCustomisations {
    /**
     * Provides a partial set of the variables needed to render any widget. If
     * variables are missing or not provided then they will be filled with the
     * application-determined defaults.
     *
     * This will not be called until after isReady() resolves.
     * @returns The variables.
     */
    provideVariables?(): {
        currentUserId: string;
        userDisplayName?: string;
        userHttpAvatarUrl?: string;
        clientId?: string;
        clientTheme?: string;
        clientLanguage?: string;
        deviceId?: string;
        baseUrl?: string;
    };
    /**
     * Resolves to whether or not the customisation point is ready for variables
     * to be provided. This will block widgets being rendered.
     * If not provided, the app will assume that the customisation is always ready.
     * @returns a promise which resolves when ready.
     */
    isReady?(): Promise<void>;
}

export { }
