"use strict";
/**
 * Inject preload files before the content of the targeted files
 *
 * @example
 * new HtmlWebpackInjectPreload({
 *  files: [
 *    {
 *      match: /.*\.woff2/$,
 *      attributes: { rel: 'preload', as: 'font', type: 'font/woff2',
 * crossorigin: true },
 *    },
 *    {
 *      match: /vendors\.[a-z-0-9]*.css/$,
 *      attributes: { rel: 'preload', as: 'style' },
 *    },
 *  ],
 * })
 *
 * @class InjectPreloadFiles
 */
class HtmlWebpackInjectPreload {
    /**
     * Creates an instance of HtmlWebpackInjectPreload.
     *
     * @memberof InjectPreloadFiles
     */
    constructor(options) {
        this.options = {
            files: [],
        };
        /**
         * Extract HTMLWebpack Plugin by jahed
         *
         * @param compiler
         */
        this.extractHtmlWebpackPluginModule = (compiler) => {
            const htmlWebpackPlugin = (compiler.options.plugins || []).find(plugin => {
                return plugin.constructor.name === 'HtmlWebpackPlugin';
            });
            if (!htmlWebpackPlugin) {
                return null;
            }
            const HtmlWebpackPlugin = htmlWebpackPlugin.constructor;
            if (!HtmlWebpackPlugin || !('getHooks' in HtmlWebpackPlugin)) {
                return null;
            }
            return HtmlWebpackPlugin;
        };
        this.options = Object.assign(this.options, options);
    }
    addLinks(compilation, htmlPluginData) {
        var _a, _b, _c;
        //Get public path
        //html-webpack-plugin v5
        let publicPath = htmlPluginData.publicPath;
        //html-webpack-plugin v4
        if (typeof publicPath === 'undefined') {
            if (((_a = htmlPluginData.plugin.options) === null || _a === void 0 ? void 0 : _a.publicPath) &&
                ((_b = htmlPluginData.plugin.options) === null || _b === void 0 ? void 0 : _b.publicPath) !== 'auto') {
                publicPath = (_c = htmlPluginData.plugin.options) === null || _c === void 0 ? void 0 : _c.publicPath;
            }
            else {
                publicPath =
                    typeof compilation.options.output.publicPath === 'string'
                        ? compilation.options.output.publicPath
                        : '/';
            }
            //prevent wrong url
            if (publicPath[publicPath.length - 1] !== '/') {
                publicPath = publicPath + '/';
            }
        }
        //Get assets name
        const assets = new Set(Object.keys(compilation.assets));
        compilation.chunks.forEach(chunk => {
            chunk.files.forEach((file) => assets.add(file));
        });
        //Find first link index to inject before
        const linkIndex = htmlPluginData.headTags.findIndex(tag => tag.tagName === 'link');
        assets.forEach(asset => {
            for (let index = 0; index < this.options.files.length; index++) {
                const file = this.options.files[index];
                if (file.match.test(asset)) {
                    let href = file.attributes && file.attributes.href
                        ? file.attributes.href
                        : false;
                    if (href === false || typeof href === 'undefined') {
                        href = asset;
                    }
                    href = href[0] === '/' ? href : publicPath + href;
                    const preload = {
                        tagName: 'link',
                        attributes: Object.assign({
                            rel: 'preload',
                            href,
                        }, file.attributes),
                        voidTag: true,
                        meta: {
                            plugin: 'html-webpack-inject-preload',
                        },
                    };
                    if (linkIndex > -1) {
                        //before link
                        htmlPluginData.headTags.splice(linkIndex, 0, preload);
                    }
                    else {
                        // before everything
                        htmlPluginData.headTags.unshift(preload);
                    }
                }
            }
        });
        return htmlPluginData;
    }
    apply(compiler) {
        compiler.hooks.compilation.tap('HtmlWebpackInjectPreload', compilation => {
            const HtmlWebpackPlugin = this.extractHtmlWebpackPluginModule(compiler);
            if (!HtmlWebpackPlugin) {
                throw new Error('HtmlWebpackInjectPreload needs to be used with html-webpack-plugin 4 or 5');
            }
            const hooks = HtmlWebpackPlugin.getHooks(compilation);
            hooks.alterAssetTagGroups.tapAsync('HtmlWebpackInjectPreload', (htmlPluginData, callback) => {
                try {
                    callback(null, this.addLinks(compilation, htmlPluginData));
                }
                catch (error) {
                    callback(error);
                }
            });
        });
    }
}
module.exports = HtmlWebpackInjectPreload;
