#!/usr/bin/env node

"use strict";
/*
Copyright 2023 The Matrix.org Foundation C.I.C.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Applies the following lint rules to the src/i18n/strings/en_EN.json file:
 *  + ensures the translation key is not equal to its value
 *  + ensures the translation key contains only alphanumerics and underscores (temporarily allows @ and . for compatibility)
 *  + ensures no forbidden hardcoded words are found (specified new line delimited in environment variable HARDCODED_WORDS)
 *    unless they are explicitly allowed (keys specified new line delimited in environment variable ALLOWED_HARDCODED_KEYS)
 *
 * Usage: node scripts/lint-i18n.js
 */
const common_1 = require("./common");
const hardcodedWords = (_b = (_a = process.env.HARDCODED_WORDS) === null || _a === void 0 ? void 0 : _a.toLowerCase().split("\n").map(k => k.trim())) !== null && _b !== void 0 ? _b : [];
const allowedHardcodedKeys = (_d = (_c = process.env.ALLOWED_HARDCODED_KEYS) === null || _c === void 0 ? void 0 : _c.split("\n").map(k => k.trim())) !== null && _d !== void 0 ? _d : [];
const input = (0, common_1.getTranslations)();
function nonNullable(value) {
    return value !== null && value !== undefined;
}
function expandTranslations(translation) {
    if ((0, common_1.isPluralisedTranslation)(translation)) {
        return [translation.one, translation.other].filter(nonNullable);
    }
    else {
        return [translation];
    }
}
function lintTranslation(keys, value) {
    const key = keys[keys.length - 1];
    const fullKey = keys.join(common_1.NESTING_KEY);
    // Check for invalid characters in the translation key
    if (!!key.replace(/[a-z0-9@_.]+/gi, "")) {
        console.log(`"${fullKey}": key contains invalid characters`);
        return true;
    }
    // Check that the translated string does not match the key.
    if (key === input[key] || ((0, common_1.isPluralisedTranslation)(value) && (key === value.other || key === value.one))) {
        console.log(`"${fullKey}": key matches value`);
        return true;
    }
    if (hardcodedWords.length > 0) {
        const words = expandTranslations(value).join(" ").toLowerCase().split(" ");
        if (!allowedHardcodedKeys.includes(fullKey) && hardcodedWords.some(word => words.includes(word))) {
            console.log(`"${fullKey}": contains forbidden hardcoded word`);
            return true;
        }
    }
    return false;
}
function traverseTranslations(translations, keys = []) {
    const filtered = [];
    Object.keys(translations).forEach(key => {
        const value = translations[key];
        if (typeof value === "object" && !(0, common_1.isPluralisedTranslation)(value)) {
            filtered.push(...traverseTranslations(value, [...keys, key]));
            return;
        }
        if (lintTranslation([...keys, key], value)) {
            filtered.push(key);
        }
    });
    return filtered;
}
const filtered = traverseTranslations(input);
if (filtered.length > 0) {
    console.log(`${filtered.length} invalid translation keys`);
    process.exit(1);
}
