/*
Copyright 2025 New Vector Ltd.

SPDX-License-Identifier: AGPL-3.0-only OR LicenseRef-Element-Commercial
Please see LICENSE files in the repository root for full details.
*/
/**
 * A generic API client.
 */
export class Api {
    baseUrl;
    _request;
    constructor(baseUrl) {
        this.baseUrl = baseUrl;
    }
    /**
     * Set the request context to use for making requests.
     * @param request - The request context to use.
     */
    setRequest(request) {
        this._request = request;
    }
    async request(verb, path, token, data) {
        if (!this._request) {
            throw new Error("No request context set");
        }
        const url = `${this.baseUrl}${path}`;
        const res = await this._request.fetch(url, {
            data,
            method: verb,
            headers: token
                ? {
                    Authorization: `Bearer ${token}`,
                }
                : undefined,
        });
        if (!res.ok()) {
            throw new Error(`Request to ${url} failed with status ${res.status()}: ${JSON.stringify(await res.json())}`);
        }
        return res.json();
    }
}
/**
 * A client-server API for interacting with a Matrix homeserver.
 */
export class ClientServerApi extends Api {
    constructor(baseUrl) {
        super(`${baseUrl}/_matrix/client`);
    }
    /**
     * Register a user on the homeserver.
     * @param userId - The user ID to register.
     * @param password - The password to use for the user.
     */
    async loginUser(userId, password) {
        const json = await this.request("POST", "/v3/login", undefined, {
            type: "m.login.password",
            identifier: {
                type: "m.id.user",
                user: userId,
            },
            password: password,
        });
        return {
            password,
            accessToken: json.access_token,
            userId: json.user_id,
            deviceId: json.device_id,
            homeServer: json.home_server || json.user_id.split(":").slice(1).join(":"),
            username: userId.slice(1).split(":")[0],
        };
    }
}
