/*
Copyright 2024-2025 New Vector Ltd.

SPDX-License-Identifier: AGPL-3.0-only OR LicenseRef-Element-Commercial
Please see LICENSE files in the repository root for full details.
*/
import { AbstractStartedContainer, GenericContainer, Wait } from "testcontainers";
import { MailpitClient } from "mailpit-api";
/**
 * A testcontainer for Mailpit.
 *
 * Exposes port 8025.
 * Waits for listening ports.
 * Disables SMTP authentication.
 */
export class MailpitContainer extends GenericContainer {
    constructor() {
        super("axllent/mailpit:latest");
        this.withExposedPorts(8025).withWaitStrategy(Wait.forListeningPorts()).withEnvironment({
            MP_SMTP_AUTH_ALLOW_INSECURE: "true",
            MP_SMTP_AUTH_ACCEPT_ANY: "true",
        });
    }
    /**
     * Start the Mailpit container.
     */
    async start() {
        return new StartedMailpitContainer(await super.start());
    }
}
/**
 * A started Mailpit container.
 */
export class StartedMailpitContainer extends AbstractStartedContainer {
    client;
    constructor(container) {
        super(container);
        this.client = new MailpitClient(`http://${container.getHost()}:${container.getMappedPort(8025)}`);
    }
    /**
     * Get the hostname to use to connect to the Mailpit container from inside the docker network.
     */
    get internalHost() {
        return "mailpit";
    }
    /**
     * Get the port to use to connect to the Mailpit container from inside the docker network.
     */
    get internalSmtpPort() {
        return 1025;
    }
}
