/*
Copyright 2024 - 2025 New Vector Ltd.
Copyright 2024 The Matrix.org Foundation C.I.C.

SPDX-License-Identifier: AGPL-3.0-only OR GPL-3.0-only OR LicenseRef-Element-Commercial
Please see LICENSE files in the repository root for full details.
*/
/**
 * Test reporter which compares the reported screenshots vs those on disk to find stale screenshots
 * Only intended to run from within GitHub Actions
 */
import { glob } from "glob";
import path from "node:path";
/**
 * The annotation type used to mark screenshots in tests.
 * `_` prefix hides it from the HTML reporter
 */
export const ANNOTATION = "_screenshot";
class StaleScreenshotReporter {
    snapshotRoots = new Set();
    screenshots = new Set();
    failing = false;
    success = true;
    onBegin(config) {
        for (const project of config.projects) {
            this.snapshotRoots.add(project.snapshotDir);
        }
    }
    onTestEnd(test) {
        if (!test.ok()) {
            this.failing = true;
        }
        for (const annotation of test.annotations) {
            if (annotation.type === ANNOTATION && annotation.description) {
                this.screenshots.add(annotation.description);
            }
        }
    }
    error(msg, file) {
        if (process.env.GITHUB_ACTIONS) {
            console.log(`::error file=${file}::${msg}`);
        }
        console.error(msg, file);
        this.success = false;
    }
    async onExit() {
        if (this.failing)
            return;
        if (!this.snapshotRoots.size) {
            this.error("No snapshot directories found, did you set the snapshotDir in your Playwright config?", "");
            return;
        }
        const screenshotFiles = new Set();
        for (const snapshotRoot of this.snapshotRoots) {
            const files = await glob(`**/*.png`, { cwd: snapshotRoot });
            for (const file of files) {
                screenshotFiles.add(path.join(snapshotRoot, file));
            }
        }
        for (const screenshot of screenshotFiles) {
            if (screenshot.split("-").at(-1) !== "linux.png") {
                this.error("Found screenshot belonging to different platform, this should not be checked in", screenshot);
            }
        }
        for (const screenshot of this.screenshots) {
            screenshotFiles.delete(screenshot);
        }
        if (screenshotFiles.size > 0) {
            for (const screenshot of screenshotFiles) {
                this.error("Stale screenshot file", screenshot);
            }
        }
        if (!this.success) {
            process.exit(1);
        }
    }
}
export default StaleScreenshotReporter;
