/*
Copyright 2024-2025 New Vector Ltd.
Copyright 2023 The Matrix.org Foundation C.I.C.

SPDX-License-Identifier: AGPL-3.0-only OR LicenseRef-Element-Commercial
Please see LICENSE files in the repository root for full details.
*/
import { sample, uniqueId } from "lodash-es";
import { test as base } from "./services.js";
/** Adds an initScript to the given page which will populate localStorage appropriately so that Element will use the given credentials. */
export async function populateLocalStorageWithCredentials(page, credentials) {
    await page.addInitScript(({ credentials }) => {
        window.localStorage.setItem("mx_hs_url", credentials.homeserverBaseUrl);
        window.localStorage.setItem("mx_user_id", credentials.userId);
        window.localStorage.setItem("mx_access_token", credentials.accessToken);
        window.localStorage.setItem("mx_device_id", credentials.deviceId);
        window.localStorage.setItem("mx_is_guest", "false");
        window.localStorage.setItem("mx_has_pickle_key", "false");
        window.localStorage.setItem("mx_has_access_token", "true");
        window.localStorage.setItem("mx_local_settings", JSON.stringify({
            // Retain any other settings which may have already been set
            ...JSON.parse(window.localStorage.getItem("mx_local_settings") ?? "{}"),
            // Ensure the language is set to a consistent value
            language: "en",
        }));
    }, { credentials });
}
export const test = base.extend({
    displayName: undefined,
    credentials: async ({ homeserver, displayName: testDisplayName }, use, testInfo) => {
        const names = ["Alice", "Bob", "Charlie", "Daniel", "Eve", "Frank", "Grace", "Hannah", "Isaac", "Judy"];
        const password = uniqueId("password_");
        const displayName = testDisplayName ?? sample(names);
        const credentials = await homeserver.registerUser(`user_${testInfo.testId}`, password, displayName);
        console.log(`Registered test user ${credentials.userId} with displayname ${displayName}`);
        await use({
            ...credentials,
            displayName,
        });
    },
    pageWithCredentials: async ({ page, credentials }, use) => {
        await populateLocalStorageWithCredentials(page, credentials);
        await use(page);
    },
    user: async ({ pageWithCredentials: page, credentials }, use) => {
        await page.goto("/");
        await page.waitForSelector(".mx_MatrixChat", { timeout: 30000 });
        await use(credentials);
    },
});
