/*
Copyright 2024-2025 New Vector Ltd.

SPDX-License-Identifier: AGPL-3.0-only OR LicenseRef-Element-Commercial
Please see LICENSE files in the repository root for full details.
*/
import { test as base } from "@playwright/test";
import { Network } from "testcontainers";
import { PostgreSqlContainer } from "@testcontainers/postgresql";
import { SynapseContainer, MailpitContainer, } from "../testcontainers/index.js";
import { Logger } from "../utils/logger.js";
export const test = base.extend({
    logger: [
        // eslint-disable-next-line no-empty-pattern
        async ({}, use) => {
            const logger = new Logger();
            await use(logger);
        },
        { scope: "worker" },
    ],
    network: [
        // eslint-disable-next-line no-empty-pattern
        async ({}, use) => {
            const network = await new Network().start();
            await use(network);
            await network.stop();
        },
        { scope: "worker" },
    ],
    postgres: [
        async ({ logger, network }, use) => {
            const container = await new PostgreSqlContainer("postgres:13.3-alpine")
                .withNetwork(network)
                .withNetworkAliases("postgres")
                .withLogConsumer(logger.getConsumer("postgres"))
                .withTmpFs({
                "/dev/shm/pgdata/data": "",
            })
                .withEnvironment({
                PG_DATA: "/dev/shm/pgdata/data",
            })
                .withCommand([
                "-c",
                "shared_buffers=128MB",
                "-c",
                `fsync=off`,
                "-c",
                `synchronous_commit=off`,
                "-c",
                "full_page_writes=off",
            ])
                .start();
            await use(container);
            await container.stop();
        },
        { scope: "worker" },
    ],
    mailpit: [
        async ({ logger, network }, use) => {
            const container = await new MailpitContainer()
                .withNetwork(network)
                .withNetworkAliases("mailpit")
                .withLogConsumer(logger.getConsumer("mailpit"))
                .start();
            await use(container);
            await container.stop();
        },
        { scope: "worker" },
    ],
    mailpitClient: async ({ mailpit: container }, use) => {
        await container.client.deleteMessages();
        await use(container.client);
    },
    synapseConfig: [{}, { scope: "worker" }],
    _homeserver: [
        async ({ logger }, use) => {
            const container = new SynapseContainer().withLogConsumer(logger.getConsumer("synapse"));
            await use(container);
        },
        { scope: "worker" },
    ],
    homeserver: [
        async ({ logger, network, _homeserver: homeserver, synapseConfig, mas }, use) => {
            if (homeserver instanceof SynapseContainer) {
                homeserver.withConfig(synapseConfig);
            }
            const container = await homeserver
                .withNetwork(network)
                .withNetworkAliases("homeserver")
                .withLogConsumer(logger.getConsumer("homeserver"))
                .withMatrixAuthenticationService(mas)
                .start();
            await use(container);
            await container.stop();
        },
        { scope: "worker" },
    ],
    mas: [
        // eslint-disable-next-line no-empty-pattern
        async ({}, use) => {
            // we stub the mas fixture to allow `homeserver` to depend on it to ensure
            // when it is specified by `masHomeserver` it is started before the homeserver
            await use(undefined);
        },
        { scope: "worker" },
    ],
    context: async ({ logger, context, request, homeserver }, use, testInfo) => {
        homeserver.setRequest(request);
        await logger.onTestStarted(context);
        await use(context);
        await logger.onTestFinished(testInfo);
        await homeserver.onTestFinished(testInfo);
    },
});
