const electron = require('electron');

/** Ways to communicate between the renderer and main process  */
var IPCMode;
(function (IPCMode) {
    /** Configures Electron IPC to receive messages from renderers */
    IPCMode[IPCMode["Classic"] = 1] = "Classic";
    /** Configures Electron protocol module to receive messages from renderers */
    IPCMode[IPCMode["Protocol"] = 2] = "Protocol";
    /**
     * Configures both methods for best compatibility.
     *
     * Renderers favour IPC but fall back to protocol if IPC has not
     * been configured in a preload script
     */
    IPCMode[IPCMode["Both"] = 3] = "Both";
})(IPCMode || (IPCMode = {}));
/**
 * Utility for creating namespaced IPC channels and protocol routes
 */
function ipcChannelUtils(namespace) {
    return {
        createUrl: (channel) => {
            // sentry_key in the url stops these messages from being picked up by our HTTP instrumentations
            return `${namespace}://${channel}/sentry_key`;
        },
        urlMatches: function (url, channel) {
            return url.startsWith(this.createUrl(channel));
        },
        createKey: (channel) => {
            return `${namespace}.${channel}`;
        },
        namespace,
    };
}

/**
 * This preload script may be used with sandbox mode enabled which means regular require is not available.
 */
/**
 * Hook up IPC to the window object and uses contextBridge if available.
 *
 * @param namespace An optional namespace to use for the IPC channels
 */
function hookupIpc(namespace = 'sentry-ipc') {
    const ipcUtil = ipcChannelUtils(namespace);
    // eslint-disable-next-line no-restricted-globals
    window.__SENTRY_IPC__ = window.__SENTRY_IPC__ || {};
    // eslint-disable-next-line no-restricted-globals
    if (window.__SENTRY_IPC__[ipcUtil.namespace]) {
        // eslint-disable-next-line no-console
        console.log('Sentry Electron preload has already been run');
    }
    else {
        const ipcObject = {
            sendRendererStart: () => electron.ipcRenderer.send(ipcUtil.createKey('start')),
            sendScope: (scopeJson) => electron.ipcRenderer.send(ipcUtil.createKey('scope'), scopeJson),
            sendEnvelope: (envelope) => electron.ipcRenderer.send(ipcUtil.createKey('envelope'), envelope),
            sendStatus: (status) => electron.ipcRenderer.send(ipcUtil.createKey('status'), status),
            sendStructuredLog: (log) => electron.ipcRenderer.send(ipcUtil.createKey('structured-log'), log),
        };
        // eslint-disable-next-line no-restricted-globals
        window.__SENTRY_IPC__[ipcUtil.namespace] = ipcObject;
        // We attempt to use contextBridge if it's available
        if (electron.contextBridge) {
            // This will fail if contextIsolation is not enabled
            try {
                // eslint-disable-next-line no-restricted-globals
                electron.contextBridge.exposeInMainWorld('__SENTRY_IPC__', window.__SENTRY_IPC__);
            }
            catch (e) {
                //
            }
        }
    }
}

hookupIpc();
//# sourceMappingURL=default.js.map
